# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pysudoers']

package_data = \
{'': ['*']}

install_requires = \
['toml>=0.9,<0.10']

setup_kwargs = {
    'name': 'pysudoers',
    'version': '2.0.1',
    'description': 'Python interface to the Linux sudoers file',
    'long_description': '# pysudoers\n\nThis library provides a [Python][1] interface to the Linux sudoers file.  python-sudoers is open sourced under the [BSD 3-Clause license](LICENSE.txt).\n\n![checks](https://github.com/broadinstitute/python-sudoers/workflows/checks/badge.svg?branch=main)\n[![codecov](https://codecov.io/gh/broadinstitute/python-sudoers/branch/main/graph/badge.svg)](https://codecov.io/gh/broadinstitute/python-sudoers)\n\n## Basics\n\n`pysudoers` runs on [Python][1] >= 3.6\n\n## Features\n\nThis library parses a sudoers file into its component parts.  It\'s not 100% compliant with the EBNF format of the file (yet), but it\'s getting there.  Currently, the script parses out 6 distinct line types from the file:\n\n* Defaults (This is only a string currently.  Pieces of a Defaults setting are not parsed/separated.)\n* Cmnd_Alias\n* Host_Alias\n* Runas_Alias\n* User_Alias\n* User specifications (which we call **rules**)\n\nAs user specifications are the most complicated, they are most likely the area that needs the most improvement.  Currently, the following pieces of a user specification are separated out as part of the parsing:\n\n* User list\n* Host list\n* Command list (containing):\n  * Tags\n  * Run As notations\n  * Commands\n\n## Installing\n\nYou can use pip to install pysudoers:\n\n```sh\npip install pysudoers\n```\n\n## Examples\n\nParsing of the `sudoers` file is done as part of initializing the `Sudoers` object.  So, you can start using the properties under `Sudoers` immediately.  The following example will print out all the different "types" from the file:\n\n```python\nfrom pysudoers import Sudoers\n\nsobj = Sudoers(path="tmp/sudoers")\n\nfor default in sobj.defaults:\n    print(default)\n\nfor key in sobj.host_aliases:\n    print(key)\n    print(sobj.host_aliases[key])\n\nfor key in sobj.cmnd_aliases:\n    print(key)\n    print(sobj.cmnd_aliases[key])\n\nfor key in sobj.runas_aliases:\n    print(key)\n    print(sobj.runas_aliases[key])\n\nfor key in sobj.user_aliases:\n    print(key)\n    print(sobj.user_aliases[key])\n\nfor rule in sobj.rules:\n    print(rule)\n```\n\nNow, suppose you want to print out all the user specifications (rules), but you only want to see the users and hosts for each rule.\n\n```python\nfrom pysudoers import Sudoers\n\nsobj = Sudoers(path="tmp/sudoers")\n\nfor rule in sobj.rules:\n    print("%s | %s" % (",".join(rule["users"]), ",".join(rule["hosts"])))\n```\n\n## Contributing\n\nPull requests to add functionality and fix bugs are always welcome.  Please check the CONTRIBUTING.md for specifics on contributions.\n\n### Testing\n\nWe try to have a high level of test coverage on the code.  Therefore, when adding anything to the repo, tests should be written to test a new feature or to test a bug fix so that there won\'t be a regression.  This library is setup to be pretty simple to build a working development environment using [Docker][3].  Therefore, it is suggested that you have [Docker][3] installed where you clone this repository to make development easier.\n\nTo start a development environment, you should be able to just run the `dev.sh` script.  This script will use the `Dockerfile` in this repository to build a [Docker][3] container with all the dependencies for development installed using [Poetry][2].\n\n```sh\n./dev.sh\n```\n\nThe first time you run the script, it should build the [Docker][3] image and then drop you into the container\'s shell.  The directory where you cloned this repository should be volume mounted in to `/usr/src`, which should also be the current working directory.  From there, you can make changes as you see fit.  Tests can be run from the `/usr/src` directory by simply typing `green` as [green][4] has been setup to with the correct parameters.\n\n## Changelog\n\nTo generate the `CHANGELOG.md`, you will need [Docker][3] and a GitHub personal access token.  We currently use [github-changelog-generator](https://github.com/github-changelog-generator/github-changelog-generator) for this purpose.  The following should generate the file using information from GitHub:\n\n```sh\ndocker run -it --rm \\\n    -e CHANGELOG_GITHUB_TOKEN=\'yourtokenhere\' \\\n    -v "$(pwd)":/working \\\n    -w /working \\\n    ferrarimarco/github-changelog-generator --verbose\n```\n\nTo generate the log for an upcoming release that has not yet been tagged, you can run a command to include the upcoming release version.  For example, `2.0.0`:\n\n```sh\ndocker run -it --rm \\\n    -e CHANGELOG_GITHUB_TOKEN=\'yourtokenhere\' \\\n    -v "$(pwd)":/working \\\n    -w /working \\\n    ferrarimarco/github-changelog-generator --verbose --future-release 2.0.0 --unreleased\n```\n\n## Releases\n\nReleases to the codebase are typically done using the [bump2version][5] tool.  This tool takes care of updating the version in all necessary files, updating its own configuration, and making a GitHub commit and tag.  We typically do version bumps as part of a PR, so you don\'t want to have [bump2version][5] tag the version at the same time it does the commit as commit hashes may change.  Therefore, to bump the version a patch level, one would run the command:\n\n```sh\nbump2version --verbose --no-tag patch\n```\n\nOnce the PR is merged, you can then checkout the new `main` branch and tag it using the new version number that is now in `.bumpversion.cfg`:\n\n```sh\ngit checkout main\ngit pull --rebase\ngit tag 1.0.0 -m \'Bump version: 0.1.0 → 1.0.0\'\ngit push --tags\n```\n\n[1]: https://www.python.org/ "Python"\n[2]: https://python-poetry.org/ "Poetry"\n[3]: https://www.docker.com/ "Docker"\n[4]: https://github.com/CleanCut/green "green"\n[5]: https://pypi.org/project/bump2version/ "bump2version"\n',
    'author': 'Andrew Teixeira',
    'author_email': 'teixeira@broadinstitute.org',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/broadinstitute/python-sudoers.git',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
