from ....Internal.Core import Core
from ....Internal.CommandsGroup import CommandsGroup
from ....Internal import Conversions
from ....Internal.Utilities import trim_str_response
from .... import enums


# noinspection PyPep8Naming,PyAttributeOutsideInit,SpellCheckingInspection
class Generator:
	"""Generator commands group definition. 5 total commands, 0 Sub-groups, 5 group commands"""

	def __init__(self, core: Core, parent):
		self._core = core
		self._base = CommandsGroup("generator", core, parent)

	def get_arbitrary(self) -> str:
		"""TEST:BB:GENerator:ARBitrary \n
		Selects the ARB waveform to be tested. \n
			:return: filename: string
		"""
		response = self._core.io.query_str('TEST:BB:GENerator:ARBitrary?')
		return trim_str_response(response)

	def set_arbitrary(self, filename: str) -> None:
		"""TEST:BB:GENerator:ARBitrary \n
		Selects the ARB waveform to be tested. \n
			:param filename: string
		"""
		param = Conversions.value_to_quoted_str(filename)
		self._core.io.write(f'TEST:BB:GENerator:ARBitrary {param}')

	def get_frequency(self) -> float:
		"""TEST:BB:GENerator:FREQuency \n
		Sets the frequency of the test sine or constant I/Q test signal. \n
			:return: frequency: float Range: -250E6 to 250E6
		"""
		response = self._core.io.query_str('TEST:BB:GENerator:FREQuency?')
		return Conversions.str_to_float(response)

	def set_frequency(self, frequency: float) -> None:
		"""TEST:BB:GENerator:FREQuency \n
		Sets the frequency of the test sine or constant I/Q test signal. \n
			:param frequency: float Range: -250E6 to 250E6
		"""
		param = Conversions.decimal_value_to_str(frequency)
		self._core.io.write(f'TEST:BB:GENerator:FREQuency {param}')

	def get_gain(self) -> float:
		"""TEST:BB:GENerator:GAIN \n
		Sets the gain for a sine or constant I/Q test signal. \n
			:return: gain: float Range: -1 to 1
		"""
		response = self._core.io.query_str('TEST:BB:GENerator:GAIN?')
		return Conversions.str_to_float(response)

	def set_gain(self, gain: float) -> None:
		"""TEST:BB:GENerator:GAIN \n
		Sets the gain for a sine or constant I/Q test signal. \n
			:param gain: float Range: -1 to 1
		"""
		param = Conversions.decimal_value_to_str(gain)
		self._core.io.write(f'TEST:BB:GENerator:GAIN {param}')

	# noinspection PyTypeChecker
	def get_source(self) -> enums.TestBbGenIqSour:
		"""TEST:BB:GENerator:SOURce \n
		Selects the test signal source. \n
			:return: iq_source: SINE| CONStant| ARB SINE Generates a sine waveform. To determine the frequency, use command method RsSmbv.Test.Bb.Generator.frequency. CONStant Uses a constant I/Q test signal. ARB Selects a signal generated generated by the ARB.
		"""
		response = self._core.io.query_str('TEST:BB:GENerator:SOURce?')
		return Conversions.str_to_scalar_enum(response, enums.TestBbGenIqSour)

	def set_source(self, iq_source: enums.TestBbGenIqSour) -> None:
		"""TEST:BB:GENerator:SOURce \n
		Selects the test signal source. \n
			:param iq_source: SINE| CONStant| ARB SINE Generates a sine waveform. To determine the frequency, use command method RsSmbv.Test.Bb.Generator.frequency. CONStant Uses a constant I/Q test signal. ARB Selects a signal generated generated by the ARB.
		"""
		param = Conversions.enum_scalar_to_str(iq_source, enums.TestBbGenIqSour)
		self._core.io.write(f'TEST:BB:GENerator:SOURce {param}')

	def get_state(self) -> bool:
		"""TEST:BB:GENerator:STATe \n
		Starts the test generator. \n
			:return: state: 0| 1| OFF| ON
		"""
		response = self._core.io.query_str('TEST:BB:GENerator:STATe?')
		return Conversions.str_to_bool(response)

	def set_state(self, state: bool) -> None:
		"""TEST:BB:GENerator:STATe \n
		Starts the test generator. \n
			:param state: 0| 1| OFF| ON
		"""
		param = Conversions.bool_to_str(state)
		self._core.io.write(f'TEST:BB:GENerator:STATe {param}')
