from ..........Internal.Core import Core
from ..........Internal.CommandsGroup import CommandsGroup
from ..........Internal import Conversions
from ..........Internal.RepeatedCapability import RepeatedCapability
from .......... import repcap


# noinspection PyPep8Naming,PyAttributeOutsideInit,SpellCheckingInspection
class Cqi:
	"""Cqi commands group definition. 1 total commands, 0 Sub-groups, 1 group commands
	Repeated Capability: TwoStreams, default value after init: TwoStreams.Nr0"""

	def __init__(self, core: Core, parent):
		self._core = core
		self._base = CommandsGroup("cqi", core, parent)
		self._base.rep_cap = RepeatedCapability(self._base.group_name, 'repcap_twoStreams_get', 'repcap_twoStreams_set', repcap.TwoStreams.Nr0)

	def clone(self) -> 'Cqi':
		"""Clones the group by creating new object from it and its whole existing sub-groups
		Also copies all the existing default Repeated Capabilities setting,
		which you can change independently without affecting the original group"""
		new_group = Cqi(self._core, self._base.parent)
		self._base.synchronize_repcaps(new_group)
		return new_group

	def repcap_twoStreams_set(self, enum_value: repcap.TwoStreams) -> None:
		"""Repeated Capability default value numeric suffix.
		This value is used, if you do not explicitely set it in the child set/get methods, or if you leave it to TwoStreams.Default
		Default value after init: TwoStreams.Nr0"""
		self._base.set_repcap_enum_value(enum_value)

	def repcap_twoStreams_get(self) -> repcap.TwoStreams:
		"""Returns the current default repeated capability for the child set/get methods"""
		# noinspection PyTypeChecker
		return self._base.get_repcap_enum_value()

	def set(self, cqi: int, stream=repcap.Stream.Default, channel=repcap.Channel.Default, twoStreams=repcap.TwoStreams.Default) -> None:
		"""[SOURce<HW>]:BB:W3GPp:MSTation<ST>:DPCCh:HS:MIMO:TTI<CH>:CQI<DI> \n
		Selects the CQI report transmitted during the PCI/CQI slots of the corresponding TTI.
			INTRO_CMD_HELP: For single stream transmission (method RsSmbv.Source.Bb.W3Gpp.Mstation.Dpcch.Hs.Mimo.Tti.Cqi.set) , this command set the CQI values of the following cases: \n
			- The CQI (the value for CQI Type B report)
			- The CQIS (the CQI value in case a CQI Type A report when one transport block is preferred)
			INTRO_CMD_HELP: For dual stream transmission (BB:W3GP:MST:DPCC:HS:MIMO:TTI:CQI2) , this command sets: \n
			- The CQI1, the first of the two CQI values of CQI Type A report when two transport blocks are preferred
			- The CQI2, the second of the two CQI values of CQI Type A report when two transport blocks are preferred. The CQI then is calculated as follows: CQI = 15*CQI1+CQI2+31 \n
			:param cqi: integer Range: 0 to 30
			:param stream: optional repeated capability selector. Default value: Nr1 (settable in the interface 'Mstation')
			:param channel: optional repeated capability selector. Default value: Nr1 (settable in the interface 'Tti')
			:param twoStreams: optional repeated capability selector. Default value: Nr0 (settable in the interface 'Cqi')"""
		param = Conversions.decimal_value_to_str(cqi)
		stream_cmd_val = self._base.get_repcap_cmd_value(stream, repcap.Stream)
		channel_cmd_val = self._base.get_repcap_cmd_value(channel, repcap.Channel)
		twoStreams_cmd_val = self._base.get_repcap_cmd_value(twoStreams, repcap.TwoStreams)
		self._core.io.write(f'SOURce<HwInstance>:BB:W3GPp:MSTation{stream_cmd_val}:DPCCh:HS:MIMO:TTI{channel_cmd_val}:CQI{twoStreams_cmd_val} {param}')

	def get(self, stream=repcap.Stream.Default, channel=repcap.Channel.Default, twoStreams=repcap.TwoStreams.Default) -> int:
		"""[SOURce<HW>]:BB:W3GPp:MSTation<ST>:DPCCh:HS:MIMO:TTI<CH>:CQI<DI> \n
		Selects the CQI report transmitted during the PCI/CQI slots of the corresponding TTI.
			INTRO_CMD_HELP: For single stream transmission (method RsSmbv.Source.Bb.W3Gpp.Mstation.Dpcch.Hs.Mimo.Tti.Cqi.set) , this command set the CQI values of the following cases: \n
			- The CQI (the value for CQI Type B report)
			- The CQIS (the CQI value in case a CQI Type A report when one transport block is preferred)
			INTRO_CMD_HELP: For dual stream transmission (BB:W3GP:MST:DPCC:HS:MIMO:TTI:CQI2) , this command sets: \n
			- The CQI1, the first of the two CQI values of CQI Type A report when two transport blocks are preferred
			- The CQI2, the second of the two CQI values of CQI Type A report when two transport blocks are preferred. The CQI then is calculated as follows: CQI = 15*CQI1+CQI2+31 \n
			:param stream: optional repeated capability selector. Default value: Nr1 (settable in the interface 'Mstation')
			:param channel: optional repeated capability selector. Default value: Nr1 (settable in the interface 'Tti')
			:param twoStreams: optional repeated capability selector. Default value: Nr0 (settable in the interface 'Cqi')
			:return: cqi: integer Range: 0 to 30"""
		stream_cmd_val = self._base.get_repcap_cmd_value(stream, repcap.Stream)
		channel_cmd_val = self._base.get_repcap_cmd_value(channel, repcap.Channel)
		twoStreams_cmd_val = self._base.get_repcap_cmd_value(twoStreams, repcap.TwoStreams)
		response = self._core.io.query_str(f'SOURce<HwInstance>:BB:W3GPp:MSTation{stream_cmd_val}:DPCCh:HS:MIMO:TTI{channel_cmd_val}:CQI{twoStreams_cmd_val}?')
		return Conversions.str_to_int(response)
