from .........Internal.Core import Core
from .........Internal.CommandsGroup import CommandsGroup
from .........Internal import Conversions
from .........Internal.RepeatedCapability import RepeatedCapability
from ......... import repcap


# noinspection PyPep8Naming,PyAttributeOutsideInit,SpellCheckingInspection
class Row:
	"""Row commands group definition. 1 total commands, 0 Sub-groups, 1 group commands
	Repeated Capability: RowIx, default value after init: RowIx.Nr0"""

	def __init__(self, core: Core, parent):
		self._core = core
		self._base = CommandsGroup("row", core, parent)
		self._base.rep_cap = RepeatedCapability(self._base.group_name, 'repcap_rowIx_get', 'repcap_rowIx_set', repcap.RowIx.Nr0)

	def clone(self) -> 'Row':
		"""Clones the group by creating new object from it and its whole existing sub-groups
		Also copies all the existing default Repeated Capabilities setting,
		which you can change independently without affecting the original group"""
		new_group = Row(self._core, self._base.parent)
		self._base.synchronize_repcaps(new_group)
		return new_group

	def repcap_rowIx_set(self, enum_value: repcap.RowIx) -> None:
		"""Repeated Capability default value numeric suffix.
		This value is used, if you do not explicitely set it in the child set/get methods, or if you leave it to RowIx.Default
		Default value after init: RowIx.Nr0"""
		self._base.set_repcap_enum_value(enum_value)

	def repcap_rowIx_get(self) -> repcap.RowIx:
		"""Returns the current default repeated capability for the child set/get methods"""
		# noinspection PyTypeChecker
		return self._base.get_repcap_enum_value()

	def set(self, ant_port_mapping: bool, stream=repcap.Stream.Default, channel=repcap.Channel.Default, rowIx=repcap.RowIx.Default) -> None:
		"""[SOURce<HW>]:BB:EUTRa:UL:UE<ST>:APMap:AP20Map<CH>:ROW<BBID> \n
		Sets which antenna port will be generated by which baseband. \n
			:param ant_port_mapping: 0| 1| OFF| ON
			:param stream: optional repeated capability selector. Default value: Nr1 (settable in the interface 'Ue')
			:param channel: optional repeated capability selector. Default value: Nr1 (settable in the interface 'Ap20Map')
			:param rowIx: optional repeated capability selector. Default value: Nr0 (settable in the interface 'Row')"""
		param = Conversions.bool_to_str(ant_port_mapping)
		stream_cmd_val = self._base.get_repcap_cmd_value(stream, repcap.Stream)
		channel_cmd_val = self._base.get_repcap_cmd_value(channel, repcap.Channel)
		rowIx_cmd_val = self._base.get_repcap_cmd_value(rowIx, repcap.RowIx)
		self._core.io.write(f'SOURce<HwInstance>:BB:EUTRa:UL:UE{stream_cmd_val}:APMap:AP20Map{channel_cmd_val}:ROW{rowIx_cmd_val} {param}')

	def get(self, stream=repcap.Stream.Default, channel=repcap.Channel.Default, rowIx=repcap.RowIx.Default) -> bool:
		"""[SOURce<HW>]:BB:EUTRa:UL:UE<ST>:APMap:AP20Map<CH>:ROW<BBID> \n
		Sets which antenna port will be generated by which baseband. \n
			:param stream: optional repeated capability selector. Default value: Nr1 (settable in the interface 'Ue')
			:param channel: optional repeated capability selector. Default value: Nr1 (settable in the interface 'Ap20Map')
			:param rowIx: optional repeated capability selector. Default value: Nr0 (settable in the interface 'Row')
			:return: ant_port_mapping: 0| 1| OFF| ON"""
		stream_cmd_val = self._base.get_repcap_cmd_value(stream, repcap.Stream)
		channel_cmd_val = self._base.get_repcap_cmd_value(channel, repcap.Channel)
		rowIx_cmd_val = self._base.get_repcap_cmd_value(rowIx, repcap.RowIx)
		response = self._core.io.query_str(f'SOURce<HwInstance>:BB:EUTRa:UL:UE{stream_cmd_val}:APMap:AP20Map{channel_cmd_val}:ROW{rowIx_cmd_val}?')
		return Conversions.str_to_bool(response)
