"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VpcProvider = exports.AmiProvider = exports.Cluster = void 0;
const path = require("path");
const autoscaling = require("@aws-cdk/aws-autoscaling");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const logs = require("@aws-cdk/aws-logs");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
const DEFAULT_INSTANCE_TYPE = ec2.InstanceType.of(ec2.InstanceClass.M6G, ec2.InstanceSize.MEDIUM);
/**
 * Represents the k3sCluster construct.
 *
 * @stability stable
 */
class Cluster extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e;
        super(scope, id);
        // VPC configuration
        const vpc = (_a = props.vpc) !== null && _a !== void 0 ? _a : new ec2.Vpc(this, 'Vpc', { maxAzs: 3, natGateways: 1 });
        // S3 bucket to host K3s token + kubeconfig file
        const k3sBucket = new s3.Bucket(this, 'k3sBucket', {
            removalPolicy: (_b = props.bucketRemovalPolicy) !== null && _b !== void 0 ? _b : cdk.RemovalPolicy.RETAIN,
        });
        // Delete S3 Object CustomResource
        if (props.bucketRemovalPolicy === cdk.RemovalPolicy.DESTROY) {
            const onEvent = new lambda.Function(this, 'onEventHandler', {
                runtime: lambda.Runtime.PYTHON_3_8,
                code: lambda.Code.fromAsset(path.join(__dirname, '../custom-resource-handler')),
                handler: 'index.on_event',
            });
            const deleteS3ObjectProvider = new cr.Provider(this, 'deleteS3ObjectProvider', {
                onEventHandler: onEvent,
                logRetention: logs.RetentionDays.ONE_DAY,
            });
            const CRdeleteS3ObjectProvider = new cdk.CustomResource(this, 'CRdeleteS3ObjectProvider', {
                serviceToken: deleteS3ObjectProvider.serviceToken,
                properties: {
                    Bucket: k3sBucket.bucketName,
                },
            });
            CRdeleteS3ObjectProvider.node.addDependency(k3sBucket);
            k3sBucket.grantDelete(onEvent);
            k3sBucket.grantReadWrite(onEvent);
        }
        // control plane node Security Group
        const k3scontrolplanesg = new ec2.SecurityGroup(this, 'k3s-controlplane-SG', { vpc });
        k3scontrolplanesg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(22), 'SSH');
        k3scontrolplanesg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(6443), 'K3s port');
        // worker nodes Security Group
        const k3sworkersg = new ec2.SecurityGroup(this, 'k3s-worker-SG', { vpc });
        // for this prototype the workers are being placed in a public subnet
        // ideally they should land on a private subnet
        /// also ingress traffic - ssh (bastion style) or 6443 - should come from the control plane node only
        k3sworkersg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(22), 'SSH');
        k3sworkersg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(6443), 'K3s port');
        // check if the user requires a particular instance type for workers and control plane
        // if not, the default instance type is used
        this.controlPlaneInstanceType = (_c = props.controlPlaneInstanceType) !== null && _c !== void 0 ? _c : DEFAULT_INSTANCE_TYPE;
        this.workerInstanceType = (_d = props.workerInstanceType) !== null && _d !== void 0 ? _d : DEFAULT_INSTANCE_TYPE;
        // create control plane node
        const k3scontrolplane = new ec2.Instance(this, 'k3s-controlplane', {
            instanceType: this.controlPlaneInstanceType,
            machineImage: new AmiProvider().amiId,
            vpc,
            vpcSubnets: {
                subnets: vpc.publicSubnets,
            },
            instanceName: 'k3s-controlplane',
            securityGroup: k3scontrolplanesg,
        });
        k3scontrolplane.addUserData(`
       #!/bin/bash
       curl -L -o k3s https://github.com/rancher/k3s/releases/download/v1.16.9%2Bk3s1/k3s-arm64
       chmod +x k3s
       ./k3s server &
       sleep 30
       ENDPOINT=$(curl http://169.254.169.254/latest/meta-data/public-hostname) 
       cp /etc/rancher/k3s/k3s.yaml /etc/rancher/k3s/kubeconfig.yaml
       sed -i s/127.0.0.1/$ENDPOINT/ /etc/rancher/k3s/kubeconfig.yaml
       aws s3 cp /var/lib/rancher/k3s/server/node-token s3://${k3sBucket.bucketName}/node-token
       aws s3 cp /etc/rancher/k3s/kubeconfig.yaml s3://${k3sBucket.bucketName}/kubeconfig.yaml
     `);
        this.endpointUri = k3scontrolplane.instancePublicIp;
        // create launch template for worker ASG
        // prepare the userData
        const userData = ec2.UserData.forLinux();
        userData.addCommands(`
          #!/bin/bash
          LOGFILE='/var/log/k3s.log'
          curl -L -o k3s https://github.com/rancher/k3s/releases/download/v1.16.13%2Bk3s1/k3s-arm64
          chmod +x k3s
          echo the bucket name is ${k3sBucket.bucketName} 
          aws s3 cp s3://${k3sBucket.bucketName}/node-token /node-token 
          (./k3s agent --server https://${k3scontrolplane.instancePrivateIp}:6443 \
          --token $(cat /node-token) 2>&1 | tee -a $LOGFILE || echo "failed" > $LOGFILE &)
    `);
        const lt = new ec2.CfnLaunchTemplate(this, 'WorkerLaunchTemplate', {
            launchTemplateData: {
                imageId: new AmiProvider().amiId.getImage(this).imageId,
                instanceType: this.workerInstanceType.toString(),
                instanceMarketOptions: {
                    marketType: props.spotWorkerNodes ? 'spot' : undefined,
                    spotOptions: props.spotWorkerNodes ? {
                        spotInstanceType: 'one-time',
                    } : undefined,
                },
                userData: cdk.Fn.base64(userData.render()),
            },
        });
        // create worker ASG
        const workerAsg = new autoscaling.AutoScalingGroup(this, 'WorkerAsg', {
            instanceType: this.workerInstanceType,
            machineImage: new AmiProvider().amiId,
            vpc,
            vpcSubnets: {
                subnetType: ec2.SubnetType.PUBLIC,
            },
            minCapacity: (_e = props.workerMinCapacity) !== null && _e !== void 0 ? _e : 3,
        });
        const cfnAsg = workerAsg.node.tryFindChild('ASG');
        cfnAsg.addPropertyDeletionOverride('LaunchConfigurationName');
        cfnAsg.addPropertyOverride('LaunchTemplate', {
            LaunchTemplateId: lt.ref,
            Version: lt.attrLatestVersionNumber,
        });
        workerAsg.addSecurityGroup(k3sworkersg);
        // enable the SSM session manager
        workerAsg.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        // grant the S3 write permission to the control plane node and read permissions to the worker nodes
        k3sBucket.grantWrite(k3scontrolplane.role);
        k3sBucket.grantRead(workerAsg.role);
        // endpoint info
        new cdk.CfnOutput(this, 'Endpoint', { value: `https://${k3scontrolplane.instancePublicIp}:6443` });
        // kubeconfig.yaml path
        new cdk.CfnOutput(this, 'Kubernetes configuration file', { value: `s3://${k3sBucket.bucketName}/kubeconfig.yaml` });
        workerAsg.node.addDependency(k3scontrolplane);
    }
}
exports.Cluster = Cluster;
/**
 * The AMI provider to get the latest Amazon Linux 2 AMI for ARM64.
 *
 * @stability stable
 */
class AmiProvider {
    /**
     * @stability stable
     */
    get amiId() {
        return ec2.MachineImage.latestAmazonLinux({
            cpuType: ec2.AmazonLinuxCpuType.ARM_64,
            generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
        });
    }
}
exports.AmiProvider = AmiProvider;
/**
 * The VPC provider to create or import the VPC.
 *
 * @stability stable
 */
class VpcProvider {
    /**
     * @stability stable
     */
    static getOrCreate(scope) {
        const vpc = scope.node.tryGetContext('use_default_vpc') === '1' ?
            ec2.Vpc.fromLookup(scope, 'Vpc', { isDefault: true }) :
            scope.node.tryGetContext('use_vpc_id') ?
                ec2.Vpc.fromLookup(scope, 'Vpc', { vpcId: scope.node.tryGetContext('use_vpc_id') }) :
                new ec2.Vpc(scope, 'Vpc', { maxAzs: 3, natGateways: 1 });
        return vpc;
    }
}
exports.VpcProvider = VpcProvider;
//# sourceMappingURL=data:application/json;base64,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