"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcsRunTask = exports.EcsEc2LaunchTarget = exports.EcsFargateLaunchTarget = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * Configuration for running an ECS task on Fargate.
 *
 * @see https://docs.aws.amazon.com/AmazonECS/latest/userguide/launch_types.html#launch-type-fargate
 * @stability stable
 */
class EcsFargateLaunchTarget {
    /**
     * @stability stable
     */
    constructor(options) {
        this.options = options;
    }
    /**
     * Called when the Fargate launch type configured on RunTask.
     *
     * @stability stable
     */
    bind(_task, launchTargetOptions) {
        var _d;
        if (!launchTargetOptions.taskDefinition.isFargateCompatible) {
            throw new Error('Supplied TaskDefinition is not compatible with Fargate');
        }
        return {
            parameters: {
                LaunchType: 'FARGATE',
                PlatformVersion: (_d = this.options) === null || _d === void 0 ? void 0 : _d.platformVersion,
            },
        };
    }
}
exports.EcsFargateLaunchTarget = EcsFargateLaunchTarget;
_a = JSII_RTTI_SYMBOL_1;
EcsFargateLaunchTarget[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.EcsFargateLaunchTarget", version: "1.98.0" };
/**
 * Configuration for running an ECS task on EC2.
 *
 * @see https://docs.aws.amazon.com/AmazonECS/latest/userguide/launch_types.html#launch-type-ec2
 * @stability stable
 */
class EcsEc2LaunchTarget {
    /**
     * @stability stable
     */
    constructor(options) {
        this.options = options;
    }
    /**
     * Called when the EC2 launch type is configured on RunTask.
     *
     * @stability stable
     */
    bind(_task, launchTargetOptions) {
        var _d, _e, _f, _g, _h;
        if (!launchTargetOptions.taskDefinition.isEc2Compatible) {
            throw new Error('Supplied TaskDefinition is not compatible with EC2');
        }
        if (!((_d = launchTargetOptions.cluster) === null || _d === void 0 ? void 0 : _d.hasEc2Capacity)) {
            throw new Error('Cluster for this service needs Ec2 capacity. Call addCapacity() on the cluster.');
        }
        return {
            parameters: {
                LaunchType: 'EC2',
                // takes an array of placement constraints each of which contain a single item array of constraints, flattens it
                // and renders the Json to be passed as a parameter in the state machine.
                // input: [ecs.PlacementConstraint.distinctInstances()] - distinctInstances() returns [{ type: 'distinctInstance' }]
                // output: {Type: 'distinctInstance'}
                PlacementConstraints: noEmpty(flatten(((_f = (_e = this.options) === null || _e === void 0 ? void 0 : _e.placementConstraints) !== null && _f !== void 0 ? _f : []).map((c) => c.toJson().map(uppercaseKeys)))),
                PlacementStrategy: noEmpty(flatten(((_h = (_g = this.options) === null || _g === void 0 ? void 0 : _g.placementStrategies) !== null && _h !== void 0 ? _h : []).map((c) => c.toJson().map(uppercaseKeys)))),
            },
        };
        function uppercaseKeys(obj) {
            const ret = {};
            for (const key of Object.keys(obj)) {
                ret[key.slice(0, 1).toUpperCase() + key.slice(1)] = obj[key];
            }
            return ret;
        }
        function flatten(xs) {
            return Array.prototype.concat([], ...xs);
        }
        function noEmpty(xs) {
            if (xs.length === 0) {
                return undefined;
            }
            return xs;
        }
    }
}
exports.EcsEc2LaunchTarget = EcsEc2LaunchTarget;
_b = JSII_RTTI_SYMBOL_1;
EcsEc2LaunchTarget[_b] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.EcsEc2LaunchTarget", version: "1.98.0" };
/**
 * Run a Task on ECS or Fargate.
 *
 * @stability stable
 */
class EcsRunTask extends sfn.TaskStateBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _d, _e;
        super(scope, id, props);
        this.props = props;
        /**
         * Manage allowed network traffic for this service.
         *
         * @stability stable
         */
        this.connections = new ec2.Connections();
        this.securityGroups = [];
        this.integrationPattern = (_d = props.integrationPattern) !== null && _d !== void 0 ? _d : sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, EcsRunTask.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN && !sfn.FieldUtils.containsTaskToken(props.containerOverrides)) {
            throw new Error('Task Token is required in `containerOverrides` for callback. Use Context.taskToken to set the token.');
        }
        if (!this.props.taskDefinition.defaultContainer) {
            throw new Error('A TaskDefinition must have at least one essential container');
        }
        if (this.props.taskDefinition.networkMode === ecs.NetworkMode.AWS_VPC) {
            this.configureAwsVpcNetworking();
        }
        else {
            // Either None, Bridge or Host networking. Copy SecurityGroup from ASG.
            this.validateNoNetworkingProps();
            this.connections.addSecurityGroup(...this.props.cluster.connections.securityGroups);
        }
        for (const override of (_e = this.props.containerOverrides) !== null && _e !== void 0 ? _e : []) {
            const name = override.containerDefinition.containerName;
            if (!cdk.Token.isUnresolved(name)) {
                const cont = this.props.taskDefinition.node.tryFindChild(name);
                if (!cont) {
                    throw new Error(`Overrides mention container with name '${name}', but no such container in task definition`);
                }
            }
        }
        this.taskPolicies = this.makePolicyStatements();
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('ecs', 'runTask', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                Cluster: this.props.cluster.clusterArn,
                TaskDefinition: this.props.taskDefinition.family,
                NetworkConfiguration: this.networkConfiguration,
                Overrides: renderOverrides(this.props.containerOverrides),
                ...this.props.launchTarget.bind(this, { taskDefinition: this.props.taskDefinition, cluster: this.props.cluster }).parameters,
            }),
        };
    }
    configureAwsVpcNetworking() {
        var _d, _e;
        const subnetSelection = (_d = this.props.subnets) !== null && _d !== void 0 ? _d : { subnetType: this.props.assignPublicIp ? ec2.SubnetType.PUBLIC : ec2.SubnetType.PRIVATE };
        this.networkConfiguration = {
            AwsvpcConfiguration: {
                AssignPublicIp: this.props.assignPublicIp ? (this.props.assignPublicIp ? 'ENABLED' : 'DISABLED') : undefined,
                Subnets: this.props.cluster.vpc.selectSubnets(subnetSelection).subnetIds,
                SecurityGroups: cdk.Lazy.list({ produce: () => { var _d; return (_d = this.securityGroups) === null || _d === void 0 ? void 0 : _d.map(sg => sg.securityGroupId); } }),
            },
        };
        // Make sure we have a security group if we're using AWSVPC networking
        this.securityGroups = (_e = this.props.securityGroups) !== null && _e !== void 0 ? _e : [new ec2.SecurityGroup(this, 'SecurityGroup', { vpc: this.props.cluster.vpc })];
        this.connections.addSecurityGroup(...this.securityGroups);
    }
    validateNoNetworkingProps() {
        if (this.props.subnets !== undefined || this.props.securityGroups !== undefined) {
            throw new Error(`Supplied TaskDefinition must have 'networkMode' of 'AWS_VPC' to use 'vpcSubnets' and 'securityGroup'. Received: ${this.props.taskDefinition.networkMode}`);
        }
    }
    makePolicyStatements() {
        const stack = cdk.Stack.of(this);
        // https://docs.aws.amazon.com/step-functions/latest/dg/ecs-iam.html
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['ecs:RunTask'],
                resources: [this.getTaskDefinitionFamilyArn()],
            }),
            new iam.PolicyStatement({
                actions: ['ecs:StopTask', 'ecs:DescribeTasks'],
                resources: ['*'],
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: this.taskExecutionRoles().map((r) => r.roleArn),
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForECSTaskRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
    /**
     * Returns the ARN of the task definition family by removing the
     * revision from the task definition ARN
     * Before - arn:aws:ecs:us-west-2:123456789012:task-definition/hello_world:8
     * After - arn:aws:ecs:us-west-2:123456789012:task-definition/hello_world
     */
    getTaskDefinitionFamilyArn() {
        const arnComponents = cdk.Stack.of(this).parseArn(this.props.taskDefinition.taskDefinitionArn);
        let { resourceName } = arnComponents;
        if (resourceName) {
            resourceName = resourceName.split(':')[0];
        }
        return cdk.Stack.of(this).formatArn({ ...arnComponents, resourceName });
    }
    taskExecutionRoles() {
        // Need to be able to pass both Task and Execution role, apparently
        const ret = new Array();
        ret.push(this.props.taskDefinition.taskRole);
        if (this.props.taskDefinition.executionRole) {
            ret.push(this.props.taskDefinition.executionRole);
        }
        return ret;
    }
}
exports.EcsRunTask = EcsRunTask;
_c = JSII_RTTI_SYMBOL_1;
EcsRunTask[_c] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.EcsRunTask", version: "1.98.0" };
EcsRunTask.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
function renderOverrides(containerOverrides) {
    var _d;
    if (!containerOverrides || containerOverrides.length === 0) {
        return undefined;
    }
    const ret = new Array();
    for (const override of containerOverrides) {
        ret.push({
            Name: override.containerDefinition.containerName,
            Command: override.command,
            Cpu: override.cpu,
            Memory: override.memoryLimit,
            MemoryReservation: override.memoryReservation,
            Environment: (_d = override.environment) === null || _d === void 0 ? void 0 : _d.map((e) => ({
                Name: e.name,
                Value: e.value,
            })),
        });
    }
    return { ContainerOverrides: ret };
}
//# sourceMappingURL=data:application/json;base64,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