# -*- coding: utf-8 -*-
from distutils.core import setup

modules = \
['argser']
extras_require = \
{'tabulate': ['tabulate>=0.8.5,<0.9.0']}

setup_kwargs = {
    'name': 'argser',
    'version': '0.0.4',
    'description': 'Arguments parsing without boilerplate.',
    'long_description': '# argser\n\n[![PyPI version](https://badge.fury.io/py/argser.svg)](http://badge.fury.io/py/argser)\n[![Build Status](https://github.com/vanyakosmos/argser/workflows/build/badge.svg)](https://github.com/vanyakosmos/argser/actions?workflow=build)\n[![Coverage](https://codecov.io/gh/vanyakosmos/argser/branch/master/graph/badge.svg)](https://codecov.io/gh/vanyakosmos/argser)\n[![License](https://img.shields.io/github/license/mashape/apistatus.svg)](https://pypi.python.org/pypi/argser/)\n[![Downloads](https://pepy.tech/badge/argser)](https://pepy.tech/project/argser)\n\nArguments parsing without boilerplate.\n\n------\n\n## simple example\n\n```python\nfrom argser import parse_args\n\nclass Args:\n    a = \'a\'\n    foo = 1\n    bar: bool\n\n\nargs = parse_args(Args)\n```\n\n<details>\n<summary>argparse alternative</summary>\n    \n```python\nfrom argparse import ArgumentParser\n\nparser = ArgumentParser()\nparser.add_argument(\'-a\', type=str, default=\'a\', help="str, default: \'a\'")\nparser.add_argument(\'--foo\', \'-f\', dest=\'foo\', type=int, default=1, help="int, default: 1")\nparser.add_argument(\'--bar\', \'-b\', dest=\'bar\', action=\'store_true\', help="bool, default: None")\nparser.add_argument(\'--no-bar\', \'--no-b\', dest=\'bar\', action=\'store_false\')\nparser.set_defaults(bar=None)\n\nargs = parser.parse_args()\nprint(args)\n```\n</details>\n\n```\npython playground.py -a "aaa bbb" -f 100500 --no-b\n>> Args(bar=False, a=\'aaa bbb\', foo=100500)\n```\n\n```\n❯ python playground.py -h\nusage: playground.py [-h] [--bar] [--no-bar] [-a [A]] [--foo [FOO]]\n\noptional arguments:\n  -h, --help            show this help message and exit\n  --bar, -b             bool, default: None.\n  --no-bar, --no-b\n  -a [A]                str, default: \'a\'.\n  --foo [FOO], -f [FOO]\n                        int, default: 1.\n```\n\n## sub commands\n\n```python\nfrom argser import parse_args, sub_command\n\nclass Args:\n    a: bool\n    b = []\n    c = 5\n    \n    class Sub1:\n        d = 1\n        e = \'2\'\n    \n    sub1 = sub_command(Sub1)\n    \n    class Sub2:\n        f = 1\n        g = \'2\'\n    \n    sub2 = sub_command(Sub2)\n\n\nargs = parse_args(Args, \'-a -c 10\')\nassert args.a is True\nassert args.c == 10\nassert args.sub1 is None\nassert args.sub2 is None\n\nargs = parse_args(Args, \'-a -c 10 sub1 -d 5\')\nassert args.sub1.d == 5\nassert args.sub2 is None\n\nargs = parse_args(Args, \'-a -c 10 sub2 -g "foo bar"\')\nassert args.sub1 is None\nassert args.sub2.g == "foo bar"\n```\n\n```\n❯ python playground.py -h\nusage: playground.py [-h] [-a] [--no-a] [-b [B [B ...]]] [-c [C]]\n                     {sub1,sub2} ...\n\npositional arguments:\n  {sub1,sub2}\n\noptional arguments:\n  -h, --help      show this help message and exit\n  -a              bool, default: None.\n  --no-a\n  -b [B [B ...]]  List[str], default: [].\n  -c [C]          int, default: 5.\n```\n\n```\n❯ python playground.py sub1 -h\nusage: playground.py sub1 [-h] [-d [D]] [-e [E]]\n\noptional arguments:\n  -h, --help  show this help message and exit\n  -d [D]      int, default: 1.\n  -e [E]      str, default: \'2\'.\n```\n\n\n## more examples\n\nfor more usage examples check out [tests.py](tests.py) file\n\n\n## arguments\n\n```python\nfrom typing import List\nfrom argser import Arg, PosArg\n\nclass Args:\n    # str / int / float\n    a1: str  # default is None\n    a2 = 2  # default is 2\n    a3: float = Arg(default=3.0, help="a3")  # default is 3.0, with additional help text\n\n    # bool\n    b1: bool  # default is None, to change use flags: --b1 or --no-b1\n    b2 = True  # default is True, to change to False: ./script.py --no-b2\n    b3 = False  # default is False, to change to True: ./script.py --b3\n    b4: bool = Arg(bool_flag=False)  # to change specify value: `--b4 1` or `--b4 false` or ...\n    \n    # list\n    l1 = []  # default = [], type = str, nargs = *\n    l2: List[int] = []  # default = [], type = int, nargs = *\n    l3 = [1.0]  # default = [], type = float, nargs = +\n    l4: List[int] = Arg(default=[], nargs=\'+\')  # default = [], type = int, nargs = +\n\n    # positional args\n    c1: float = PosArg()  # ./script.py 12.34\n\n    # one dash\n    d1: int = Arg(one_dash=True)  # ./script.py -d1 1\n    \n    # help\n    h1 = Arg()  # only default help message: "str, default: None."\n    h2 = Arg(help="foo bar")  # default + custom: "str, default: None. foo bar"\n    h3 = Arg(help="foo bar", keep_default_help=False)  # just custom: "foo bar"\n```\n\n## parse\n\nUse `parse_args` to parse arguments from string or command line.\n\nParams:\n - args_cls: class with defined arguments\n - args: arguments to parse. Either string or list of strings or None (to read from sys.args)\n - show:\n    - if True - print arguments in one line\n    - if \'table\' - print arguments as table\n - print_fn:\n - make_shortcuts: make short version of arguments: --abc -> -a, --abc_def -> --ad\n - bool_flag:\n    - if True then read bool from argument flag: `--arg` is True, `--no-arg` is False,\n    - otherwise check if arg value and truthy or falsy: `--arg 1` is True `--arg no` is False\n - one_dash: use one dash for long names: `-name` instead of `--name`\n - keep_default_help: prepend autogenerated help message to your help message\n - help_format: default help format\n - override: override values above on Arg\'s\n - kwargs: additional kwargs for `tabulate`\n\n\n\n## notes\n\n1. annotations are not static fields\n```python\nclass Args(ArgsParse):\n    com = Command()\n    a: str\n\nassert hasattr(Args, \'a\') is False\n```\nAnnotation "fields" don\'t appear in `cls.__dict__` at all.\nThat\'s why we can\'t track their order in class definition which means that we can\'t use them with `Command`. \nYou need to explicitly specify default value to convert simple annotation into field with annotation.\n\nIf you don\'t have sub-commands and don\'t have default values for arguments then it\'s okay to use them.\n\n```python\nclass Args(ArgsParse):\n    com = Command()\n    a: str = None  # now `a` is part of `com` subparser\n```\n\n2. explicitly specify type for arguments defined with `Argument` class to help your IDE\n\n```python\nclass Args(ArgsParse):\n    a: int = Argument(default=3)\n```\n\n`argser` will know about type of `a` without annotation (it can be determined by default value), \nbut if you want your IDE to know that `args.a` is `int` and not `Argument` then you need explicit annotation.\n',
    'author': 'Bachynin Ivan',
    'author_email': 'bachynin.i@gmail.com',
    'url': 'https://github.com/vaniakosmos/argser',
    'py_modules': modules,
    'extras_require': extras_require,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
