# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['sqlcon']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'sqlcon',
    'version': '0.1.3',
    'description': 'Construct indented SQL from Python.',
    'long_description': '![Tests on 3.6, 3.7, 3.8](https://github.com/tcorbettclark/sql-constructor/workflows/Tests/badge.svg) [![Known Vulnerabilities from Snyk](https://snyk.io/test/github/tcorbettclark/sql-constructor/badge.svg?targetFile=requirements.txt)](https://snyk.io/test/github/tcorbettclark/sql-constructor?targetFile=requirements.txt)\n\n# SQL Constructor\n\nA programming approach (and supporting functions) to programmatically construct\nand maintain well-formatted SQL from Python 3.6 and above.\n\n## Motivation and Philosophy\n\nLike all programming languages, the task of writing maintainable SQL is made a\nlittle bit easier through good formatting and layout, reducing the cognitive\nload on the programmers. Doing this with completely static SQL is possible e.g.\nwith static formatters, but is harder to achieve when *programmatically*\nconstructing the SQL. In my experience one easily looses the formating in *both*\nthe generating Python code *and* in the generated SQL. Ouch.\n\nThis package aims to solve this problem with a particular programming pattern\nand a small number of utility functions.\n\nIn particular, the objectives are:\n\n* To have the SQL look as much like SQL within the source Python code whilst\n  still using obvious Python to generate it. This involves trade-offs.\n* To have the generated SQL look like it could have been written directly so\n  that it can be read and understood as easily as possible.\n* To be able to write optimum SQL *in SQL* rather than trying to write optimum\n  SQL from another language but mentally thinking in the target SQL.\n\nIt should be clear that **SQL Constructor** is not an ORM. There is no abstraction.\nIn fact, quite the opposite.\n\n## SQL variants\n\nThis library has been used to generate SQL for PostgreSQL. However it should\nwork with other database engines / language variants with little or no change.\nSimilarly, it is agnostic to SQL code style conventions because that is\ncontrolled by the programmer.\n\nIn fact the whole idea is not really about SQL at all, but about managing text.\n\n## Example\n\nA simple example will illustrate the approach. From the perspective of SQL the\nsubquery is unnecessary; it is used purely to show composition.\n\n```python\nimport sqlcon\n\nsq = sqlcon.single_quote\ndq = sqlcon.double_quote\n\n\ndef select_columns(variables):\n    yield sqlcon.joinwith(dq(v) for v in variables)\n\n\ndef subquery():\n    yield """\n        SELECT\n            *\n        FROM\n            some_table\n        LEFT JOIN\n            some_other_table\n        USING\n            some_table.id = some_other_table.key\n    """, -1\n\n\ndef where_clauses(variables, conditions):\n    for condition in conditions:\n        variable, comparator, constant = condition\n        assert variable in variables, f"Unknown variable: {variable}"\n        assert comparator in ("=", "~"), f"Unknown comparator: {comparator}"\n        yield f"{dq(variable)} {comparator} {sq(constant)}"\n\n\ndef example(variables, conditions):\n    yield """\n        SELECT\n    """\n    yield 1, select_columns(variables), -1\n    yield """\n        FROM\n            (\n    """\n    yield 1, subquery(), -1\n    yield """\n            ) AS tmp\n        WHERE\n    """\n    yield sqlcon.indented_joinwith(\n        where_clauses(variables, conditions), separator=" AND "\n    )\n\n\nif __name__ == "__main__":\n    sql = example(\n        ["name", "age", "address"],\n        [("name", "=", "tim"), ("address", "~", "England")],\n    )\n    print(sqlcon.process(sql))\n\n```\n\nWhen run, this produces:\n\n```sql\nSELECT\n    "name",\n    "age",\n    "address"\nFROM\n    (\n        SELECT\n            *\n        FROM\n            some_table\n        LEFT JOIN\n            some_other_table\n        USING\n            some_table.id = some_other_table.key\n    ) AS tmp\nWHERE\n    "name" = \'tim\' AND\n    "address" ~ \'England\'\n```\n\nThe processing takes strings (for the actual SQL), integers (for manual\nindentation changes), and lists/tuples/generators for composition of the above.\nClearly this example is rather degenerate. It also mixes a few styles which is\ninconsistent but illustrates a few different approaches.\n\nNote how some relatively subtle things are happening automatically:\n\n1. Common indentation is being removed to left align the base of the generated SQL.\n1. Blank lines are being stripped intelligently. E.g. the start and end of the\n   tripple quoted strings.\n1. The indentation levels are being tracked. E.g. the subquery is indented in\n   the output but not in the `subquery()` function. So nested layers (such as\n   views within views or views within PostgreSQL functions) can be written\n   neatly without worrying about the indentation of their containing scope.\n\n## API\n\nThe **SQL Constructor** API consists of the following functions:\n\n* `process` to convert the SQL strings, indentation integers, and composition of\n  these as iterables all into output SQL.\n* `single_quote` to correctly quote literal strings for PostgreSQL.\n* `double_quote` to correctly quote identifiers for PostgresSQL.\n* `joinwith` to join lists with commas (e.g. for lists of column identifers or\n  "IN" clauses) or conditions with "AND" (e.g. for conjunctions in WHERE clauses).\n  Can be used either on a single line or across multiple lines with maintained\n  indentation.\n* `indented_joinwith` as a shorthand to prefix `joinwth` with an indent and\n  postfix with an outdent.\n\nSee the source code docstrings for details.\n\n## Tests\n\nSee the unittests in the `tests/` directory.\n\n# Alternatives\n\nBefore deciding to create **SQL Constructor** I tried various approaches "in\nanger" on real projects. Most notably:\n\n* Templating the SQL either with vanilla Jinja2 or variants designed to work\n  with SQL like JinjaSQL (e.g. https://github.com/hashedin/jinjasql).\n  Maintaining formatting is difficult, and now you are working in 2 files\n  (Python and the template) and 3 languages (templating, Python, and SQL).\n\n* Using an ORM like SQLAlchemy. Whilst superficially clean ("it\'s all just\n  Python"), anything more than the most trivial quickly requires mental\n  gymnastics *thinking* in SQL but *writing* in Python, and needing to drop out\n  of the Python API to create separate SQL support functions and views etc in\n  order to get the most from the database engine.\n\nNeither of these worked for me.\n\n# Contributions\n\nI\'d love to hear of other approaches people have tried. Perhaps there is a\nbetter way?! Similarly, pull-requests and bug reports etc are all welcome.\n\n# Release checklist\n\n* Run: `black .`\n* Run: `isort -y`\n* Run: `flake8 .`\n* Run: `nose2 -v tests`\n* Run: `poetry export -f requirements.txt >requirements.txt` (for snyk scanning)\n* Ensure `git tag`, the package version (via `poetry version`), and `sqlcon.__version__` are all equal!',
    'author': 'Timothy Corbett-Clark',
    'author_email': 'timothy.corbettclark@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/tcorbettclark/sql-constructor',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
