"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Domain = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const route53 = require("aws-cdk-lib/aws-route53");
const constructs_1 = require("constructs");
const certificate_1 = require("./certificate");
const add_1 = require("./errors/add");
const fqdn_1 = require("./fqdn");
const zone_1 = require("./zone");
class Domain extends constructs_1.Construct {
    /**
     * Initializing a `new Domain` construct instance will lookup the Route53 hosted zone
     * and define ACM DNS-validated certificate.
     *
     * After initialization you must use `assign(alias)` method to to configure `A`/`AAAA` records
     * with the `alias` as the record value.
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.assigned = false;
        const { certificate, enableIpv6 = true, region = 'us-east-1', subdomain, zone, } = props;
        this.enableIpv6 = enableIpv6;
        this.zone = zone_1.DomainZone.define(this, { zone });
        this.region = region;
        this.fqdn = new fqdn_1.FQDN(this, 'FQDN', {
            zone: this.zone,
            subdomain,
        }).value;
        this.certificate = certificate_1.DomainCertificate.define(this, {
            certificate,
            region: this.region,
            zone: this.zone,
            fqdn: this.fqdn,
        });
    }
    /**
     * Assign an alias as record target with the fully-qualified domain name.
     * This will create both `A` & `AAAA` DNS records, unless `disableIpV6` was set to `true`
     * during initialization of `Domain` construct (resulting in only `A` record being created).
     *
     * @param alias Route53 alias record target used to assign as A/AAAA record value.
     *
     * @example
     * domain.addTarget(new targets.CloudFrontTarget(distribution))
     */
    addTarget(alias) {
        // prevent accidental reassignment
        if (this.assigned) {
            add_1.addError(this, `Domain already assigned: ${this.fqdn}`);
            return;
        }
        // set assigned state flag
        this.assigned = true;
        // set IPv4 record
        new route53.ARecord(this, 'AliasRecordIpV4', {
            zone: this.zone,
            recordName: this.fqdn,
            target: route53.RecordTarget.fromAlias(alias),
        });
        // return early if IPv6 is disabled
        if (!this.enableIpv6)
            return;
        // set IPv6 record
        new route53.AaaaRecord(this, 'AliasRecordIpV6', {
            zone: this.zone,
            recordName: this.fqdn,
            target: route53.RecordTarget.fromAlias(alias),
        });
    }
    /**
     * Helper method to configure CloudFront distribution with the domain,
     * certificate and IPv6 support.
     *
     * @returns CloudFront configuration for certificate, domainNames and IPv6
     */
    configureCloudFront() {
        return {
            certificate: this.certificate,
            domainNames: [this.fqdn],
            enableIpv6: this.enableIpv6,
        };
    }
}
exports.Domain = Domain;
_a = JSII_RTTI_SYMBOL_1;
Domain[_a] = { fqn: "@alma-cdk/domain.Domain", version: "0.0.11" };
//# sourceMappingURL=data:application/json;base64,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