# AUTOGENERATED! DO NOT EDIT! File to edit: projects/m5/04_out_of_stock_detection.ipynb (unless otherwise specified).

__all__ = ['get_series_df', 'get_stats_df', 'plot_item_series', 'plot_all_item_series', 'make_oos_data']

# Cell
#export
import os
import time
import logging

import numpy as np
import pandas as pd
import matplotlib.pyplot as plt
# import seaborn as sns
from fastcore.script import call_parse, Param

from .metric import get_agg, get_df_weights
from ..core import time_taken, pool_func

# Cell
def get_series_df(train_df, rollup_matrix_csr, rollup_index, df_cal=None, fill_xmas=False):
    """Returns a dataframe with series for all 12 levels of aggregation. We also
    replace leading zeros with np.nan and if fill_xmas, replace christmas sales with average
    of the day before and day after christmas"""

    series_df = pd.DataFrame(data=rollup_matrix_csr * train_df.iloc[:, 6:].values,
                         index=rollup_index,
                         columns=train_df.iloc[:, 6:].columns)

    zero_mask = series_df.cumsum(axis=1) * 2 == series_df
    series_df[zero_mask] = np.nan

    if fill_xmas:
        xmas_days = df_cal[df_cal.date.str[-5:] == '12-25'].d.str[2:].astype('int16')
        for x in xmas_days:
            series_df[f'd_{x}'] = (series_df[f'd_{x-1}'] + series_df[f'd_{x+1}']) / 2
    return series_df


def get_stats_df(series_df):
    """Returns a dataframe that shows basic stats for all
    series in sereis_df."""

    percentiles = [.005, .025, .165, .25, .5, .75, .835, .975, .995]
    stats_df = series_df.T.describe(percentiles).T
    stats_df['fraction_0'] = ((series_df == 0).sum(axis = 1) / stats_df['count'])

    return stats_df

# Cell
def plot_item_series(item, series_df, state=None, fillna=False, start=0, end=1941):
    """Plots the level 10-12 series containing the item"""
    item_series_all = series_df.loc[series_df.index.get_level_values(1).str.contains(item)]
    if state:
        state_mask = item_series_all.index.get_level_values(1).str.contains(state)
        if fillna:
            item_series_all.loc[state_mask].iloc[:, start:end].fillna(fillna).T.plot(title=f'{item} overall in {state} and by store')
        else:
            item_series_all.loc[state_mask].iloc[:, start:end].T.plot(title=f'{item} overall in {state} and by store')
        plt.legend(bbox_to_anchor=(1,1.04), loc='lower right', ncol=1)
        for i in [1941 - 364*i for i in range(6) if start < 1941 - 364*i <= end]:
            plt.axvline(i, ls=':')
        plt.show()

    else:
        if fillna:
            item_series_all.iloc[:4, start:end].fillna(fillna).T.plot(title=f'{item} total and by state')
        else:
            item_series_all.iloc[:4, start:end].T.plot(title=f'{item} total and by state')
        plt.legend(bbox_to_anchor=(.5,.99), loc='upper center', ncol=1)
        for i in [1941 - 364*i for i in range(6) if start < 1941 - 364*i <= end]:
            plt.axvline(i, ls=':')
        plt.show()

def plot_all_item_series(item, series_df, fillna=False, start=0, end=1941):
    plot_item_series(item, series_df, state=None, fillna=fillna, start=start, end=end)
    for state in ['CA', 'TX', 'WI']:
        plot_item_series(item, series_df, state=state, fillna=fillna, start=start, end=end)

# Cell
@call_parse
def make_oos_data(PATH_DATA_RAW: Param('Path to raw data', str)='data/raw',
                  PATH_DATA_INTERIM: Param('Path to interim data', str)='data/interim') -> None:
    """Creates 2 csv files and stores them in the `PATH_DATA_INTERIM`.

    The first file is of all time series in the aggregation levels
    10, 11, and 12, stored as 'oos_series_df_level_10_11_12.csv'.

    The second file, 'oos_sales_train_evaluation.csv', has the same
    format as 'sales_train_evaluation.csv', except zero streaks
    that are believed to be caused by a stock-out are marked with
    nan.
    """

    logging.basicConfig(format='%(asctime)s   %(levelname)s   %(message)s',
                        level=logging.DEBUG,
                        filename='log.log')
    logging.info('#' * 72)
    logging.info('#' * 72)
    logging.info('Using ipca to reduce lag feature dimensions')
    start_time = time.time()

    df_stv = pd.read_csv(os.path.join(PATH_DATA_RAW, 'sales_train_evaluation.csv'))
    df_cal = pd.read_csv(os.path.join(PATH_DATA_RAW, 'calendar.csv'))
    rollup_matrix_csr, rollup_index  = get_agg(df_stv)
    series_df = get_series_df(df_stv, rollup_matrix_csr, rollup_index, df_cal=df_cal, fill_xmas=True)

    mask = series_df.index.get_level_values(0) > 9
    items = series_df[mask].index
    new_series = np.array([fix_oos(item, series_df)[0] for item in items])
    new_df = pd.DataFrame(new_series, columns=series_df.columns, index=series_df[mask].index)

    new_df.to_csv(os.path.join(PATH_DATA_INTERIM, 'oos_series_df_level_10_11_12.csv'))
    df_stv.iloc[:, 6:] = new_df.loc[12].values
    df_stv.to_csv(os.path.join(PATH_DATA_INTERIM, 'oos_sales_train_evaluation.csv'), index=False)

    logging.info(72 * '#')
    logging.info(time_taken(start_time))