# AUTOGENERATED! DO NOT EDIT! File to edit: projects/m5/training_day_by_day_models.ipynb (unless otherwise specified).

__all__ = ['load_cfg', 'prep_data', 'neptune', 'lgb_daily', 'cli_lgb_daily']

# Cell
#export
import os
import logging
import collections
import gc
import json
import time
import pickle
from itertools import chain
from pathlib import Path

import pandas as pd
import matplotlib.pyplot as plt
import lightgbm as lgb
from lightgbm.callback import record_evaluation, log_evaluation
from sklearn.model_selection import train_test_split
import neptune.new as neptune
from neptune.new.integrations.lightgbm import NeptuneCallback, create_booster_summary
from fastcore.script import call_parse, Param

from ..core import load_features, time_taken, load_file
from .fe import make_grid_df
from .metric import WRMSSE

# Cell
def load_cfg(path_cfg):
    if type(path_cfg) == str:
        with open(path_cfg, 'r') as f: return json.load(f)
    else: return  path_cfg

# Cell
def prep_data(cfg):
    df_stv = pd.read_csv(os.path.join(cfg['path_data_raw'], 'sales_train_evaluation.csv'))
    grid_df, _ = make_grid_df(df_stv)

    # Only use items with at least 68 days of sales
    first_sale = grid_df[grid_df.sales.notnull()].drop_duplicates('id')
    keep_id = first_sale[(cfg['start_test'] - first_sale.d) >= 68].id.tolist()
    df_stv_trunc = df_stv[df_stv.id.isin(keep_id)]
    grid_df = grid_df[grid_df.id.isin(keep_id)]

    #################### full valid and test sets ###################
    valid_days = [cfg['start_test'] + d - cfg['p_horizon'] for d in range(cfg['p_horizon'])]
    valid_actuals = df_stv_trunc[[f'd_{d}' for d in valid_days]].values
    e = WRMSSE(cfg['path_data_raw'], cfg['start_test'], df_stv_trunc=df_stv_trunc)
    if cfg['fobj_weight_col'] == 'total_scaled_weight': e.add_total_scaled_weight()
    if cfg['start_test'] != 1942: test_actuals = e.actuals.copy()
    prediction_df = df_stv_trunc[['id']]

    return grid_df, prediction_df, valid_actuals, e

# Cell
def neptune(cfg):
    if os.getenv('NEPTUNE_API_KEY'):
        pass

# Cell
def lgb_daily(path_cfg: str='cfg.json'):
    """Train 1 model for each day of prediction accoring to `path_cfg`."""
    cfg = load_cfg(path_cfg)
    full_grid_df, prediction_df, valid_actuals, e = prep_data(cfg)

    start_time = time.time()
    dict_eval_logs = [] # For experiment tracking
    ############### Day by day training and predicting #############
    if cfg['days_to_predict'] == "all": cfg['days_to_predict'] = range(28)
    for day_of_horizon in cfg['days_to_predict']:

        # Starting with full data and filtering for same day of week
        grid_df = full_grid_df.copy()
        test_day = cfg['start_test'] + day_of_horizon
        valid_day = test_day - 28
        same_day_of_week = [d for d in range(cfg['start_train'], test_day + 1) if d%7 == (test_day)%7]
        grid_df = grid_df[grid_df.d.isin(same_day_of_week)]
        with open(cfg['features_json'], 'r') as f:
                    dict_features = json.load(f)

        index = grid_df.index
        grid_df = pd.concat([
                    grid_df,
                    load_features(cfg['path_features'], dict_features, reindex_with=index,
                                  shift_index=cfg['num_series'] * day_of_horizon)
                ], axis=1)

        remove_features = ['id', 'd', cfg['target']]
        feature_cols = [col for col in list(grid_df) if col not in remove_features]

        ################## test, train and valid set ####################
        valid_mask = (grid_df.d == valid_day)
        train_mask = (grid_df.d >= cfg['start_train']) & (grid_df.d < valid_day) & (grid_df[cfg['target']].notnull())
        test_mask = (grid_df.d == test_day)

        train_x, train_y = grid_df[train_mask][feature_cols], grid_df[train_mask][cfg['target']]
        valid_x, valid_y = grid_df[valid_mask][feature_cols], grid_df[valid_mask][cfg['target']]
        test_x, test_y = grid_df[test_mask][feature_cols], grid_df[test_mask][cfg['target']]

        ################## Fit custom objective and metric ##################
        w_12_train = e.w_12.reindex(grid_df[train_mask].id)
        w_12_eval = e.w_12.reindex(grid_df[valid_mask].id)
        w_12_test = e.w_12.reindex(grid_df[test_mask].id)

        if cfg['fobj']:
            get_fobj = getattr(e, f'get_weighted_{cfg["fobj"]}_fobj')
            fobj = get_fobj(w_12_train, cfg['fobj_weight_col'], cfg['weight_hess'])
        else:
            fobj = None

        if cfg['feval']:
            if cfg['feval'] == 'feval':
                feval = e.feval
            else:
                get_feval = getattr(e, f'get_weighted_{cfg["feval"]}_feval')
                feval = get_feval(w_12_eval, cfg['feval_weight_col'])
        else:
            feval = None

        # Set evaluator actuals to valid day for early stopping
        e.actuals = valid_actuals[:, day_of_horizon].reshape((-1,1))

        ############# lightgbm datasets for training #############
        if cfg['weight_col']:
            weight_train = w_12_train[cfg['weight_col']].values
            weight_eval = w_12_eval[cfg['weight_col']].values
            weight_test = w_12_test[cfg['weight_col']].values
        else:
            weight_train, weight_eval, weight_test = None, None, None
        train_data = lgb.Dataset(train_x, label=train_y, weight=weight_train)
        valid_data = lgb.Dataset(valid_x, label=valid_y, weight=weight_eval)
        test_data = lgb.Dataset(test_x, label=test_y, weight=weight_test)

        ####################### Training ##########################
        dict_eval_log = {}
        estimator = lgb.train(
            cfg['lgb_params'],
            train_set=train_data,
            valid_sets=[valid_data],
            valid_names=['valid'],
            fobj = fobj,
            feval = feval,
            callbacks=[record_evaluation(dict_eval_log), log_evaluation(100)],
        )
        booster_summary = create_booster_summary(booster=estimator, max_num_features=25)
        cfg[f'bs_{day_of_horizon}'] = booster_summary
        dict_eval_logs.append(dict_eval_log)

        preds = estimator.predict(test_x)
        prediction_df.loc[:, f'F{day_of_horizon + 1}'] = preds
        gc.collect()

    # Saving predictions in submission ready format
    tmp = prediction_df.copy()
    prediction_df.id = prediction_df.id.str.replace('evaluation', 'validation')
    path = 'lgb_daily' + '_' + Path(path_cfg).stem + '_' + 'submission.csv'
    prediction_df.to_csv(path, index=False)

    neptune(cfg)
    time_taken(start_time)

@call_parse
def cli_lgb_daily(path_cfg: Param('path to the configuration json', str)='cfg.json'):
    lgb_daily(path_cfg)
