"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TagMutability = exports.Repository = exports.RepositoryBase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const os_1 = require("os");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const ecr_generated_1 = require("./ecr.generated");
const lifecycle_1 = require("./lifecycle");
/**
 * Base class for ECR repository.
 *
 * Reused between imported repositories and owned repositories.
 *
 * @stability stable
 */
class RepositoryBase extends core_1.Resource {
    /**
     * The URI of this repository (represents the latest image):.
     *
     * ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY
     *
     * @stability stable
     */
    get repositoryUri() {
        return this.repositoryUriForTag();
    }
    /**
     * Returns the URL of the repository. Can be used in `docker push/pull`.
     *
     * ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY[:TAG]
     *
     * @param tag Optional image tag.
     * @stability stable
     */
    repositoryUriForTag(tag) {
        const tagSuffix = tag ? `:${tag}` : '';
        return this.repositoryUriWithSuffix(tagSuffix);
    }
    /**
     * Returns the URL of the repository. Can be used in `docker push/pull`.
     *
     * ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY[@DIGEST]
     *
     * @param digest Optional image digest.
     * @stability stable
     */
    repositoryUriForDigest(digest) {
        const digestSuffix = digest ? `@${digest}` : '';
        return this.repositoryUriWithSuffix(digestSuffix);
    }
    /**
     * Returns the repository URI, with an appended suffix, if provided.
     * @param suffix An image tag or an image digest.
     * @private
     */
    repositoryUriWithSuffix(suffix) {
        const parts = this.stack.parseArn(this.repositoryArn);
        return `${parts.account}.dkr.ecr.${parts.region}.${this.stack.urlSuffix}/${this.repositoryName}${suffix}`;
    }
    /**
     * Define a CloudWatch event that triggers when something happens to this repository.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule.
     * @param options Options for adding the rule.
     * @stability stable
     */
    onCloudTrailEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.ecr'],
            detailType: ['AWS API Call via CloudTrail'],
            detail: {
                requestParameters: {
                    repositoryName: [this.repositoryName],
                },
            },
        });
        return rule;
    }
    /**
     * Defines an AWS CloudWatch event rule that can trigger a target when an image is pushed to this repository.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule.
     * @param options Options for adding the rule.
     * @stability stable
     */
    onCloudTrailImagePushed(id, options = {}) {
        const rule = this.onCloudTrailEvent(id, options);
        rule.addEventPattern({
            detail: {
                eventName: ['PutImage'],
                requestParameters: {
                    imageTag: options.imageTag ? [options.imageTag] : undefined,
                },
            },
        });
        return rule;
    }
    /**
     * Defines an AWS CloudWatch event rule that can trigger a target when an image scan is completed.
     *
     * @param id The id of the rule.
     * @param options Options for adding the rule.
     * @stability stable
     */
    onImageScanCompleted(id, options = {}) {
        var _c;
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.ecr'],
            detailType: ['ECR Image Scan'],
            detail: {
                'repository-name': [this.repositoryName],
                'scan-status': ['COMPLETE'],
                'image-tags': (_c = options.imageTags) !== null && _c !== void 0 ? _c : undefined,
            },
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers for repository events.
     *
     * Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     *
     * @stability stable
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.ecr'],
            resources: [this.repositoryArn],
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * Grant the given principal identity permissions to perform the actions on this repository.
     *
     * @stability stable
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipalOrResource({
            grantee,
            actions,
            resourceArns: [this.repositoryArn],
            resourceSelfArns: [],
            resource: this,
        });
    }
    /**
     * Grant the given identity permissions to use the images in this repository.
     *
     * @stability stable
     */
    grantPull(grantee) {
        const ret = this.grant(grantee, 'ecr:BatchCheckLayerAvailability', 'ecr:GetDownloadUrlForLayer', 'ecr:BatchGetImage');
        iam.Grant.addToPrincipal({
            grantee,
            actions: ['ecr:GetAuthorizationToken'],
            resourceArns: ['*'],
            scope: this,
        });
        return ret;
    }
    /**
     * Grant the given identity permissions to pull and push images to this repository.
     *
     * @stability stable
     */
    grantPullPush(grantee) {
        this.grantPull(grantee);
        return this.grant(grantee, 'ecr:PutImage', 'ecr:InitiateLayerUpload', 'ecr:UploadLayerPart', 'ecr:CompleteLayerUpload');
    }
}
exports.RepositoryBase = RepositoryBase;
_a = JSII_RTTI_SYMBOL_1;
RepositoryBase[_a] = { fqn: "@aws-cdk/aws-ecr.RepositoryBase", version: "1.119.0" };
/**
 * Define an ECR repository.
 *
 * @stability stable
 */
class Repository extends RepositoryBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.repositoryName,
        });
        this.lifecycleRules = new Array();
        Repository.validateRepositoryName(this.physicalName);
        const resource = new ecr_generated_1.CfnRepository(this, 'Resource', {
            repositoryName: this.physicalName,
            // It says "Text", but they actually mean "Object".
            repositoryPolicyText: core_1.Lazy.any({ produce: () => this.policyDocument }),
            lifecyclePolicy: core_1.Lazy.any({ produce: () => this.renderLifecyclePolicy() }),
            imageScanningConfiguration: !props.imageScanOnPush ? undefined : {
                scanOnPush: true,
            },
            imageTagMutability: props.imageTagMutability || undefined,
        });
        resource.applyRemovalPolicy(props.removalPolicy);
        this.registryId = props.lifecycleRegistryId;
        if (props.lifecycleRules) {
            props.lifecycleRules.forEach(this.addLifecycleRule.bind(this));
        }
        this.repositoryName = this.getResourceNameAttribute(resource.ref);
        this.repositoryArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'ecr',
            resource: 'repository',
            resourceName: this.physicalName,
        });
    }
    /**
     * Import a repository.
     *
     * @stability stable
     */
    static fromRepositoryAttributes(scope, id, attrs) {
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryName = attrs.repositoryName;
                this.repositoryArn = attrs.repositoryArn;
            }
            addToResourcePolicy(_statement) {
                // dropped
                return { statementAdded: false };
            }
        }
        return new Import(scope, id);
    }
    /**
     * @stability stable
     */
    static fromRepositoryArn(scope, id, repositoryArn) {
        // if repositoryArn is a token, the repository name is also required. this is because
        // repository names can include "/" (e.g. foo/bar/myrepo) and it is impossible to
        // parse the name from an ARN using CloudFormation's split/select.
        if (core_1.Token.isUnresolved(repositoryArn)) {
            throw new Error('"repositoryArn" is a late-bound value, and therefore "repositoryName" is required. Use `fromRepositoryAttributes` instead');
        }
        const repositoryName = repositoryArn.split('/').slice(1).join('/');
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryName = repositoryName;
                this.repositoryArn = repositoryArn;
            }
            addToResourcePolicy(_statement) {
                // dropped
                return { statementAdded: false };
            }
        }
        return new Import(scope, id);
    }
    /**
     * @stability stable
     */
    static fromRepositoryName(scope, id, repositoryName) {
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryName = repositoryName;
                this.repositoryArn = Repository.arnForLocalRepository(repositoryName, scope);
            }
            addToResourcePolicy(_statement) {
                // dropped
                return { statementAdded: false };
            }
        }
        return new Import(scope, id);
    }
    /**
     * Returns an ECR ARN for a repository that resides in the same account/region as the current stack.
     *
     * @stability stable
     */
    static arnForLocalRepository(repositoryName, scope, account) {
        return core_1.Stack.of(scope).formatArn({
            account,
            service: 'ecr',
            resource: 'repository',
            resourceName: repositoryName,
        });
    }
    static validateRepositoryName(physicalName) {
        const repositoryName = physicalName;
        if (!repositoryName || core_1.Token.isUnresolved(repositoryName)) {
            // the name is a late-bound value, not a defined string,
            // so skip validation
            return;
        }
        const errors = [];
        // Rules codified from https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecr-repository.html
        if (repositoryName.length < 2 || repositoryName.length > 256) {
            errors.push('Repository name must be at least 2 and no more than 256 characters');
        }
        const isPatternMatch = /^(?:[a-z0-9]+(?:[._-][a-z0-9]+)*\/)*[a-z0-9]+(?:[._-][a-z0-9]+)*$/.test(repositoryName);
        if (!isPatternMatch) {
            errors.push('Repository name must follow the specified pattern: (?:[a-z0-9]+(?:[._-][a-z0-9]+)*/)*[a-z0-9]+(?:[._-][a-z0-9]+)*');
        }
        if (errors.length > 0) {
            throw new Error(`Invalid ECR repository name (value: ${repositoryName})${os_1.EOL}${errors.join(os_1.EOL)}`);
        }
    }
    /**
     * Add a policy statement to the repository's resource policy.
     *
     * @stability stable
     */
    addToResourcePolicy(statement) {
        if (this.policyDocument === undefined) {
            this.policyDocument = new iam.PolicyDocument();
        }
        this.policyDocument.addStatements(statement);
        return { statementAdded: false, policyDependable: this.policyDocument };
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    validate() {
        var _c;
        const errors = super.validate();
        errors.push(...((_c = this.policyDocument) === null || _c === void 0 ? void 0 : _c.validateForResourcePolicy()) || []);
        return errors;
    }
    /**
     * Add a life cycle rule to the repository.
     *
     * Life cycle rules automatically expire images from the repository that match
     * certain conditions.
     *
     * @stability stable
     */
    addLifecycleRule(rule) {
        // Validate rule here so users get errors at the expected location
        if (rule.tagStatus === undefined) {
            rule = { ...rule, tagStatus: rule.tagPrefixList === undefined ? lifecycle_1.TagStatus.ANY : lifecycle_1.TagStatus.TAGGED };
        }
        if (rule.tagStatus === lifecycle_1.TagStatus.TAGGED && (rule.tagPrefixList === undefined || rule.tagPrefixList.length === 0)) {
            throw new Error('TagStatus.Tagged requires the specification of a tagPrefixList');
        }
        if (rule.tagStatus !== lifecycle_1.TagStatus.TAGGED && rule.tagPrefixList !== undefined) {
            throw new Error('tagPrefixList can only be specified when tagStatus is set to Tagged');
        }
        if ((rule.maxImageAge !== undefined) === (rule.maxImageCount !== undefined)) {
            throw new Error(`Life cycle rule must contain exactly one of 'maxImageAge' and 'maxImageCount', got: ${JSON.stringify(rule)}`);
        }
        if (rule.tagStatus === lifecycle_1.TagStatus.ANY && this.lifecycleRules.filter(r => r.tagStatus === lifecycle_1.TagStatus.ANY).length > 0) {
            throw new Error('Life cycle can only have one TagStatus.Any rule');
        }
        this.lifecycleRules.push({ ...rule });
    }
    /**
     * Render the life cycle policy object
     */
    renderLifecyclePolicy() {
        const stack = core_1.Stack.of(this);
        let lifecyclePolicyText;
        if (this.lifecycleRules.length === 0 && !this.registryId) {
            return undefined;
        }
        if (this.lifecycleRules.length > 0) {
            lifecyclePolicyText = JSON.stringify(stack.resolve({
                rules: this.orderedLifecycleRules().map(renderLifecycleRule),
            }));
        }
        return {
            lifecyclePolicyText,
            registryId: this.registryId,
        };
    }
    /**
     * Return life cycle rules with automatic ordering applied.
     *
     * Also applies validation of the 'any' rule.
     */
    orderedLifecycleRules() {
        var _c;
        if (this.lifecycleRules.length === 0) {
            return [];
        }
        const prioritizedRules = this.lifecycleRules.filter(r => r.rulePriority !== undefined && r.tagStatus !== lifecycle_1.TagStatus.ANY);
        const autoPrioritizedRules = this.lifecycleRules.filter(r => r.rulePriority === undefined && r.tagStatus !== lifecycle_1.TagStatus.ANY);
        const anyRules = this.lifecycleRules.filter(r => r.tagStatus === lifecycle_1.TagStatus.ANY);
        if (anyRules.length > 0 && anyRules[0].rulePriority !== undefined && autoPrioritizedRules.length > 0) {
            // Supporting this is too complex for very little value. We just prohibit it.
            throw new Error("Cannot combine prioritized TagStatus.Any rule with unprioritized rules. Remove rulePriority from the 'Any' rule.");
        }
        const prios = prioritizedRules.map(r => r.rulePriority);
        let autoPrio = (prios.length > 0 ? Math.max(...prios) : 0) + 1;
        const ret = new Array();
        for (const rule of prioritizedRules.concat(autoPrioritizedRules).concat(anyRules)) {
            ret.push({
                ...rule,
                rulePriority: (_c = rule.rulePriority) !== null && _c !== void 0 ? _c : autoPrio++,
            });
        }
        // Do validation on the final array--might still be wrong because the user supplied all prios, but incorrectly.
        validateAnyRuleLast(ret);
        return ret;
    }
}
exports.Repository = Repository;
_b = JSII_RTTI_SYMBOL_1;
Repository[_b] = { fqn: "@aws-cdk/aws-ecr.Repository", version: "1.119.0" };
function validateAnyRuleLast(rules) {
    const anyRules = rules.filter(r => r.tagStatus === lifecycle_1.TagStatus.ANY);
    if (anyRules.length === 1) {
        const maxPrio = Math.max(...rules.map(r => r.rulePriority));
        if (anyRules[0].rulePriority !== maxPrio) {
            throw new Error(`TagStatus.Any rule must have highest priority, has ${anyRules[0].rulePriority} which is smaller than ${maxPrio}`);
        }
    }
}
/**
 * Render the lifecycle rule to JSON
 */
function renderLifecycleRule(rule) {
    var _c, _d;
    return {
        rulePriority: rule.rulePriority,
        description: rule.description,
        selection: {
            tagStatus: rule.tagStatus || lifecycle_1.TagStatus.ANY,
            tagPrefixList: rule.tagPrefixList,
            countType: rule.maxImageAge !== undefined ? "sinceImagePushed" /* SINCE_IMAGE_PUSHED */ : "imageCountMoreThan" /* IMAGE_COUNT_MORE_THAN */,
            countNumber: (_d = (_c = rule.maxImageAge) === null || _c === void 0 ? void 0 : _c.toDays()) !== null && _d !== void 0 ? _d : rule.maxImageCount,
            countUnit: rule.maxImageAge !== undefined ? 'days' : undefined,
        },
        action: {
            type: 'expire',
        },
    };
}
/**
 * The tag mutability setting for your repository.
 *
 * @stability stable
 */
var TagMutability;
(function (TagMutability) {
    TagMutability["MUTABLE"] = "MUTABLE";
    TagMutability["IMMUTABLE"] = "IMMUTABLE";
})(TagMutability = exports.TagMutability || (exports.TagMutability = {}));
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVwb3NpdG9yeS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInJlcG9zaXRvcnkudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSwyQkFBeUI7QUFDekIsOENBQThDO0FBQzlDLHdDQUF3QztBQUN4Qyx3Q0FBdUY7QUFFdkYsbURBQWdEO0FBQ2hELDJDQUF1RDs7Ozs7Ozs7QUE2Q3ZELE1BQXNCLGNBQWUsU0FBUSxlQUFROzs7Ozs7OztJQVduRCxJQUFXLGFBQWE7UUFDdEIsT0FBTyxJQUFJLENBQUMsbUJBQW1CLEVBQUUsQ0FBQztJQUNwQyxDQUFDOzs7Ozs7Ozs7SUFHTSxtQkFBbUIsQ0FBQyxHQUFZO1FBQ3JDLE1BQU0sU0FBUyxHQUFHLEdBQUcsQ0FBQyxDQUFDLENBQUMsSUFBSSxHQUFHLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDLHVCQUF1QixDQUFDLFNBQVMsQ0FBQyxDQUFDO0lBQ2pELENBQUM7Ozs7Ozs7OztJQUdNLHNCQUFzQixDQUFDLE1BQWU7UUFDM0MsTUFBTSxZQUFZLEdBQUcsTUFBTSxDQUFDLENBQUMsQ0FBQyxJQUFJLE1BQU0sRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7UUFDaEQsT0FBTyxJQUFJLENBQUMsdUJBQXVCLENBQUMsWUFBWSxDQUFDLENBQUM7SUFDcEQsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyx1QkFBdUIsQ0FBQyxNQUFlO1FBQzdDLE1BQU0sS0FBSyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUN0RCxPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sWUFBWSxLQUFLLENBQUMsTUFBTSxJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxJQUFJLElBQUksQ0FBQyxjQUFjLEdBQUcsTUFBTSxFQUFFLENBQUM7SUFDNUcsQ0FBQzs7Ozs7Ozs7Ozs7SUFHTSxpQkFBaUIsQ0FBQyxFQUFVLEVBQUUsVUFBaUMsRUFBRTtRQUN0RSxNQUFNLElBQUksR0FBRyxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRSxPQUFPLENBQUMsQ0FBQztRQUNoRCxJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUMvQixJQUFJLENBQUMsZUFBZSxDQUFDO1lBQ25CLE1BQU0sRUFBRSxDQUFDLFNBQVMsQ0FBQztZQUNuQixVQUFVLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQztZQUMzQyxNQUFNLEVBQUU7Z0JBQ04saUJBQWlCLEVBQUU7b0JBQ2pCLGNBQWMsRUFBRSxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUM7aUJBQ3RDO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFDSCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7Ozs7Ozs7Ozs7O0lBR00sdUJBQXVCLENBQUMsRUFBVSxFQUFFLFVBQTBDLEVBQUU7UUFDckYsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEVBQUUsRUFBRSxPQUFPLENBQUMsQ0FBQztRQUNqRCxJQUFJLENBQUMsZUFBZSxDQUFDO1lBQ25CLE1BQU0sRUFBRTtnQkFDTixTQUFTLEVBQUUsQ0FBQyxVQUFVLENBQUM7Z0JBQ3ZCLGlCQUFpQixFQUFFO29CQUNqQixRQUFRLEVBQUUsT0FBTyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVM7aUJBQzVEO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFDSCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7Ozs7Ozs7O0lBRU0sb0JBQW9CLENBQUMsRUFBVSxFQUFFLFVBQXVDLEVBQUU7O1FBQy9FLE1BQU0sSUFBSSxHQUFHLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsRUFBRSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQ2hELElBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQy9CLElBQUksQ0FBQyxlQUFlLENBQUM7WUFDbkIsTUFBTSxFQUFFLENBQUMsU0FBUyxDQUFDO1lBQ25CLFVBQVUsRUFBRSxDQUFDLGdCQUFnQixDQUFDO1lBQzlCLE1BQU0sRUFBRTtnQkFDTixpQkFBaUIsRUFBRSxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUM7Z0JBQ3hDLGFBQWEsRUFBRSxDQUFDLFVBQVUsQ0FBQztnQkFDM0IsWUFBWSxRQUFFLE9BQU8sQ0FBQyxTQUFTLG1DQUFJLFNBQVM7YUFDN0M7U0FDRixDQUFDLENBQUM7UUFDSCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7Ozs7Ozs7OztJQUdNLE9BQU8sQ0FBQyxFQUFVLEVBQUUsVUFBaUMsRUFBRTtRQUM1RCxNQUFNLElBQUksR0FBRyxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRSxPQUFPLENBQUMsQ0FBQztRQUNoRCxJQUFJLENBQUMsZUFBZSxDQUFDO1lBQ25CLE1BQU0sRUFBRSxDQUFDLFNBQVMsQ0FBQztZQUNuQixTQUFTLEVBQUUsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDO1NBQ2hDLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQy9CLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQzs7Ozs7O0lBRU0sS0FBSyxDQUFDLE9BQXVCLEVBQUUsR0FBRyxPQUFpQjtRQUN4RCxPQUFPLEdBQUcsQ0FBQyxLQUFLLENBQUMsd0JBQXdCLENBQUM7WUFDeEMsT0FBTztZQUNQLE9BQU87WUFDUCxZQUFZLEVBQUUsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDO1lBQ2xDLGdCQUFnQixFQUFFLEVBQUU7WUFDcEIsUUFBUSxFQUFFLElBQUk7U0FDZixDQUFDLENBQUM7SUFDTCxDQUFDOzs7Ozs7SUFHTSxTQUFTLENBQUMsT0FBdUI7UUFDdEMsTUFBTSxHQUFHLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLEVBQUUsaUNBQWlDLEVBQUUsNEJBQTRCLEVBQUUsbUJBQW1CLENBQUMsQ0FBQztRQUV0SCxHQUFHLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQztZQUN2QixPQUFPO1lBQ1AsT0FBTyxFQUFFLENBQUMsMkJBQTJCLENBQUM7WUFDdEMsWUFBWSxFQUFFLENBQUMsR0FBRyxDQUFDO1lBQ25CLEtBQUssRUFBRSxJQUFJO1NBQ1osQ0FBQyxDQUFDO1FBRUgsT0FBTyxHQUFHLENBQUM7SUFDYixDQUFDOzs7Ozs7SUFHTSxhQUFhLENBQUMsT0FBdUI7UUFDMUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUN4QixPQUFPLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxFQUN2QixjQUFjLEVBQ2QseUJBQXlCLEVBQ3pCLHFCQUFxQixFQUNyQix5QkFBeUIsQ0FBQyxDQUFDO0lBQy9CLENBQUM7O0FBN0hILHdDQThIQzs7Ozs7Ozs7QUF3Q0QsTUFBYSxVQUFXLFNBQVEsY0FBYzs7OztJQStGNUMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxRQUF5QixFQUFFO1FBQ25FLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ2YsWUFBWSxFQUFFLEtBQUssQ0FBQyxjQUFjO1NBQ25DLENBQUMsQ0FBQztRQVBZLG1CQUFjLEdBQUcsSUFBSSxLQUFLLEVBQWlCLENBQUM7UUFTM0QsVUFBVSxDQUFDLHNCQUFzQixDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQztRQUVyRCxNQUFNLFFBQVEsR0FBRyxJQUFJLDZCQUFhLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUNuRCxjQUFjLEVBQUUsSUFBSSxDQUFDLFlBQVk7WUFDakMsbURBQW1EO1lBQ25ELG9CQUFvQixFQUFFLFdBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLGNBQWMsRUFBRSxDQUFDO1lBQ3RFLGVBQWUsRUFBRSxXQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxxQkFBcUIsRUFBRSxFQUFFLENBQUM7WUFDMUUsMEJBQTBCLEVBQUUsQ0FBQyxLQUFLLENBQUMsZUFBZSxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDO2dCQUMvRCxVQUFVLEVBQUUsSUFBSTthQUNqQjtZQUNELGtCQUFrQixFQUFFLEtBQUssQ0FBQyxrQkFBa0IsSUFBSSxTQUFTO1NBQzFELENBQUMsQ0FBQztRQUVILFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUM7UUFFakQsSUFBSSxDQUFDLFVBQVUsR0FBRyxLQUFLLENBQUMsbUJBQW1CLENBQUM7UUFDNUMsSUFBSSxLQUFLLENBQUMsY0FBYyxFQUFFO1lBQ3hCLEtBQUssQ0FBQyxjQUFjLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztTQUNoRTtRQUVELElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSxDQUFDLHdCQUF3QixDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUNsRSxJQUFJLENBQUMsYUFBYSxHQUFHLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxRQUFRLENBQUMsT0FBTyxFQUFFO1lBQ2xFLE9BQU8sRUFBRSxLQUFLO1lBQ2QsUUFBUSxFQUFFLFlBQVk7WUFDdEIsWUFBWSxFQUFFLElBQUksQ0FBQyxZQUFZO1NBQ2hDLENBQUMsQ0FBQztJQUNMLENBQUM7Ozs7OztJQTVITSxNQUFNLENBQUMsd0JBQXdCLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBMkI7UUFDOUYsTUFBTSxNQUFPLFNBQVEsY0FBYztZQUFuQzs7Z0JBQ2tCLG1CQUFjLEdBQUcsS0FBSyxDQUFDLGNBQWMsQ0FBQztnQkFDdEMsa0JBQWEsR0FBRyxLQUFLLENBQUMsYUFBYSxDQUFDO1lBTXRELENBQUM7WUFKUSxtQkFBbUIsQ0FBQyxVQUErQjtnQkFDeEQsVUFBVTtnQkFDVixPQUFPLEVBQUUsY0FBYyxFQUFFLEtBQUssRUFBRSxDQUFDO1lBQ25DLENBQUM7U0FDRjtRQUVELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQy9CLENBQUM7Ozs7SUFFTSxNQUFNLENBQUMsaUJBQWlCLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsYUFBcUI7UUFFakYscUZBQXFGO1FBQ3JGLGlGQUFpRjtRQUNqRixrRUFBa0U7UUFDbEUsSUFBSSxZQUFLLENBQUMsWUFBWSxDQUFDLGFBQWEsQ0FBQyxFQUFFO1lBQ3JDLE1BQU0sSUFBSSxLQUFLLENBQUMsMkhBQTJILENBQUMsQ0FBQztTQUM5STtRQUVELE1BQU0sY0FBYyxHQUFHLGFBQWEsQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUVuRSxNQUFNLE1BQU8sU0FBUSxjQUFjO1lBQW5DOztnQkFDUyxtQkFBYyxHQUFHLGNBQWMsQ0FBQztnQkFDaEMsa0JBQWEsR0FBRyxhQUFhLENBQUM7WUFNdkMsQ0FBQztZQUpRLG1CQUFtQixDQUFDLFVBQStCO2dCQUN4RCxVQUFVO2dCQUNWLE9BQU8sRUFBRSxjQUFjLEVBQUUsS0FBSyxFQUFFLENBQUM7WUFDbkMsQ0FBQztTQUNGO1FBRUQsT0FBTyxJQUFJLE1BQU0sQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDL0IsQ0FBQzs7OztJQUVNLE1BQU0sQ0FBQyxrQkFBa0IsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxjQUFzQjtRQUNuRixNQUFNLE1BQU8sU0FBUSxjQUFjO1lBQW5DOztnQkFDUyxtQkFBYyxHQUFHLGNBQWMsQ0FBQztnQkFDaEMsa0JBQWEsR0FBRyxVQUFVLENBQUMscUJBQXFCLENBQUMsY0FBYyxFQUFFLEtBQUssQ0FBQyxDQUFDO1lBTWpGLENBQUM7WUFKUSxtQkFBbUIsQ0FBQyxVQUErQjtnQkFDeEQsVUFBVTtnQkFDVixPQUFPLEVBQUUsY0FBYyxFQUFFLEtBQUssRUFBRSxDQUFDO1lBQ25DLENBQUM7U0FDRjtRQUVELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQy9CLENBQUM7Ozs7OztJQUdNLE1BQU0sQ0FBQyxxQkFBcUIsQ0FBQyxjQUFzQixFQUFFLEtBQWlCLEVBQUUsT0FBZ0I7UUFDN0YsT0FBTyxZQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLFNBQVMsQ0FBQztZQUMvQixPQUFPO1lBQ1AsT0FBTyxFQUFFLEtBQUs7WUFDZCxRQUFRLEVBQUUsWUFBWTtZQUN0QixZQUFZLEVBQUUsY0FBYztTQUM3QixDQUFDLENBQUM7SUFDTCxDQUFDO0lBR08sTUFBTSxDQUFDLHNCQUFzQixDQUFDLFlBQW9CO1FBQ3hELE1BQU0sY0FBYyxHQUFHLFlBQVksQ0FBQztRQUNwQyxJQUFJLENBQUMsY0FBYyxJQUFJLFlBQUssQ0FBQyxZQUFZLENBQUMsY0FBYyxDQUFDLEVBQUU7WUFDekQsd0RBQXdEO1lBQ3hELHFCQUFxQjtZQUNyQixPQUFPO1NBQ1I7UUFFRCxNQUFNLE1BQU0sR0FBYSxFQUFFLENBQUM7UUFFNUIsc0hBQXNIO1FBQ3RILElBQUksY0FBYyxDQUFDLE1BQU0sR0FBRyxDQUFDLElBQUksY0FBYyxDQUFDLE1BQU0sR0FBRyxHQUFHLEVBQUU7WUFDNUQsTUFBTSxDQUFDLElBQUksQ0FBQyxvRUFBb0UsQ0FBQyxDQUFDO1NBQ25GO1FBQ0QsTUFBTSxjQUFjLEdBQUcsbUVBQW1FLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBQ2hILElBQUksQ0FBQyxjQUFjLEVBQUU7WUFDbkIsTUFBTSxDQUFDLElBQUksQ0FBQyxtSEFBbUgsQ0FBQyxDQUFDO1NBQ2xJO1FBRUQsSUFBSSxNQUFNLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUNyQixNQUFNLElBQUksS0FBSyxDQUFDLHVDQUF1QyxjQUFjLElBQUksUUFBRyxHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUMsUUFBRyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1NBQ3BHO0lBQ0gsQ0FBQzs7Ozs7O0lBeUNNLG1CQUFtQixDQUFDLFNBQThCO1FBQ3ZELElBQUksSUFBSSxDQUFDLGNBQWMsS0FBSyxTQUFTLEVBQUU7WUFDckMsSUFBSSxDQUFDLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxjQUFjLEVBQUUsQ0FBQztTQUNoRDtRQUNELElBQUksQ0FBQyxjQUFjLENBQUMsYUFBYSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQzdDLE9BQU8sRUFBRSxjQUFjLEVBQUUsS0FBSyxFQUFFLGdCQUFnQixFQUFFLElBQUksQ0FBQyxjQUFjLEVBQUUsQ0FBQztJQUMxRSxDQUFDOzs7Ozs7Ozs7SUFFUyxRQUFROztRQUNoQixNQUFNLE1BQU0sR0FBRyxLQUFLLENBQUMsUUFBUSxFQUFFLENBQUM7UUFDaEMsTUFBTSxDQUFDLElBQUksQ0FBQyxHQUFHLE9BQUEsSUFBSSxDQUFDLGNBQWMsMENBQUUseUJBQXlCLE9BQU0sRUFBRSxDQUFDLENBQUM7UUFDdkUsT0FBTyxNQUFNLENBQUM7SUFDaEIsQ0FBQzs7Ozs7Ozs7O0lBR00sZ0JBQWdCLENBQUMsSUFBbUI7UUFDekMsa0VBQWtFO1FBQ2xFLElBQUksSUFBSSxDQUFDLFNBQVMsS0FBSyxTQUFTLEVBQUU7WUFDaEMsSUFBSSxHQUFHLEVBQUUsR0FBRyxJQUFJLEVBQUUsU0FBUyxFQUFFLElBQUksQ0FBQyxhQUFhLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxxQkFBUyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMscUJBQVMsQ0FBQyxNQUFNLEVBQUUsQ0FBQztTQUNwRztRQUVELElBQUksSUFBSSxDQUFDLFNBQVMsS0FBSyxxQkFBUyxDQUFDLE1BQU0sSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLEtBQUssU0FBUyxJQUFJLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxLQUFLLENBQUMsQ0FBQyxFQUFFO1lBQ2hILE1BQU0sSUFBSSxLQUFLLENBQUMsZ0VBQWdFLENBQUMsQ0FBQztTQUNuRjtRQUNELElBQUksSUFBSSxDQUFDLFNBQVMsS0FBSyxxQkFBUyxDQUFDLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYSxLQUFLLFNBQVMsRUFBRTtZQUMzRSxNQUFNLElBQUksS0FBSyxDQUFDLHFFQUFxRSxDQUFDLENBQUM7U0FDeEY7UUFDRCxJQUFJLENBQUMsSUFBSSxDQUFDLFdBQVcsS0FBSyxTQUFTLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxhQUFhLEtBQUssU0FBUyxDQUFDLEVBQUU7WUFDM0UsTUFBTSxJQUFJLEtBQUssQ0FBQyx1RkFBdUYsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7U0FDaEk7UUFFRCxJQUFJLElBQUksQ0FBQyxTQUFTLEtBQUsscUJBQVMsQ0FBQyxHQUFHLElBQUksSUFBSSxDQUFDLGNBQWMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUyxLQUFLLHFCQUFTLENBQUMsR0FBRyxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUNqSCxNQUFNLElBQUksS0FBSyxDQUFDLGlEQUFpRCxDQUFDLENBQUM7U0FDcEU7UUFFRCxJQUFJLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxFQUFFLEdBQUcsSUFBSSxFQUFFLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7O09BRUc7SUFDSyxxQkFBcUI7UUFDM0IsTUFBTSxLQUFLLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUM3QixJQUFJLG1CQUF3QixDQUFDO1FBRTdCLElBQUksSUFBSSxDQUFDLGNBQWMsQ0FBQyxNQUFNLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRTtZQUFFLE9BQU8sU0FBUyxDQUFDO1NBQUU7UUFFL0UsSUFBSSxJQUFJLENBQUMsY0FBYyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDbEMsbUJBQW1CLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDO2dCQUNqRCxLQUFLLEVBQUUsSUFBSSxDQUFDLHFCQUFxQixFQUFFLENBQUMsR0FBRyxDQUFDLG1CQUFtQixDQUFDO2FBQzdELENBQUMsQ0FBQyxDQUFDO1NBQ0w7UUFFRCxPQUFPO1lBQ0wsbUJBQW1CO1lBQ25CLFVBQVUsRUFBRSxJQUFJLENBQUMsVUFBVTtTQUM1QixDQUFDO0lBQ0osQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxxQkFBcUI7O1FBQzNCLElBQUksSUFBSSxDQUFDLGNBQWMsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQUUsT0FBTyxFQUFFLENBQUM7U0FBRTtRQUVwRCxNQUFNLGdCQUFnQixHQUFHLElBQUksQ0FBQyxjQUFjLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLFlBQVksS0FBSyxTQUFTLElBQUksQ0FBQyxDQUFDLFNBQVMsS0FBSyxxQkFBUyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ3hILE1BQU0sb0JBQW9CLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsWUFBWSxLQUFLLFNBQVMsSUFBSSxDQUFDLENBQUMsU0FBUyxLQUFLLHFCQUFTLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDNUgsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUyxLQUFLLHFCQUFTLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDaEYsSUFBSSxRQUFRLENBQUMsTUFBTSxHQUFHLENBQUMsSUFBSSxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsWUFBWSxLQUFLLFNBQVMsSUFBSSxvQkFBb0IsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQ3BHLDZFQUE2RTtZQUM3RSxNQUFNLElBQUksS0FBSyxDQUFDLGtIQUFrSCxDQUFDLENBQUM7U0FDckk7UUFFRCxNQUFNLEtBQUssR0FBRyxnQkFBZ0IsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsWUFBYSxDQUFDLENBQUM7UUFDekQsSUFBSSxRQUFRLEdBQUcsQ0FBQyxLQUFLLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxHQUFHLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUM7UUFFL0QsTUFBTSxHQUFHLEdBQUcsSUFBSSxLQUFLLEVBQWlCLENBQUM7UUFDdkMsS0FBSyxNQUFNLElBQUksSUFBSSxnQkFBZ0IsQ0FBQyxNQUFNLENBQUMsb0JBQW9CLENBQUMsQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLEVBQUU7WUFDakYsR0FBRyxDQUFDLElBQUksQ0FBQztnQkFDUCxHQUFHLElBQUk7Z0JBQ1AsWUFBWSxRQUFFLElBQUksQ0FBQyxZQUFZLG1DQUFJLFFBQVEsRUFBRTthQUM5QyxDQUFDLENBQUM7U0FDSjtRQUVELCtHQUErRztRQUMvRyxtQkFBbUIsQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUN6QixPQUFPLEdBQUcsQ0FBQztJQUNiLENBQUM7O0FBek5ILGdDQTBOQzs7O0FBRUQsU0FBUyxtQkFBbUIsQ0FBQyxLQUFzQjtJQUNqRCxNQUFNLFFBQVEsR0FBRyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLFNBQVMsS0FBSyxxQkFBUyxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ2xFLElBQUksUUFBUSxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7UUFDekIsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxHQUFHLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsWUFBYSxDQUFDLENBQUMsQ0FBQztRQUM3RCxJQUFJLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxZQUFZLEtBQUssT0FBTyxFQUFFO1lBQ3hDLE1BQU0sSUFBSSxLQUFLLENBQUMsc0RBQXNELFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxZQUFZLDBCQUEwQixPQUFPLEVBQUUsQ0FBQyxDQUFDO1NBQ3BJO0tBQ0Y7QUFDSCxDQUFDO0FBRUQ7O0dBRUc7QUFDSCxTQUFTLG1CQUFtQixDQUFDLElBQW1COztJQUM5QyxPQUFPO1FBQ0wsWUFBWSxFQUFFLElBQUksQ0FBQyxZQUFZO1FBQy9CLFdBQVcsRUFBRSxJQUFJLENBQUMsV0FBVztRQUM3QixTQUFTLEVBQUU7WUFDVCxTQUFTLEVBQUUsSUFBSSxDQUFDLFNBQVMsSUFBSSxxQkFBUyxDQUFDLEdBQUc7WUFDMUMsYUFBYSxFQUFFLElBQUksQ0FBQyxhQUFhO1lBQ2pDLFNBQVMsRUFBRSxJQUFJLENBQUMsV0FBVyxLQUFLLFNBQVMsQ0FBQyxDQUFDLDZDQUE4QixDQUFDLGlEQUFnQztZQUMxRyxXQUFXLGNBQUUsSUFBSSxDQUFDLFdBQVcsMENBQUUsTUFBTSxxQ0FBTSxJQUFJLENBQUMsYUFBYTtZQUM3RCxTQUFTLEVBQUUsSUFBSSxDQUFDLFdBQVcsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsU0FBUztTQUMvRDtRQUNELE1BQU0sRUFBRTtZQUNOLElBQUksRUFBRSxRQUFRO1NBQ2Y7S0FDRixDQUFDO0FBQ0osQ0FBQzs7Ozs7O0FBa0JELElBQVksYUFPWDtBQVBELFdBQVksYUFBYTtJQUV2QixvQ0FBbUIsQ0FBQTtJQUduQix3Q0FBdUIsQ0FBQTtBQUV6QixDQUFDLEVBUFcsYUFBYSxHQUFiLHFCQUFhLEtBQWIscUJBQWEsUUFPeEIiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBFT0wgfSBmcm9tICdvcyc7XG5pbXBvcnQgKiBhcyBldmVudHMgZnJvbSAnQGF3cy1jZGsvYXdzLWV2ZW50cyc7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgeyBJUmVzb3VyY2UsIExhenksIFJlbW92YWxQb2xpY3ksIFJlc291cmNlLCBTdGFjaywgVG9rZW4gfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IElDb25zdHJ1Y3QsIENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgQ2ZuUmVwb3NpdG9yeSB9IGZyb20gJy4vZWNyLmdlbmVyYXRlZCc7XG5pbXBvcnQgeyBMaWZlY3ljbGVSdWxlLCBUYWdTdGF0dXMgfSBmcm9tICcuL2xpZmVjeWNsZSc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgSVJlcG9zaXRvcnkgZXh0ZW5kcyBJUmVzb3VyY2Uge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcmVwb3NpdG9yeU5hbWU6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSByZXBvc2l0b3J5QXJuOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSByZXBvc2l0b3J5VXJpOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVwb3NpdG9yeVVyaUZvclRhZyh0YWc/OiBzdHJpbmcpOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZXBvc2l0b3J5VXJpRm9yRGlnZXN0KGRpZ2VzdD86IHN0cmluZyk6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIGFkZFRvUmVzb3VyY2VQb2xpY3koc3RhdGVtZW50OiBpYW0uUG9saWN5U3RhdGVtZW50KTogaWFtLkFkZFRvUmVzb3VyY2VQb2xpY3lSZXN1bHQ7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBncmFudChncmFudGVlOiBpYW0uSUdyYW50YWJsZSwgLi4uYWN0aW9uczogc3RyaW5nW10pOiBpYW0uR3JhbnQ7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgZ3JhbnRQdWxsKGdyYW50ZWU6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50O1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIGdyYW50UHVsbFB1c2goZ3JhbnRlZTogaWFtLklHcmFudGFibGUpOiBpYW0uR3JhbnQ7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgb25DbG91ZFRyYWlsRXZlbnQoaWQ6IHN0cmluZywgb3B0aW9ucz86IGV2ZW50cy5PbkV2ZW50T3B0aW9ucyk6IGV2ZW50cy5SdWxlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIG9uQ2xvdWRUcmFpbEltYWdlUHVzaGVkKGlkOiBzdHJpbmcsIG9wdGlvbnM/OiBPbkNsb3VkVHJhaWxJbWFnZVB1c2hlZE9wdGlvbnMpOiBldmVudHMuUnVsZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIG9uSW1hZ2VTY2FuQ29tcGxldGVkKGlkOiBzdHJpbmcsIG9wdGlvbnM/OiBPbkltYWdlU2NhbkNvbXBsZXRlZE9wdGlvbnMpOiBldmVudHMuUnVsZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBvbkV2ZW50KGlkOiBzdHJpbmcsIG9wdGlvbnM/OiBldmVudHMuT25FdmVudE9wdGlvbnMpOiBldmVudHMuUnVsZTtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgUmVwb3NpdG9yeUJhc2UgZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElSZXBvc2l0b3J5IHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IHJlcG9zaXRvcnlOYW1lOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgcmVwb3NpdG9yeUFybjogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGFic3RyYWN0IGFkZFRvUmVzb3VyY2VQb2xpY3koc3RhdGVtZW50OiBpYW0uUG9saWN5U3RhdGVtZW50KTogaWFtLkFkZFRvUmVzb3VyY2VQb2xpY3lSZXN1bHQ7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBnZXQgcmVwb3NpdG9yeVVyaSgpIHtcbiAgICByZXR1cm4gdGhpcy5yZXBvc2l0b3J5VXJpRm9yVGFnKCk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVwb3NpdG9yeVVyaUZvclRhZyh0YWc/OiBzdHJpbmcpOiBzdHJpbmcge1xuICAgIGNvbnN0IHRhZ1N1ZmZpeCA9IHRhZyA/IGA6JHt0YWd9YCA6ICcnO1xuICAgIHJldHVybiB0aGlzLnJlcG9zaXRvcnlVcmlXaXRoU3VmZml4KHRhZ1N1ZmZpeCk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVwb3NpdG9yeVVyaUZvckRpZ2VzdChkaWdlc3Q/OiBzdHJpbmcpOiBzdHJpbmcge1xuICAgIGNvbnN0IGRpZ2VzdFN1ZmZpeCA9IGRpZ2VzdCA/IGBAJHtkaWdlc3R9YCA6ICcnO1xuICAgIHJldHVybiB0aGlzLnJlcG9zaXRvcnlVcmlXaXRoU3VmZml4KGRpZ2VzdFN1ZmZpeCk7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyB0aGUgcmVwb3NpdG9yeSBVUkksIHdpdGggYW4gYXBwZW5kZWQgc3VmZml4LCBpZiBwcm92aWRlZC5cbiAgICogQHBhcmFtIHN1ZmZpeCBBbiBpbWFnZSB0YWcgb3IgYW4gaW1hZ2UgZGlnZXN0LlxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgcHJpdmF0ZSByZXBvc2l0b3J5VXJpV2l0aFN1ZmZpeChzdWZmaXg/OiBzdHJpbmcpOiBzdHJpbmcge1xuICAgIGNvbnN0IHBhcnRzID0gdGhpcy5zdGFjay5wYXJzZUFybih0aGlzLnJlcG9zaXRvcnlBcm4pO1xuICAgIHJldHVybiBgJHtwYXJ0cy5hY2NvdW50fS5ka3IuZWNyLiR7cGFydHMucmVnaW9ufS4ke3RoaXMuc3RhY2sudXJsU3VmZml4fS8ke3RoaXMucmVwb3NpdG9yeU5hbWV9JHtzdWZmaXh9YDtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBvbkNsb3VkVHJhaWxFdmVudChpZDogc3RyaW5nLCBvcHRpb25zOiBldmVudHMuT25FdmVudE9wdGlvbnMgPSB7fSk6IGV2ZW50cy5SdWxlIHtcbiAgICBjb25zdCBydWxlID0gbmV3IGV2ZW50cy5SdWxlKHRoaXMsIGlkLCBvcHRpb25zKTtcbiAgICBydWxlLmFkZFRhcmdldChvcHRpb25zLnRhcmdldCk7XG4gICAgcnVsZS5hZGRFdmVudFBhdHRlcm4oe1xuICAgICAgc291cmNlOiBbJ2F3cy5lY3InXSxcbiAgICAgIGRldGFpbFR5cGU6IFsnQVdTIEFQSSBDYWxsIHZpYSBDbG91ZFRyYWlsJ10sXG4gICAgICBkZXRhaWw6IHtcbiAgICAgICAgcmVxdWVzdFBhcmFtZXRlcnM6IHtcbiAgICAgICAgICByZXBvc2l0b3J5TmFtZTogW3RoaXMucmVwb3NpdG9yeU5hbWVdLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICB9KTtcbiAgICByZXR1cm4gcnVsZTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBvbkNsb3VkVHJhaWxJbWFnZVB1c2hlZChpZDogc3RyaW5nLCBvcHRpb25zOiBPbkNsb3VkVHJhaWxJbWFnZVB1c2hlZE9wdGlvbnMgPSB7fSk6IGV2ZW50cy5SdWxlIHtcbiAgICBjb25zdCBydWxlID0gdGhpcy5vbkNsb3VkVHJhaWxFdmVudChpZCwgb3B0aW9ucyk7XG4gICAgcnVsZS5hZGRFdmVudFBhdHRlcm4oe1xuICAgICAgZGV0YWlsOiB7XG4gICAgICAgIGV2ZW50TmFtZTogWydQdXRJbWFnZSddLFxuICAgICAgICByZXF1ZXN0UGFyYW1ldGVyczoge1xuICAgICAgICAgIGltYWdlVGFnOiBvcHRpb25zLmltYWdlVGFnID8gW29wdGlvbnMuaW1hZ2VUYWddIDogdW5kZWZpbmVkLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICB9KTtcbiAgICByZXR1cm4gcnVsZTtcbiAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIG9uSW1hZ2VTY2FuQ29tcGxldGVkKGlkOiBzdHJpbmcsIG9wdGlvbnM6IE9uSW1hZ2VTY2FuQ29tcGxldGVkT3B0aW9ucyA9IHt9KTogZXZlbnRzLlJ1bGUge1xuICAgIGNvbnN0IHJ1bGUgPSBuZXcgZXZlbnRzLlJ1bGUodGhpcywgaWQsIG9wdGlvbnMpO1xuICAgIHJ1bGUuYWRkVGFyZ2V0KG9wdGlvbnMudGFyZ2V0KTtcbiAgICBydWxlLmFkZEV2ZW50UGF0dGVybih7XG4gICAgICBzb3VyY2U6IFsnYXdzLmVjciddLFxuICAgICAgZGV0YWlsVHlwZTogWydFQ1IgSW1hZ2UgU2NhbiddLFxuICAgICAgZGV0YWlsOiB7XG4gICAgICAgICdyZXBvc2l0b3J5LW5hbWUnOiBbdGhpcy5yZXBvc2l0b3J5TmFtZV0sXG4gICAgICAgICdzY2FuLXN0YXR1cyc6IFsnQ09NUExFVEUnXSxcbiAgICAgICAgJ2ltYWdlLXRhZ3MnOiBvcHRpb25zLmltYWdlVGFncyA/PyB1bmRlZmluZWQsXG4gICAgICB9LFxuICAgIH0pO1xuICAgIHJldHVybiBydWxlO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIG9uRXZlbnQoaWQ6IHN0cmluZywgb3B0aW9uczogZXZlbnRzLk9uRXZlbnRPcHRpb25zID0ge30pIHtcbiAgICBjb25zdCBydWxlID0gbmV3IGV2ZW50cy5SdWxlKHRoaXMsIGlkLCBvcHRpb25zKTtcbiAgICBydWxlLmFkZEV2ZW50UGF0dGVybih7XG4gICAgICBzb3VyY2U6IFsnYXdzLmVjciddLFxuICAgICAgcmVzb3VyY2VzOiBbdGhpcy5yZXBvc2l0b3J5QXJuXSxcbiAgICB9KTtcbiAgICBydWxlLmFkZFRhcmdldChvcHRpb25zLnRhcmdldCk7XG4gICAgcmV0dXJuIHJ1bGU7XG4gIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgZ3JhbnQoZ3JhbnRlZTogaWFtLklHcmFudGFibGUsIC4uLmFjdGlvbnM6IHN0cmluZ1tdKSB7XG4gICAgcmV0dXJuIGlhbS5HcmFudC5hZGRUb1ByaW5jaXBhbE9yUmVzb3VyY2Uoe1xuICAgICAgZ3JhbnRlZSxcbiAgICAgIGFjdGlvbnMsXG4gICAgICByZXNvdXJjZUFybnM6IFt0aGlzLnJlcG9zaXRvcnlBcm5dLFxuICAgICAgcmVzb3VyY2VTZWxmQXJuczogW10sXG4gICAgICByZXNvdXJjZTogdGhpcyxcbiAgICB9KTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGdyYW50UHVsbChncmFudGVlOiBpYW0uSUdyYW50YWJsZSkge1xuICAgIGNvbnN0IHJldCA9IHRoaXMuZ3JhbnQoZ3JhbnRlZSwgJ2VjcjpCYXRjaENoZWNrTGF5ZXJBdmFpbGFiaWxpdHknLCAnZWNyOkdldERvd25sb2FkVXJsRm9yTGF5ZXInLCAnZWNyOkJhdGNoR2V0SW1hZ2UnKTtcblxuICAgIGlhbS5HcmFudC5hZGRUb1ByaW5jaXBhbCh7XG4gICAgICBncmFudGVlLFxuICAgICAgYWN0aW9uczogWydlY3I6R2V0QXV0aG9yaXphdGlvblRva2VuJ10sXG4gICAgICByZXNvdXJjZUFybnM6IFsnKiddLFxuICAgICAgc2NvcGU6IHRoaXMsXG4gICAgfSk7XG5cbiAgICByZXR1cm4gcmV0O1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGdyYW50UHVsbFB1c2goZ3JhbnRlZTogaWFtLklHcmFudGFibGUpIHtcbiAgICB0aGlzLmdyYW50UHVsbChncmFudGVlKTtcbiAgICByZXR1cm4gdGhpcy5ncmFudChncmFudGVlLFxuICAgICAgJ2VjcjpQdXRJbWFnZScsXG4gICAgICAnZWNyOkluaXRpYXRlTGF5ZXJVcGxvYWQnLFxuICAgICAgJ2VjcjpVcGxvYWRMYXllclBhcnQnLFxuICAgICAgJ2VjcjpDb21wbGV0ZUxheWVyVXBsb2FkJyk7XG4gIH1cbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIE9uQ2xvdWRUcmFpbEltYWdlUHVzaGVkT3B0aW9ucyBleHRlbmRzIGV2ZW50cy5PbkV2ZW50T3B0aW9ucyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGltYWdlVGFnPzogc3RyaW5nO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgT25JbWFnZVNjYW5Db21wbGV0ZWRPcHRpb25zIGV4dGVuZHMgZXZlbnRzLk9uRXZlbnRPcHRpb25zIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBpbWFnZVRhZ3M/OiBzdHJpbmdbXTtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBSZXBvc2l0b3J5UHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHJlcG9zaXRvcnlOYW1lPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBsaWZlY3ljbGVSdWxlcz86IExpZmVjeWNsZVJ1bGVbXTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbGlmZWN5Y2xlUmVnaXN0cnlJZD86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSByZW1vdmFsUG9saWN5PzogUmVtb3ZhbFBvbGljeTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgaW1hZ2VTY2FuT25QdXNoPzogYm9vbGVhbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBpbWFnZVRhZ011dGFiaWxpdHk/OiBUYWdNdXRhYmlsaXR5O1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIFJlcG9zaXRvcnlBdHRyaWJ1dGVzIHtcbiAgcmVhZG9ubHkgcmVwb3NpdG9yeU5hbWU6IHN0cmluZztcbiAgcmVhZG9ubHkgcmVwb3NpdG9yeUFybjogc3RyaW5nO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIFJlcG9zaXRvcnkgZXh0ZW5kcyBSZXBvc2l0b3J5QmFzZSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBmcm9tUmVwb3NpdG9yeUF0dHJpYnV0ZXMoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgYXR0cnM6IFJlcG9zaXRvcnlBdHRyaWJ1dGVzKTogSVJlcG9zaXRvcnkge1xuICAgIGNsYXNzIEltcG9ydCBleHRlbmRzIFJlcG9zaXRvcnlCYXNlIHtcbiAgICAgIHB1YmxpYyByZWFkb25seSByZXBvc2l0b3J5TmFtZSA9IGF0dHJzLnJlcG9zaXRvcnlOYW1lO1xuICAgICAgcHVibGljIHJlYWRvbmx5IHJlcG9zaXRvcnlBcm4gPSBhdHRycy5yZXBvc2l0b3J5QXJuO1xuXG4gICAgICBwdWJsaWMgYWRkVG9SZXNvdXJjZVBvbGljeShfc3RhdGVtZW50OiBpYW0uUG9saWN5U3RhdGVtZW50KTogaWFtLkFkZFRvUmVzb3VyY2VQb2xpY3lSZXN1bHQge1xuICAgICAgICAvLyBkcm9wcGVkXG4gICAgICAgIHJldHVybiB7IHN0YXRlbWVudEFkZGVkOiBmYWxzZSB9O1xuICAgICAgfVxuICAgIH1cblxuICAgIHJldHVybiBuZXcgSW1wb3J0KHNjb3BlLCBpZCk7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGZyb21SZXBvc2l0b3J5QXJuKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHJlcG9zaXRvcnlBcm46IHN0cmluZyk6IElSZXBvc2l0b3J5IHtcblxuICAgIC8vIGlmIHJlcG9zaXRvcnlBcm4gaXMgYSB0b2tlbiwgdGhlIHJlcG9zaXRvcnkgbmFtZSBpcyBhbHNvIHJlcXVpcmVkLiB0aGlzIGlzIGJlY2F1c2VcbiAgICAvLyByZXBvc2l0b3J5IG5hbWVzIGNhbiBpbmNsdWRlIFwiL1wiIChlLmcuIGZvby9iYXIvbXlyZXBvKSBhbmQgaXQgaXMgaW1wb3NzaWJsZSB0b1xuICAgIC8vIHBhcnNlIHRoZSBuYW1lIGZyb20gYW4gQVJOIHVzaW5nIENsb3VkRm9ybWF0aW9uJ3Mgc3BsaXQvc2VsZWN0LlxuICAgIGlmIChUb2tlbi5pc1VucmVzb2x2ZWQocmVwb3NpdG9yeUFybikpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignXCJyZXBvc2l0b3J5QXJuXCIgaXMgYSBsYXRlLWJvdW5kIHZhbHVlLCBhbmQgdGhlcmVmb3JlIFwicmVwb3NpdG9yeU5hbWVcIiBpcyByZXF1aXJlZC4gVXNlIGBmcm9tUmVwb3NpdG9yeUF0dHJpYnV0ZXNgIGluc3RlYWQnKTtcbiAgICB9XG5cbiAgICBjb25zdCByZXBvc2l0b3J5TmFtZSA9IHJlcG9zaXRvcnlBcm4uc3BsaXQoJy8nKS5zbGljZSgxKS5qb2luKCcvJyk7XG5cbiAgICBjbGFzcyBJbXBvcnQgZXh0ZW5kcyBSZXBvc2l0b3J5QmFzZSB7XG4gICAgICBwdWJsaWMgcmVwb3NpdG9yeU5hbWUgPSByZXBvc2l0b3J5TmFtZTtcbiAgICAgIHB1YmxpYyByZXBvc2l0b3J5QXJuID0gcmVwb3NpdG9yeUFybjtcblxuICAgICAgcHVibGljIGFkZFRvUmVzb3VyY2VQb2xpY3koX3N0YXRlbWVudDogaWFtLlBvbGljeVN0YXRlbWVudCk6IGlhbS5BZGRUb1Jlc291cmNlUG9saWN5UmVzdWx0IHtcbiAgICAgICAgLy8gZHJvcHBlZFxuICAgICAgICByZXR1cm4geyBzdGF0ZW1lbnRBZGRlZDogZmFsc2UgfTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICByZXR1cm4gbmV3IEltcG9ydChzY29wZSwgaWQpO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBmcm9tUmVwb3NpdG9yeU5hbWUoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcmVwb3NpdG9yeU5hbWU6IHN0cmluZyk6IElSZXBvc2l0b3J5IHtcbiAgICBjbGFzcyBJbXBvcnQgZXh0ZW5kcyBSZXBvc2l0b3J5QmFzZSB7XG4gICAgICBwdWJsaWMgcmVwb3NpdG9yeU5hbWUgPSByZXBvc2l0b3J5TmFtZTtcbiAgICAgIHB1YmxpYyByZXBvc2l0b3J5QXJuID0gUmVwb3NpdG9yeS5hcm5Gb3JMb2NhbFJlcG9zaXRvcnkocmVwb3NpdG9yeU5hbWUsIHNjb3BlKTtcblxuICAgICAgcHVibGljIGFkZFRvUmVzb3VyY2VQb2xpY3koX3N0YXRlbWVudDogaWFtLlBvbGljeVN0YXRlbWVudCk6IGlhbS5BZGRUb1Jlc291cmNlUG9saWN5UmVzdWx0IHtcbiAgICAgICAgLy8gZHJvcHBlZFxuICAgICAgICByZXR1cm4geyBzdGF0ZW1lbnRBZGRlZDogZmFsc2UgfTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICByZXR1cm4gbmV3IEltcG9ydChzY29wZSwgaWQpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgYXJuRm9yTG9jYWxSZXBvc2l0b3J5KHJlcG9zaXRvcnlOYW1lOiBzdHJpbmcsIHNjb3BlOiBJQ29uc3RydWN0LCBhY2NvdW50Pzogc3RyaW5nKTogc3RyaW5nIHtcbiAgICByZXR1cm4gU3RhY2sub2Yoc2NvcGUpLmZvcm1hdEFybih7XG4gICAgICBhY2NvdW50LFxuICAgICAgc2VydmljZTogJ2VjcicsXG4gICAgICByZXNvdXJjZTogJ3JlcG9zaXRvcnknLFxuICAgICAgcmVzb3VyY2VOYW1lOiByZXBvc2l0b3J5TmFtZSxcbiAgICB9KTtcbiAgfVxuXG5cbiAgcHJpdmF0ZSBzdGF0aWMgdmFsaWRhdGVSZXBvc2l0b3J5TmFtZShwaHlzaWNhbE5hbWU6IHN0cmluZykge1xuICAgIGNvbnN0IHJlcG9zaXRvcnlOYW1lID0gcGh5c2ljYWxOYW1lO1xuICAgIGlmICghcmVwb3NpdG9yeU5hbWUgfHwgVG9rZW4uaXNVbnJlc29sdmVkKHJlcG9zaXRvcnlOYW1lKSkge1xuICAgICAgLy8gdGhlIG5hbWUgaXMgYSBsYXRlLWJvdW5kIHZhbHVlLCBub3QgYSBkZWZpbmVkIHN0cmluZyxcbiAgICAgIC8vIHNvIHNraXAgdmFsaWRhdGlvblxuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIGNvbnN0IGVycm9yczogc3RyaW5nW10gPSBbXTtcblxuICAgIC8vIFJ1bGVzIGNvZGlmaWVkIGZyb20gaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXJlc291cmNlLWVjci1yZXBvc2l0b3J5Lmh0bWxcbiAgICBpZiAocmVwb3NpdG9yeU5hbWUubGVuZ3RoIDwgMiB8fCByZXBvc2l0b3J5TmFtZS5sZW5ndGggPiAyNTYpIHtcbiAgICAgIGVycm9ycy5wdXNoKCdSZXBvc2l0b3J5IG5hbWUgbXVzdCBiZSBhdCBsZWFzdCAyIGFuZCBubyBtb3JlIHRoYW4gMjU2IGNoYXJhY3RlcnMnKTtcbiAgICB9XG4gICAgY29uc3QgaXNQYXR0ZXJuTWF0Y2ggPSAvXig/OlthLXowLTldKyg/OlsuXy1dW2EtejAtOV0rKSpcXC8pKlthLXowLTldKyg/OlsuXy1dW2EtejAtOV0rKSokLy50ZXN0KHJlcG9zaXRvcnlOYW1lKTtcbiAgICBpZiAoIWlzUGF0dGVybk1hdGNoKSB7XG4gICAgICBlcnJvcnMucHVzaCgnUmVwb3NpdG9yeSBuYW1lIG11c3QgZm9sbG93IHRoZSBzcGVjaWZpZWQgcGF0dGVybjogKD86W2EtejAtOV0rKD86Wy5fLV1bYS16MC05XSspKi8pKlthLXowLTldKyg/OlsuXy1dW2EtejAtOV0rKSonKTtcbiAgICB9XG5cbiAgICBpZiAoZXJyb3JzLmxlbmd0aCA+IDApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgSW52YWxpZCBFQ1IgcmVwb3NpdG9yeSBuYW1lICh2YWx1ZTogJHtyZXBvc2l0b3J5TmFtZX0pJHtFT0x9JHtlcnJvcnMuam9pbihFT0wpfWApO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyByZWFkb25seSByZXBvc2l0b3J5TmFtZTogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgcmVwb3NpdG9yeUFybjogc3RyaW5nO1xuICBwcml2YXRlIHJlYWRvbmx5IGxpZmVjeWNsZVJ1bGVzID0gbmV3IEFycmF5PExpZmVjeWNsZVJ1bGU+KCk7XG4gIHByaXZhdGUgcmVhZG9ubHkgcmVnaXN0cnlJZD86IHN0cmluZztcbiAgcHJpdmF0ZSBwb2xpY3lEb2N1bWVudD86IGlhbS5Qb2xpY3lEb2N1bWVudDtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogUmVwb3NpdG9yeVByb3BzID0ge30pIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgIHBoeXNpY2FsTmFtZTogcHJvcHMucmVwb3NpdG9yeU5hbWUsXG4gICAgfSk7XG5cbiAgICBSZXBvc2l0b3J5LnZhbGlkYXRlUmVwb3NpdG9yeU5hbWUodGhpcy5waHlzaWNhbE5hbWUpO1xuXG4gICAgY29uc3QgcmVzb3VyY2UgPSBuZXcgQ2ZuUmVwb3NpdG9yeSh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICByZXBvc2l0b3J5TmFtZTogdGhpcy5waHlzaWNhbE5hbWUsXG4gICAgICAvLyBJdCBzYXlzIFwiVGV4dFwiLCBidXQgdGhleSBhY3R1YWxseSBtZWFuIFwiT2JqZWN0XCIuXG4gICAgICByZXBvc2l0b3J5UG9saWN5VGV4dDogTGF6eS5hbnkoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLnBvbGljeURvY3VtZW50IH0pLFxuICAgICAgbGlmZWN5Y2xlUG9saWN5OiBMYXp5LmFueSh7IHByb2R1Y2U6ICgpID0+IHRoaXMucmVuZGVyTGlmZWN5Y2xlUG9saWN5KCkgfSksXG4gICAgICBpbWFnZVNjYW5uaW5nQ29uZmlndXJhdGlvbjogIXByb3BzLmltYWdlU2Nhbk9uUHVzaCA/IHVuZGVmaW5lZCA6IHtcbiAgICAgICAgc2Nhbk9uUHVzaDogdHJ1ZSxcbiAgICAgIH0sXG4gICAgICBpbWFnZVRhZ011dGFiaWxpdHk6IHByb3BzLmltYWdlVGFnTXV0YWJpbGl0eSB8fCB1bmRlZmluZWQsXG4gICAgfSk7XG5cbiAgICByZXNvdXJjZS5hcHBseVJlbW92YWxQb2xpY3kocHJvcHMucmVtb3ZhbFBvbGljeSk7XG5cbiAgICB0aGlzLnJlZ2lzdHJ5SWQgPSBwcm9wcy5saWZlY3ljbGVSZWdpc3RyeUlkO1xuICAgIGlmIChwcm9wcy5saWZlY3ljbGVSdWxlcykge1xuICAgICAgcHJvcHMubGlmZWN5Y2xlUnVsZXMuZm9yRWFjaCh0aGlzLmFkZExpZmVjeWNsZVJ1bGUuYmluZCh0aGlzKSk7XG4gICAgfVxuXG4gICAgdGhpcy5yZXBvc2l0b3J5TmFtZSA9IHRoaXMuZ2V0UmVzb3VyY2VOYW1lQXR0cmlidXRlKHJlc291cmNlLnJlZik7XG4gICAgdGhpcy5yZXBvc2l0b3J5QXJuID0gdGhpcy5nZXRSZXNvdXJjZUFybkF0dHJpYnV0ZShyZXNvdXJjZS5hdHRyQXJuLCB7XG4gICAgICBzZXJ2aWNlOiAnZWNyJyxcbiAgICAgIHJlc291cmNlOiAncmVwb3NpdG9yeScsXG4gICAgICByZXNvdXJjZU5hbWU6IHRoaXMucGh5c2ljYWxOYW1lLFxuICAgIH0pO1xuICB9XG5cbiAgcHVibGljIGFkZFRvUmVzb3VyY2VQb2xpY3koc3RhdGVtZW50OiBpYW0uUG9saWN5U3RhdGVtZW50KTogaWFtLkFkZFRvUmVzb3VyY2VQb2xpY3lSZXN1bHQge1xuICAgIGlmICh0aGlzLnBvbGljeURvY3VtZW50ID09PSB1bmRlZmluZWQpIHtcbiAgICAgIHRoaXMucG9saWN5RG9jdW1lbnQgPSBuZXcgaWFtLlBvbGljeURvY3VtZW50KCk7XG4gICAgfVxuICAgIHRoaXMucG9saWN5RG9jdW1lbnQuYWRkU3RhdGVtZW50cyhzdGF0ZW1lbnQpO1xuICAgIHJldHVybiB7IHN0YXRlbWVudEFkZGVkOiBmYWxzZSwgcG9saWN5RGVwZW5kYWJsZTogdGhpcy5wb2xpY3lEb2N1bWVudCB9O1xuICB9XG5cbiAgcHJvdGVjdGVkIHZhbGlkYXRlKCk6IHN0cmluZ1tdIHtcbiAgICBjb25zdCBlcnJvcnMgPSBzdXBlci52YWxpZGF0ZSgpO1xuICAgIGVycm9ycy5wdXNoKC4uLnRoaXMucG9saWN5RG9jdW1lbnQ/LnZhbGlkYXRlRm9yUmVzb3VyY2VQb2xpY3koKSB8fCBbXSk7XG4gICAgcmV0dXJuIGVycm9ycztcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWRkTGlmZWN5Y2xlUnVsZShydWxlOiBMaWZlY3ljbGVSdWxlKSB7XG4gICAgLy8gVmFsaWRhdGUgcnVsZSBoZXJlIHNvIHVzZXJzIGdldCBlcnJvcnMgYXQgdGhlIGV4cGVjdGVkIGxvY2F0aW9uXG4gICAgaWYgKHJ1bGUudGFnU3RhdHVzID09PSB1bmRlZmluZWQpIHtcbiAgICAgIHJ1bGUgPSB7IC4uLnJ1bGUsIHRhZ1N0YXR1czogcnVsZS50YWdQcmVmaXhMaXN0ID09PSB1bmRlZmluZWQgPyBUYWdTdGF0dXMuQU5ZIDogVGFnU3RhdHVzLlRBR0dFRCB9O1xuICAgIH1cblxuICAgIGlmIChydWxlLnRhZ1N0YXR1cyA9PT0gVGFnU3RhdHVzLlRBR0dFRCAmJiAocnVsZS50YWdQcmVmaXhMaXN0ID09PSB1bmRlZmluZWQgfHwgcnVsZS50YWdQcmVmaXhMaXN0Lmxlbmd0aCA9PT0gMCkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignVGFnU3RhdHVzLlRhZ2dlZCByZXF1aXJlcyB0aGUgc3BlY2lmaWNhdGlvbiBvZiBhIHRhZ1ByZWZpeExpc3QnKTtcbiAgICB9XG4gICAgaWYgKHJ1bGUudGFnU3RhdHVzICE9PSBUYWdTdGF0dXMuVEFHR0VEICYmIHJ1bGUudGFnUHJlZml4TGlzdCAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ3RhZ1ByZWZpeExpc3QgY2FuIG9ubHkgYmUgc3BlY2lmaWVkIHdoZW4gdGFnU3RhdHVzIGlzIHNldCB0byBUYWdnZWQnKTtcbiAgICB9XG4gICAgaWYgKChydWxlLm1heEltYWdlQWdlICE9PSB1bmRlZmluZWQpID09PSAocnVsZS5tYXhJbWFnZUNvdW50ICE9PSB1bmRlZmluZWQpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYExpZmUgY3ljbGUgcnVsZSBtdXN0IGNvbnRhaW4gZXhhY3RseSBvbmUgb2YgJ21heEltYWdlQWdlJyBhbmQgJ21heEltYWdlQ291bnQnLCBnb3Q6ICR7SlNPTi5zdHJpbmdpZnkocnVsZSl9YCk7XG4gICAgfVxuXG4gICAgaWYgKHJ1bGUudGFnU3RhdHVzID09PSBUYWdTdGF0dXMuQU5ZICYmIHRoaXMubGlmZWN5Y2xlUnVsZXMuZmlsdGVyKHIgPT4gci50YWdTdGF0dXMgPT09IFRhZ1N0YXR1cy5BTlkpLmxlbmd0aCA+IDApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignTGlmZSBjeWNsZSBjYW4gb25seSBoYXZlIG9uZSBUYWdTdGF0dXMuQW55IHJ1bGUnKTtcbiAgICB9XG5cbiAgICB0aGlzLmxpZmVjeWNsZVJ1bGVzLnB1c2goeyAuLi5ydWxlIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlbmRlciB0aGUgbGlmZSBjeWNsZSBwb2xpY3kgb2JqZWN0XG4gICAqL1xuICBwcml2YXRlIHJlbmRlckxpZmVjeWNsZVBvbGljeSgpOiBDZm5SZXBvc2l0b3J5LkxpZmVjeWNsZVBvbGljeVByb3BlcnR5IHwgdW5kZWZpbmVkIHtcbiAgICBjb25zdCBzdGFjayA9IFN0YWNrLm9mKHRoaXMpO1xuICAgIGxldCBsaWZlY3ljbGVQb2xpY3lUZXh0OiBhbnk7XG5cbiAgICBpZiAodGhpcy5saWZlY3ljbGVSdWxlcy5sZW5ndGggPT09IDAgJiYgIXRoaXMucmVnaXN0cnlJZCkgeyByZXR1cm4gdW5kZWZpbmVkOyB9XG5cbiAgICBpZiAodGhpcy5saWZlY3ljbGVSdWxlcy5sZW5ndGggPiAwKSB7XG4gICAgICBsaWZlY3ljbGVQb2xpY3lUZXh0ID0gSlNPTi5zdHJpbmdpZnkoc3RhY2sucmVzb2x2ZSh7XG4gICAgICAgIHJ1bGVzOiB0aGlzLm9yZGVyZWRMaWZlY3ljbGVSdWxlcygpLm1hcChyZW5kZXJMaWZlY3ljbGVSdWxlKSxcbiAgICAgIH0pKTtcbiAgICB9XG5cbiAgICByZXR1cm4ge1xuICAgICAgbGlmZWN5Y2xlUG9saWN5VGV4dCxcbiAgICAgIHJlZ2lzdHJ5SWQ6IHRoaXMucmVnaXN0cnlJZCxcbiAgICB9O1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybiBsaWZlIGN5Y2xlIHJ1bGVzIHdpdGggYXV0b21hdGljIG9yZGVyaW5nIGFwcGxpZWQuXG4gICAqXG4gICAqIEFsc28gYXBwbGllcyB2YWxpZGF0aW9uIG9mIHRoZSAnYW55JyBydWxlLlxuICAgKi9cbiAgcHJpdmF0ZSBvcmRlcmVkTGlmZWN5Y2xlUnVsZXMoKTogTGlmZWN5Y2xlUnVsZVtdIHtcbiAgICBpZiAodGhpcy5saWZlY3ljbGVSdWxlcy5sZW5ndGggPT09IDApIHsgcmV0dXJuIFtdOyB9XG5cbiAgICBjb25zdCBwcmlvcml0aXplZFJ1bGVzID0gdGhpcy5saWZlY3ljbGVSdWxlcy5maWx0ZXIociA9PiByLnJ1bGVQcmlvcml0eSAhPT0gdW5kZWZpbmVkICYmIHIudGFnU3RhdHVzICE9PSBUYWdTdGF0dXMuQU5ZKTtcbiAgICBjb25zdCBhdXRvUHJpb3JpdGl6ZWRSdWxlcyA9IHRoaXMubGlmZWN5Y2xlUnVsZXMuZmlsdGVyKHIgPT4gci5ydWxlUHJpb3JpdHkgPT09IHVuZGVmaW5lZCAmJiByLnRhZ1N0YXR1cyAhPT0gVGFnU3RhdHVzLkFOWSk7XG4gICAgY29uc3QgYW55UnVsZXMgPSB0aGlzLmxpZmVjeWNsZVJ1bGVzLmZpbHRlcihyID0+IHIudGFnU3RhdHVzID09PSBUYWdTdGF0dXMuQU5ZKTtcbiAgICBpZiAoYW55UnVsZXMubGVuZ3RoID4gMCAmJiBhbnlSdWxlc1swXS5ydWxlUHJpb3JpdHkgIT09IHVuZGVmaW5lZCAmJiBhdXRvUHJpb3JpdGl6ZWRSdWxlcy5sZW5ndGggPiAwKSB7XG4gICAgICAvLyBTdXBwb3J0aW5nIHRoaXMgaXMgdG9vIGNvbXBsZXggZm9yIHZlcnkgbGl0dGxlIHZhbHVlLiBXZSBqdXN0IHByb2hpYml0IGl0LlxuICAgICAgdGhyb3cgbmV3IEVycm9yKFwiQ2Fubm90IGNvbWJpbmUgcHJpb3JpdGl6ZWQgVGFnU3RhdHVzLkFueSBydWxlIHdpdGggdW5wcmlvcml0aXplZCBydWxlcy4gUmVtb3ZlIHJ1bGVQcmlvcml0eSBmcm9tIHRoZSAnQW55JyBydWxlLlwiKTtcbiAgICB9XG5cbiAgICBjb25zdCBwcmlvcyA9IHByaW9yaXRpemVkUnVsZXMubWFwKHIgPT4gci5ydWxlUHJpb3JpdHkhKTtcbiAgICBsZXQgYXV0b1ByaW8gPSAocHJpb3MubGVuZ3RoID4gMCA/IE1hdGgubWF4KC4uLnByaW9zKSA6IDApICsgMTtcblxuICAgIGNvbnN0IHJldCA9IG5ldyBBcnJheTxMaWZlY3ljbGVSdWxlPigpO1xuICAgIGZvciAoY29uc3QgcnVsZSBvZiBwcmlvcml0aXplZFJ1bGVzLmNvbmNhdChhdXRvUHJpb3JpdGl6ZWRSdWxlcykuY29uY2F0KGFueVJ1bGVzKSkge1xuICAgICAgcmV0LnB1c2goe1xuICAgICAgICAuLi5ydWxlLFxuICAgICAgICBydWxlUHJpb3JpdHk6IHJ1bGUucnVsZVByaW9yaXR5ID8/IGF1dG9QcmlvKyssXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICAvLyBEbyB2YWxpZGF0aW9uIG9uIHRoZSBmaW5hbCBhcnJheS0tbWlnaHQgc3RpbGwgYmUgd3JvbmcgYmVjYXVzZSB0aGUgdXNlciBzdXBwbGllZCBhbGwgcHJpb3MsIGJ1dCBpbmNvcnJlY3RseS5cbiAgICB2YWxpZGF0ZUFueVJ1bGVMYXN0KHJldCk7XG4gICAgcmV0dXJuIHJldDtcbiAgfVxufVxuXG5mdW5jdGlvbiB2YWxpZGF0ZUFueVJ1bGVMYXN0KHJ1bGVzOiBMaWZlY3ljbGVSdWxlW10pIHtcbiAgY29uc3QgYW55UnVsZXMgPSBydWxlcy5maWx0ZXIociA9PiByLnRhZ1N0YXR1cyA9PT0gVGFnU3RhdHVzLkFOWSk7XG4gIGlmIChhbnlSdWxlcy5sZW5ndGggPT09IDEpIHtcbiAgICBjb25zdCBtYXhQcmlvID0gTWF0aC5tYXgoLi4ucnVsZXMubWFwKHIgPT4gci5ydWxlUHJpb3JpdHkhKSk7XG4gICAgaWYgKGFueVJ1bGVzWzBdLnJ1bGVQcmlvcml0eSAhPT0gbWF4UHJpbykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBUYWdTdGF0dXMuQW55IHJ1bGUgbXVzdCBoYXZlIGhpZ2hlc3QgcHJpb3JpdHksIGhhcyAke2FueVJ1bGVzWzBdLnJ1bGVQcmlvcml0eX0gd2hpY2ggaXMgc21hbGxlciB0aGFuICR7bWF4UHJpb31gKTtcbiAgICB9XG4gIH1cbn1cblxuLyoqXG4gKiBSZW5kZXIgdGhlIGxpZmVjeWNsZSBydWxlIHRvIEpTT05cbiAqL1xuZnVuY3Rpb24gcmVuZGVyTGlmZWN5Y2xlUnVsZShydWxlOiBMaWZlY3ljbGVSdWxlKSB7XG4gIHJldHVybiB7XG4gICAgcnVsZVByaW9yaXR5OiBydWxlLnJ1bGVQcmlvcml0eSxcbiAgICBkZXNjcmlwdGlvbjogcnVsZS5kZXNjcmlwdGlvbixcbiAgICBzZWxlY3Rpb246IHtcbiAgICAgIHRhZ1N0YXR1czogcnVsZS50YWdTdGF0dXMgfHwgVGFnU3RhdHVzLkFOWSxcbiAgICAgIHRhZ1ByZWZpeExpc3Q6IHJ1bGUudGFnUHJlZml4TGlzdCxcbiAgICAgIGNvdW50VHlwZTogcnVsZS5tYXhJbWFnZUFnZSAhPT0gdW5kZWZpbmVkID8gQ291bnRUeXBlLlNJTkNFX0lNQUdFX1BVU0hFRCA6IENvdW50VHlwZS5JTUFHRV9DT1VOVF9NT1JFX1RIQU4sXG4gICAgICBjb3VudE51bWJlcjogcnVsZS5tYXhJbWFnZUFnZT8udG9EYXlzKCkgPz8gcnVsZS5tYXhJbWFnZUNvdW50LFxuICAgICAgY291bnRVbml0OiBydWxlLm1heEltYWdlQWdlICE9PSB1bmRlZmluZWQgPyAnZGF5cycgOiB1bmRlZmluZWQsXG4gICAgfSxcbiAgICBhY3Rpb246IHtcbiAgICAgIHR5cGU6ICdleHBpcmUnLFxuICAgIH0sXG4gIH07XG59XG5cbi8qKlxuICogU2VsZWN0IGltYWdlcyBiYXNlZCBvbiBjb3VudHNcbiAqL1xuY29uc3QgZW51bSBDb3VudFR5cGUge1xuICAvKipcbiAgICogU2V0IGEgbGltaXQgb24gdGhlIG51bWJlciBvZiBpbWFnZXMgaW4geW91ciByZXBvc2l0b3J5XG4gICAqL1xuICBJTUFHRV9DT1VOVF9NT1JFX1RIQU4gPSAnaW1hZ2VDb3VudE1vcmVUaGFuJyxcblxuICAvKipcbiAgICogU2V0IGFuIGFnZSBsaW1pdCBvbiB0aGUgaW1hZ2VzIGluIHlvdXIgcmVwb3NpdG9yeVxuICAgKi9cbiAgU0lOQ0VfSU1BR0VfUFVTSEVEID0gJ3NpbmNlSW1hZ2VQdXNoZWQnLFxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgZW51bSBUYWdNdXRhYmlsaXR5IHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgTVVUQUJMRSA9ICdNVVRBQkxFJyxcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgSU1NVVRBQkxFID0gJ0lNTVVUQUJMRScsXG5cbn1cbiJdfQ==