"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InputValidator = void 0;
const cdk = require("aws-cdk-lib");
/**
 * Class to validate that inputs match requirements.
 */
class InputValidator {
    /**
     * Validates length is between allowed min and max lengths.
     */
    static validateLength(resourceName, inputName, minLength, maxLength, inputString) {
        if (!cdk.Token.isUnresolved(inputString) && inputString !== undefined && (inputString.length < minLength || inputString.length > maxLength)) {
            throw new Error(`Invalid ${inputName} for resource ${resourceName}, must have length between ${minLength} and ${maxLength}, got: '${this.truncateString(inputString, 100)}'`);
        }
    }
    /**
     * Validates string matches the allowed regex pattern.
     */
    static validateRegex(resourceName, inputName, regexp, inputString) {
        if (!cdk.Token.isUnresolved(inputString) && inputString !== undefined && !regexp.test(inputString)) {
            throw new Error(`Invalid ${inputName} for resource ${resourceName}, must match regex pattern ${regexp}, got: '${this.truncateString(inputString, 100)}'`);
        }
    }
    /**
     * Validates string matches the valid URL regex pattern.
     */
    static validateUrl(resourceName, inputName, inputString) {
        this.validateRegex(resourceName, inputName, /^https?:\/\/.*/, inputString);
    }
    /**
    * Validates string matches the valid email regex pattern.
    */
    static validateEmail(resourceName, inputName, inputString) {
        this.validateRegex(resourceName, inputName, /^[\w\d.%+\-]+@[a-z\d.\-]+\.[a-z]{2,4}$/i, inputString);
    }
    /**
    * Validates that a role being used as a local launch role has the role name set
    */
    static validateRoleNameSetForLocalLaunchRole(role) {
        if (role.node.defaultChild) {
            if (cdk.Token.isUnresolved(role.node.defaultChild.roleName)) {
                throw new Error(`Role ${role.node.id} used for Local Launch Role must have roleName explicitly set`);
            }
        }
    }
    static truncateString(string, maxLength) {
        if (string.length > maxLength) {
            return string.substring(0, maxLength) + '[truncated]';
        }
        return string;
    }
}
exports.InputValidator = InputValidator;
//# sourceMappingURL=data:application/json;base64,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