"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SageMakerCreateTransformJob = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
const base_types_1 = require("./base-types");
const utils_1 = require("./private/utils");
/**
 * Class representing the SageMaker Create Transform Job task.
 *
 */
class SageMakerCreateTransformJob extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_SageMakerCreateTransformJobProps(props);
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, SageMakerCreateTransformJob.SUPPORTED_INTEGRATION_PATTERNS);
        // set the sagemaker role or create new one
        if (props.role) {
            this._role = props.role;
        }
        // set the S3 Data type of the input data config objects to be 'S3Prefix' if not defined
        this.transformInput = props.transformInput.transformDataSource.s3DataSource.s3DataType
            ? props.transformInput
            : Object.assign({}, props.transformInput, {
                transformDataSource: { s3DataSource: { ...props.transformInput.transformDataSource.s3DataSource, s3DataType: base_types_1.S3DataType.S3_PREFIX } },
            });
        // set the default value for the transform resources
        this.transformResources = props.transformResources || {
            instanceCount: 1,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.M4, ec2.InstanceSize.XLARGE),
        };
        this.taskPolicies = this.makePolicyStatements();
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('sagemaker', 'createTransformJob', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject(this.renderParameters()),
        };
    }
    /**
     * The execution role for the Sagemaker transform job.
     *
     * Only available after task has been added to a state machine.
     */
    get role() {
        if (this._role === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._role;
    }
    renderParameters() {
        return {
            ...(this.props.batchStrategy ? { BatchStrategy: this.props.batchStrategy } : {}),
            ...utils_1.renderEnvironment(this.props.environment),
            ...(this.props.maxConcurrentTransforms ? { MaxConcurrentTransforms: this.props.maxConcurrentTransforms } : {}),
            ...(this.props.maxPayload ? { MaxPayloadInMB: this.props.maxPayload.toMebibytes() } : {}),
            ...this.props.modelClientOptions ? this.renderModelClientOptions(this.props.modelClientOptions) : {},
            ModelName: this.props.modelName,
            ...utils_1.renderTags(this.props.tags),
            ...this.renderTransformInput(this.transformInput),
            TransformJobName: this.props.transformJobName,
            ...this.renderTransformOutput(this.props.transformOutput),
            ...this.renderTransformResources(this.transformResources),
        };
    }
    renderModelClientOptions(options) {
        var _b;
        const retries = options.invocationsMaxRetries;
        if (!core_1.Token.isUnresolved(retries) && retries ? (retries < 0 || retries > 3) : false) {
            throw new Error(`invocationsMaxRetries should be between 0 and 3. Received: ${retries}.`);
        }
        const timeout = (_b = options.invocationsTimeout) === null || _b === void 0 ? void 0 : _b.toSeconds();
        if (!core_1.Token.isUnresolved(timeout) && timeout ? (timeout < 1 || timeout > 3600) : false) {
            throw new Error(`invocationsTimeout should be between 1 and 3600 seconds. Received: ${timeout}.`);
        }
        return {
            ModelClientConfig: {
                InvocationsMaxRetries: retries !== null && retries !== void 0 ? retries : 0,
                InvocationsTimeoutInSeconds: timeout !== null && timeout !== void 0 ? timeout : 60,
            },
        };
    }
    renderTransformInput(input) {
        return {
            TransformInput: {
                ...(input.compressionType ? { CompressionType: input.compressionType } : {}),
                ...(input.contentType ? { ContentType: input.contentType } : {}),
                DataSource: {
                    S3DataSource: {
                        S3Uri: input.transformDataSource.s3DataSource.s3Uri,
                        S3DataType: input.transformDataSource.s3DataSource.s3DataType,
                    },
                },
                ...(input.splitType ? { SplitType: input.splitType } : {}),
            },
        };
    }
    renderTransformOutput(output) {
        return {
            TransformOutput: {
                S3OutputPath: output.s3OutputPath,
                ...(output.encryptionKey ? { KmsKeyId: output.encryptionKey.keyArn } : {}),
                ...(output.accept ? { Accept: output.accept } : {}),
                ...(output.assembleWith ? { AssembleWith: output.assembleWith } : {}),
            },
        };
    }
    renderTransformResources(resources) {
        return {
            TransformResources: {
                InstanceCount: resources.instanceCount,
                InstanceType: sfn.JsonPath.isEncodedJsonPath(resources.instanceType.toString())
                    ? resources.instanceType.toString() : `ml.${resources.instanceType}`,
                ...(resources.volumeEncryptionKey ? { VolumeKmsKeyId: resources.volumeEncryptionKey.keyArn } : {}),
            },
        };
    }
    makePolicyStatements() {
        const stack = core_1.Stack.of(this);
        // create new role if doesn't exist
        if (this._role === undefined) {
            this._role = new iam.Role(this, 'SagemakerTransformRole', {
                assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
                managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSageMakerFullAccess')],
            });
        }
        // https://docs.aws.amazon.com/step-functions/latest/dg/sagemaker-iam.html
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['sagemaker:CreateTransformJob', 'sagemaker:DescribeTransformJob', 'sagemaker:StopTransformJob'],
                resources: [
                    stack.formatArn({
                        service: 'sagemaker',
                        resource: 'transform-job',
                        resourceName: '*',
                    }),
                ],
            }),
            new iam.PolicyStatement({
                actions: ['sagemaker:ListTags'],
                resources: ['*'],
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this.role.roleArn],
                conditions: {
                    StringEquals: { 'iam:PassedToService': 'sagemaker.amazonaws.com' },
                },
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForSageMakerTransformJobsRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.SageMakerCreateTransformJob = SageMakerCreateTransformJob;
_a = JSII_RTTI_SYMBOL_1;
SageMakerCreateTransformJob[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.SageMakerCreateTransformJob", version: "1.146.0" };
SageMakerCreateTransformJob.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,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