"use strict";
var _a, _b, _c, _d, _e;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssembleWith = exports.SplitType = exports.BatchStrategy = exports.AcceleratorType = exports.AcceleratorClass = exports.Mode = exports.ContainerDefinition = exports.CompressionType = exports.InputMode = exports.RecordWrapperType = exports.S3DataDistributionType = exports.S3DataType = exports.DockerImage = exports.S3Location = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ecr_assets_1 = require("@aws-cdk/aws-ecr-assets");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
/**
 * Constructs `IS3Location` objects.
 *
 */
class S3Location {
    /**
     * An `IS3Location` built with a determined bucket and key prefix.
     *
     * @param bucket    is the bucket where the objects are to be stored.
     * @param keyPrefix is the key prefix used by the location.
     */
    static fromBucket(bucket, keyPrefix) {
        return new StandardS3Location({ bucket, keyPrefix, uri: bucket.urlForObject(keyPrefix) });
    }
    /**
     * An `IS3Location` determined fully by a JSON Path from the task input.
     *
     * Due to the dynamic nature of those locations, the IAM grants that will be set by `grantRead` and `grantWrite`
     * apply to the `*` resource.
     *
     * @param expression the JSON expression resolving to an S3 location URI.
     */
    static fromJsonExpression(expression) {
        return new StandardS3Location({ uri: sfn.JsonPath.stringAt(expression) });
    }
}
exports.S3Location = S3Location;
_a = JSII_RTTI_SYMBOL_1;
S3Location[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.S3Location", version: "1.146.0" };
/**
 * Creates `IDockerImage` instances.
 *
 */
class DockerImage {
    /**
     * Reference a Docker image stored in an ECR repository.
     *
     * @param repository the ECR repository where the image is hosted.
     * @param tag an optional `tag`
     */
    static fromEcrRepository(repository, tag = 'latest') {
        return new StandardDockerImage({ repository, imageUri: repository.repositoryUriForTag(tag) });
    }
    /**
     * Reference a Docker image which URI is obtained from the task's input.
     *
     * @param expression           the JSON path expression with the task input.
     * @param allowAnyEcrImagePull whether ECR access should be permitted (set to `false` if the image will never be in ECR).
     */
    static fromJsonExpression(expression, allowAnyEcrImagePull = true) {
        return new StandardDockerImage({ imageUri: expression, allowAnyEcrImagePull });
    }
    /**
     * Reference a Docker image by it's URI.
     *
     * When referencing ECR images, prefer using `inEcr`.
     *
     * @param imageUri the URI to the docker image.
     */
    static fromRegistry(imageUri) {
        return new StandardDockerImage({ imageUri });
    }
    /**
     * Reference a Docker image that is provided as an Asset in the current app.
     *
     * @param scope the scope in which to create the Asset.
     * @param id    the ID for the asset in the construct tree.
     * @param props the configuration props of the asset.
     */
    static fromAsset(scope, id, props) {
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new StandardDockerImage({ repository: asset.repository, imageUri: asset.imageUri });
    }
}
exports.DockerImage = DockerImage;
_b = JSII_RTTI_SYMBOL_1;
DockerImage[_b] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.DockerImage", version: "1.146.0" };
/**
 * S3 Data Type.
 *
 */
var S3DataType;
(function (S3DataType) {
    /**
     * Manifest File Data Type
     */
    S3DataType["MANIFEST_FILE"] = "ManifestFile";
    /**
     * S3 Prefix Data Type
     */
    S3DataType["S3_PREFIX"] = "S3Prefix";
    /**
     * Augmented Manifest File Data Type
     */
    S3DataType["AUGMENTED_MANIFEST_FILE"] = "AugmentedManifestFile";
})(S3DataType = exports.S3DataType || (exports.S3DataType = {}));
/**
 * S3 Data Distribution Type.
 *
 */
var S3DataDistributionType;
(function (S3DataDistributionType) {
    /**
     * Fully replicated S3 Data Distribution Type
     */
    S3DataDistributionType["FULLY_REPLICATED"] = "FullyReplicated";
    /**
     * Sharded By S3 Key Data Distribution Type
     */
    S3DataDistributionType["SHARDED_BY_S3_KEY"] = "ShardedByS3Key";
})(S3DataDistributionType = exports.S3DataDistributionType || (exports.S3DataDistributionType = {}));
/**
 * Define the format of the input data.
 *
 */
var RecordWrapperType;
(function (RecordWrapperType) {
    /**
     * None record wrapper type
     */
    RecordWrapperType["NONE"] = "None";
    /**
     * RecordIO record wrapper type
     */
    RecordWrapperType["RECORD_IO"] = "RecordIO";
})(RecordWrapperType = exports.RecordWrapperType || (exports.RecordWrapperType = {}));
/**
 *  Input mode that the algorithm supports.
 *
 */
var InputMode;
(function (InputMode) {
    /**
     * Pipe mode
     */
    InputMode["PIPE"] = "Pipe";
    /**
     * File mode.
     */
    InputMode["FILE"] = "File";
})(InputMode = exports.InputMode || (exports.InputMode = {}));
/**
 * Compression type of the data.
 *
 */
var CompressionType;
(function (CompressionType) {
    /**
     * None compression type
     */
    CompressionType["NONE"] = "None";
    /**
     * Gzip compression type
     */
    CompressionType["GZIP"] = "Gzip";
})(CompressionType = exports.CompressionType || (exports.CompressionType = {}));
/**
 * Describes the container, as part of model definition.
 *
 * @see https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ContainerDefinition.html
 */
class ContainerDefinition {
    constructor(options) {
        this.options = options;
        jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_ContainerDefinitionOptions(options);
    }
    /**
     * Called when the ContainerDefinition type configured on Sagemaker Task
     */
    bind(task) {
        var _f, _g, _h;
        jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_ISageMakerTask(task);
        return {
            parameters: {
                ContainerHostname: this.options.containerHostName,
                Image: (_f = this.options.image) === null || _f === void 0 ? void 0 : _f.bind(task).imageUri,
                Mode: this.options.mode,
                ModelDataUrl: (_g = this.options.modelS3Location) === null || _g === void 0 ? void 0 : _g.bind(task, { forReading: true }).uri,
                ModelPackageName: this.options.modelPackageName,
                Environment: (_h = this.options.environmentVariables) === null || _h === void 0 ? void 0 : _h.value,
            },
        };
    }
}
exports.ContainerDefinition = ContainerDefinition;
_c = JSII_RTTI_SYMBOL_1;
ContainerDefinition[_c] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.ContainerDefinition", version: "1.146.0" };
/**
 * Specifies how many models the container hosts
 *
 */
var Mode;
(function (Mode) {
    /**
     * Container hosts a single model
     */
    Mode["SINGLE_MODEL"] = "SingleModel";
    /**
     * Container hosts multiple models
     *
     * @see https://docs.aws.amazon.com/sagemaker/latest/dg/multi-model-endpoints.html
     */
    Mode["MULTI_MODEL"] = "MultiModel";
})(Mode = exports.Mode || (exports.Mode = {}));
/**
 * The generation of Elastic Inference (EI) instance
 *
 * @see https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html
 */
class AcceleratorClass {
    /**
     * @param version - Elastic Inference accelerator generation
     */
    constructor(version) {
        this.version = version;
    }
    /**
     * Custom AcceleratorType
     * @param version - Elastic Inference accelerator generation
    */
    static of(version) { return new AcceleratorClass(version); }
}
exports.AcceleratorClass = AcceleratorClass;
_d = JSII_RTTI_SYMBOL_1;
AcceleratorClass[_d] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.AcceleratorClass", version: "1.146.0" };
/**
 * Elastic Inference accelerator 1st generation
 */
AcceleratorClass.EIA1 = AcceleratorClass.of('eia1');
/**
 * Elastic Inference accelerator 2nd generation
 */
AcceleratorClass.EIA2 = AcceleratorClass.of('eia2');
/**
 * The size of the Elastic Inference (EI) instance to use for the production variant.
 * EI instances provide on-demand GPU computing for inference
 *
 * @see https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html
 */
class AcceleratorType {
    constructor(instanceTypeIdentifier) {
        this.instanceTypeIdentifier = instanceTypeIdentifier;
    }
    /**
     * AcceleratorType
     *
     * This class takes a combination of a class and size.
     */
    static of(acceleratorClass, instanceSize) {
        jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_AcceleratorClass(acceleratorClass);
        return new AcceleratorType(`ml.${acceleratorClass}.${instanceSize}`);
    }
    /**
     * Return the accelerator type as a dotted string
     */
    toString() {
        return this.instanceTypeIdentifier;
    }
}
exports.AcceleratorType = AcceleratorType;
_e = JSII_RTTI_SYMBOL_1;
AcceleratorType[_e] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.AcceleratorType", version: "1.146.0" };
/**
 * Specifies the number of records to include in a mini-batch for an HTTP inference request.
 *
 */
var BatchStrategy;
(function (BatchStrategy) {
    /**
     * Fits multiple records in a mini-batch.
     */
    BatchStrategy["MULTI_RECORD"] = "MultiRecord";
    /**
     * Use a single record when making an invocation request.
     */
    BatchStrategy["SINGLE_RECORD"] = "SingleRecord";
})(BatchStrategy = exports.BatchStrategy || (exports.BatchStrategy = {}));
/**
 * Method to use to split the transform job's data files into smaller batches.
 *
 */
var SplitType;
(function (SplitType) {
    /**
     * Input data files are not split,
     */
    SplitType["NONE"] = "None";
    /**
     * Split records on a newline character boundary.
     */
    SplitType["LINE"] = "Line";
    /**
     * Split using MXNet RecordIO format.
     */
    SplitType["RECORD_IO"] = "RecordIO";
    /**
     * Split using TensorFlow TFRecord format.
     */
    SplitType["TF_RECORD"] = "TFRecord";
})(SplitType = exports.SplitType || (exports.SplitType = {}));
/**
 * How to assemble the results of the transform job as a single S3 object.
 *
 */
var AssembleWith;
(function (AssembleWith) {
    /**
     * Concatenate the results in binary format.
     */
    AssembleWith["NONE"] = "None";
    /**
     * Add a newline character at the end of every transformed record.
     */
    AssembleWith["LINE"] = "Line";
})(AssembleWith = exports.AssembleWith || (exports.AssembleWith = {}));
class StandardDockerImage extends DockerImage {
    constructor(opts) {
        super();
        this.allowAnyEcrImagePull = !!opts.allowAnyEcrImagePull;
        this.imageUri = opts.imageUri;
        this.repository = opts.repository;
    }
    bind(task) {
        if (this.repository) {
            this.repository.grantPull(task);
        }
        if (this.allowAnyEcrImagePull) {
            task.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
                actions: [
                    'ecr:BatchCheckLayerAvailability',
                    'ecr:GetDownloadUrlForLayer',
                    'ecr:BatchGetImage',
                ],
                resources: ['*'],
            }));
        }
        return {
            imageUri: this.imageUri,
        };
    }
}
class StandardS3Location extends S3Location {
    constructor(opts) {
        super();
        this.bucket = opts.bucket;
        this.keyGlob = `${opts.keyPrefix || ''}*`;
        this.uri = opts.uri;
    }
    bind(task, opts) {
        if (this.bucket) {
            if (opts.forReading) {
                this.bucket.grantRead(task, this.keyGlob);
            }
            if (opts.forWriting) {
                this.bucket.grantWrite(task, this.keyGlob);
            }
        }
        else {
            const actions = new Array();
            if (opts.forReading) {
                actions.push('s3:GetObject', 's3:ListBucket');
            }
            if (opts.forWriting) {
                actions.push('s3:PutObject');
            }
            task.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({ actions, resources: ['*'] }));
        }
        return { uri: this.uri };
    }
}
//# sourceMappingURL=data:application/json;base64,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