"""
About:
------
The morphology_analyzer command is designed to provide morphological analysis for Arabic text using the SinaTools morph_analyzer API. Users can specify the language and desired analysis task (lemmatization, part-of-speech tagging, or full morphological analysis), and flag.

Usage:
------
Below is the usage information that can be generated by running morphology_analyzer --help.

.. code-block:: none

    morphology_analyzer --text=TEXT [OPTIONS]
    morphology_analyzer --file=FILE [OPTIONS]

Options:
--------

.. code-block:: none

  --text TEXT
        The text that needs to be morphologically analyzed. 

  --file FILE
        File containing the text to be morphologically analyzed

  --language LANGUAGE [default=MSA]
        Specifies the language for the analysis. In the current version, MSA is only supported.

  --task TASK [default=full]
        Determines the specific type of morphological analysis to be performed. Available options are:
          - lemmatization: the morphological solution includes only the lemma_id, lemma, token, and token frequency.
          - pos: the morphological solution includes only the part-of-speech, token, and token frequency.
          - root: the morphological solution includes only the root, token, and token frequency.
          - full: the morphological solution includes the lemma_id, lemma, part-of-speech, root, token, and token frequency.
        The default is full.
  
  --flag FLAG [default=1]
        The flag to filter the returned results. If the flag is `1`, the solution with the highest frequency will be returned. If the flag is `*`, all solutions will be returned, ordered descendingly, with the highest frequency solution first. The default flag if not specified is `1`.

Examples:
---------

.. code-block:: none

  morphology_analyzer --text "Your Arabic text here" --language MSA --task full --flag 1
  morphology_analyzer --file "path/to/your/file.txt" --language MSA --task full --flag 1

"""

import argparse
from sinatools.morphology.morph_analyzer import analyze
from sinatools.utils.readfile import read_file

def main():
    parser = argparse.ArgumentParser(description='Morphological Analysis using SinaTools')
      
    parser.add_argument('--text', type=str, help='Text to be morphologically analyzed')
    parser.add_argument('--file', type=str, help='File containing the text to be morphologically analyzed')
    parser.add_argument('--language', type=str, default='MSA', help='Language for analysis (default: MSA)')
    parser.add_argument('--task', type=str, default='full', choices=['lemmatization', 'pos', 'root', 'full'], help='Task for the result filter [lemmatization, pos, root, full] (default: full)')
    parser.add_argument('--flag', type=str, default='1', choices=['1','*'], help='The flag to filter the returned results')

    args = parser.parse_args()

    if args.text is None and args.file is None:
        print("Error: Either --text or --file argument must be provided.")
        return

    # Get the input either from the --text argument or from the file specified in the --file argument
    input_text = args.text if args.text else " ".join(read_file(args.file))

    # Perform morphological analysis
    results = analyze(input_text, args.language, args.task, args.flag)

    # Print the results
    for result in results:
        print(result)

if __name__ == '__main__':
    main()
