"""
Bytecode Optimizer for fr
Performs peephole optimizations on generated bytecode
"""

from typing import List, Dict, Tuple

class BytecodeOptimizer:
    """Optimizes bytecode instruction sequences"""

    def __init__(self):
        self.labels: Dict[str, int] = {}  # Label -> line number

    def optimize(self, bytecode: str) -> str:
        """Apply all optimization passes to bytecode"""
        lines = bytecode.strip().split('\n')

        # Pass 1: Combine common patterns (does INC_LOCAL conversion)
        lines = self.combine_patterns(lines)

        # Pass 2: Merge consecutive constant instructions
        lines = self.merge_constants(lines)

        # Pass 3: Optimize variable copies (LOAD+STORE -> COPY_LOCAL_REF)
        lines = self.optimize_variable_copies(lines)

        # Pass 4: Remove redundant LOAD/STORE pairs
        lines = self.remove_redundant_loads(lines)

        # Pass 5: Remove dead stores
        lines = self.remove_dead_stores(lines)

        return '\n'.join(lines)

    def merge_constants(self, lines: List[str]) -> List[str]:
        """
        Merge consecutive constant instructions into multi-arg versions.
        E.g., CONST_I64 1, CONST_I64 2 -> CONST_I64 1 2
        """
        result = []
        i = 0

        while i < len(lines):
            line = lines[i].strip()
            indent = lines[i][:len(lines[i]) - len(lines[i].lstrip())] if lines[i] else '  '

            # Check for consecutive CONST_I64 instructions
            if line.startswith('CONST_I64 '):
                constants = [line.split('CONST_I64 ', 1)[1]]
                j = i + 1
                
                # Collect consecutive CONST_I64 instructions
                while j < len(lines) and lines[j].strip().startswith('CONST_I64 '):
                    constants.append(lines[j].strip().split('CONST_I64 ', 1)[1])
                    j += 1
                
                # Only merge if we have 2 or more
                if len(constants) >= 2:
                    result.append(f"{indent}CONST_I64 {' '.join(constants)}")
                    i = j
                    continue

            # Check for consecutive CONST_F64 instructions
            if line.startswith('CONST_F64 '):
                constants = [line.split('CONST_F64 ', 1)[1]]
                j = i + 1
                
                while j < len(lines) and lines[j].strip().startswith('CONST_F64 '):
                    constants.append(lines[j].strip().split('CONST_F64 ', 1)[1])
                    j += 1
                
                if len(constants) >= 2:
                    result.append(f"{indent}CONST_F64 {' '.join(constants)}")
                    i = j
                    continue

            # Check for consecutive CONST_BOOL instructions
            if line.startswith('CONST_BOOL '):
                constants = [line.split('CONST_BOOL ', 1)[1]]
                j = i + 1
                
                while j < len(lines) and lines[j].strip().startswith('CONST_BOOL '):
                    constants.append(lines[j].strip().split('CONST_BOOL ', 1)[1])
                    j += 1
                
                if len(constants) >= 2:
                    result.append(f"{indent}CONST_BOOL {' '.join(constants)}")
                    i = j
                    continue

            # Check for consecutive CONST_STR instructions
            # Note: These need special handling due to quoted strings
            if line.startswith('CONST_STR "'):
                constants = []
                j = i
                
                while j < len(lines) and lines[j].strip().startswith('CONST_STR "'):
                    # Extract the quoted string (everything after 'CONST_STR ')
                    const_str = lines[j].strip()[10:]  # len('CONST_STR ') = 10
                    constants.append(const_str)
                    j += 1
                
                if len(constants) >= 2:
                    result.append(f"{indent}CONST_STR {' '.join(constants)}")
                    i = j
                    continue

            result.append(lines[i])
            i += 1

        return result

    def optimize_variable_copies(self, lines: List[str]) -> List[str]:
        """
        Convert LOAD x; STORE y patterns to COPY_LOCAL_REF x y when safe.
        Safe when: y is not modified before next use (simple heuristic: y is reassigned before next load)
        """
        result = []
        i = 0

        while i < len(lines):
            line = lines[i].strip()
            indent = lines[i][:len(lines[i]) - len(lines[i].lstrip())] if line else '  '

            # Pattern: LOAD x, STORE y
            if (i + 1 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('STORE ')):

                src_var = line.split()[1]
                dst_var = lines[i+1].strip().split()[1]

                # Don't optimize if it's the same variable (handled by other pass)
                if src_var != dst_var:
                    if is_safe := self._is_copy_ref_safe(
                        lines, i + 2, src_var, dst_var
                    ):
                        result.append(f"{indent}COPY_LOCAL_REF {src_var} {dst_var}")
                    else:
                        result.append(f"{indent}COPY_LOCAL {src_var} {dst_var}")
                    i += 2
                    continue

            result.append(lines[i])
            i += 1

        return result

    def _is_copy_ref_safe(self, lines: List[str], start_idx: int, src_var: str, dst_var: str, lookahead: int = 10) -> bool:
        """
        Determine if using COPY_LOCAL_REF is safe.
        Safe if dst is not modified (via ADD/SUB/etc) before being reassigned.
        """
        for i in range(start_idx, min(start_idx + lookahead, len(lines))):
            line = lines[i].strip()

            # If we hit a label or control flow, be conservative
            if line.startswith('LABEL ') or line.startswith('JUMP'):
                return False

            # If dst is stored to again, it's safe (old alias is replaced)
            if (
                line.startswith(f'STORE {dst_var}')
                or line.startswith('COPY_LOCAL_REF ')
                and dst_var in line
            ):
                return True

            # If dst is loaded for an operation, we need to copy (not safe for ref)
            if line.startswith(f'LOAD {dst_var}'):
                # Check if next instruction modifies it
                if i + 1 < len(lines):
                    next_line = lines[i+1].strip()
                    # If it's used in arithmetic, we need deep copy
                    if any(next_line.startswith(op) for op in ['ADD', 'SUB', 'MUL', 'DIV', 'MOD']):
                        return False

            # If src is modified, the alias becomes invalid
            if line.startswith(f'STORE {src_var}'):
                return False

        # Default: be conservative
        return False

    def remove_redundant_loads(self, lines: List[str]) -> List[str]:
        """Remove STORE followed by LOAD of same variable when safe"""
        result = []
        i = 0
        while i < len(lines):
            line = lines[i].strip()

            # Check for STORE N followed by LOAD N
            if i + 1 < len(lines) and line.startswith('STORE '):
                next_line = lines[i + 1].strip()
                if next_line.startswith('LOAD '):
                    var1 = line.split()[1]
                    var2 = next_line.split()[1]
                    if var1 == var2:
                        # Only optimize if the LOAD is immediately consumed by an operation
                        # Check what follows the LOAD
                        if i + 2 < len(lines):
                            following_line = lines[i + 2].strip()
                            # Safe to optimize if followed by an arithmetic/comparison operation
                            safe_ops = ['ADD_', 'SUB_', 'MUL_', 'DIV_', 'MOD_', 'CMP_', 'AND', 'OR', 'NOT']
                            is_safe = any(following_line.startswith(op) for op in safe_ops)
                            
                            if is_safe:
                                # Keep STORE, add DUP before it, skip LOAD
                                result.append(lines[i].replace('STORE', 'DUP\n  STORE'))
                                i += 2
                                continue
                        # If not safe or no following line, keep both STORE and LOAD

            result.append(lines[i])
            i += 1

        return result

    def combine_patterns(self, lines: List[str]) -> List[str]:
        """Combine common instruction patterns into optimized forms"""
        result = []
        i = 0

        while i < len(lines):
            line = lines[i].strip()
            indent = lines[i][:len(lines[i]) - len(lines[i].lstrip())] if line else '  '

            # Pattern: LOAD N, CONST_I64 1, ADD_I64, STORE N -> INC_LOCAL N
            if (i + 3 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip() == 'CONST_I64 1' and
                lines[i+2].strip() == 'ADD_I64' and
                lines[i+3].strip().startswith('STORE ')):

                load_var = line.split()[1]
                store_var = lines[i+3].strip().split()[1]

                if load_var == store_var:
                    # Replace with optimized INC instruction
                    result.append(f"{indent}INC_LOCAL {load_var}")
                    i += 4
                    continue

            # Pattern: LOAD N, CONST_I64 1, SUB_I64, STORE N -> DEC_LOCAL N
            if (i + 3 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip() == 'CONST_I64 1' and
                lines[i+2].strip() == 'SUB_I64' and
                lines[i+3].strip().startswith('STORE ')):

                load_var = line.split()[1]
                store_var = lines[i+3].strip().split()[1]

                if load_var == store_var:
                    result.append(f"{indent}DEC_LOCAL {load_var}")
                    i += 4
                    continue

            # Pattern: Multiple ADD_CONST_I64 -> ADD_CONST_I64 with multiple values
            if line.startswith('ADD_CONST_I64 '):
                constants = [line.split('ADD_CONST_I64 ', 1)[1]]
                j = i + 1
                
                while j < len(lines) and lines[j].strip().startswith('ADD_CONST_I64 '):
                    constants.append(lines[j].strip().split('ADD_CONST_I64 ', 1)[1])
                    j += 1
                
                if len(constants) >= 2:
                    result.append(f"{indent}ADD_CONST_I64 {' '.join(constants)}")
                    i = j
                    continue

            # Pattern: CONST_I64 N, ADD_I64 -> ADD_CONST_I64 N
            if (i + 1 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip() == 'ADD_I64'):

                const_val = line.split()[1]
                result.append(f"{indent}ADD_CONST_I64 {const_val}")
                i += 2
                continue

            # Pattern: CONST_I64 N, SUB_I64 -> SUB_CONST_I64 N
            if (i + 1 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip() == 'SUB_I64'):

                const_val = line.split()[1]
                result.append(f"{indent}SUB_CONST_I64 {const_val}")
                i += 2
                continue

            # Pattern: CONST_I64 N, MUL_I64 -> MUL_CONST_I64 N
            if (i + 1 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip() == 'MUL_I64'):

                const_val = line.split()[1]
                result.append(f"{indent}MUL_CONST_I64 {const_val}")
                i += 2
                continue

            # Pattern: CONST_I64 N, DIV_I64 -> DIV_CONST_I64 N
            if (i + 1 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip() == 'DIV_I64'):

                const_val = line.split()[1]
                result.append(f"{indent}DIV_CONST_I64 {const_val}")
                i += 2
                continue

            # Pattern: CONST_BOOL N, AND -> AND_CONST N
            if (i + 1 < len(lines) and
                line.startswith('CONST_BOOL ') and
                lines[i+1].strip() == 'AND'):

                const_val = line.split()[1]
                result.append(f"{indent}AND_CONST {const_val}")
                i += 2
                continue

            # Pattern: CONST_BOOL N, OR -> OR_CONST N
            if (i + 1 < len(lines) and
                line.startswith('CONST_BOOL ') and
                lines[i+1].strip() == 'OR'):

                const_val = line.split()[1]
                result.append(f"{indent}OR_CONST {const_val}")
                i += 2
                continue

            # Pattern: CONST_I64 N, AND_I64 -> AND_CONST_I64 N
            if (i + 1 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip() == 'AND_I64'):

                const_val = line.split()[1]
                result.append(f"{indent}AND_CONST_I64 {const_val}")
                i += 2
                continue

            # Pattern: CONST_I64 N, OR_I64 -> OR_CONST_I64 N
            if (i + 1 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip() == 'OR_I64'):

                const_val = line.split()[1]
                result.append(f"{indent}OR_CONST_I64 {const_val}")
                i += 2
                continue

            # Pattern: CONST_I64 N, XOR_I64 -> XOR_CONST_I64 N
            if (i + 1 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip() == 'XOR_I64'):

                const_val = line.split()[1]
                result.append(f"{indent}XOR_CONST_I64 {const_val}")
                i += 2
                continue

            # Pattern: LOAD N, LOAD N -> LOAD N, DUP
            if (i + 1 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ')):

                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]

                if var1 == var2:
                    result.extend((lines[i], f"{indent}DUP"))
                    i += 2
                    continue

            # NEW: Pattern: LOAD a, LOAD b, ADD_I64 -> LOAD2_ADD_I64 a b (before multi-LOAD combining!)
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'ADD_I64'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                result.append(f"{indent}LOAD2_ADD_I64 {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, SUB_I64 -> LOAD2_SUB_I64 a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'SUB_I64'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                result.append(f"{indent}LOAD2_SUB_I64 {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, MUL_I64 -> LOAD2_MUL_I64 a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'MUL_I64'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                result.append(f"{indent}LOAD2_MUL_I64 {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, CMP_LT -> LOAD2_CMP_LT a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'CMP_LT'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                result.append(f"{indent}LOAD2_CMP_LT {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, CMP_GT -> LOAD2_CMP_GT a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'CMP_GT'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                result.append(f"{indent}LOAD2_CMP_GT {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, CMP_LE -> LOAD2_CMP_LE a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'CMP_LE'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                result.append(f"{indent}LOAD2_CMP_LE {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, CMP_GE -> LOAD2_CMP_GE a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'CMP_GE'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                result.append(f"{indent}LOAD2_CMP_GE {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, CMP_EQ -> LOAD2_CMP_EQ a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'CMP_EQ'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                result.append(f"{indent}LOAD2_CMP_EQ {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, CMP_NE -> LOAD2_CMP_NE a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'CMP_NE'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                result.append(f"{indent}LOAD2_CMP_NE {var1} {var2}")
                i += 3
                continue

            # Pattern: Multiple consecutive LOAD instructions -> LOAD with multiple args
            # Look for sequences of LOAD instructions
            # BUT: Don't combine 2 LOADs if followed by fusable operation (let LOAD2+OP handle it)
            if line.startswith('LOAD '):
                loads = [line.strip().split()[1]]
                j = i + 1
                while j < len(lines) and lines[j].strip().startswith('LOAD '):
                    loads.append(lines[j].strip().split()[1])
                    j += 1

                # Check if this is exactly 2 LOADs followed by a fusable operation
                fusable_ops = {'ADD_I64', 'SUB_I64', 'MUL_I64', 'CMP_LT', 'CMP_GT', 'CMP_LE', 'CMP_GE', 'CMP_EQ', 'CMP_NE'}
                if len(loads) == 2 and j < len(lines) and lines[j].strip() in fusable_ops:
                    # Don't combine - let it fall through to LOAD2+OP patterns below
                    pass
                elif len(loads) >= 2:
                    # Combine multiple LOADs
                    result.append(f"{indent}LOAD {' '.join(loads)}")
                    i = j
                    continue

            # Pattern: LOAD a b (multi-LOAD), OP -> LOAD2_OP a b (must run AFTER multi-LOAD combining)
            # Check if this is a multi-arg LOAD followed by fused-capable operation
            stripped_line = line.strip()
            if stripped_line.startswith('LOAD ') and len(stripped_line.split()) == 3:  # LOAD with exactly 2 args
                parts = stripped_line.split()
                var1, var2 = parts[1], parts[2]

                # Check what operation follows
                if i + 1 < len(lines):
                    next_op = lines[i+1].strip()

                    if next_op == 'ADD_I64':
                        result.append(f"{indent}LOAD2_ADD_I64 {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'SUB_I64':
                        result.append(f"{indent}LOAD2_SUB_I64 {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'MUL_I64':
                        result.append(f"{indent}LOAD2_MUL_I64 {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'CMP_LT':
                        result.append(f"{indent}LOAD2_CMP_LT {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'CMP_GT':
                        result.append(f"{indent}LOAD2_CMP_GT {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'CMP_LE':
                        result.append(f"{indent}LOAD2_CMP_LE {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'CMP_GE':
                        result.append(f"{indent}LOAD2_CMP_GE {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'CMP_EQ':
                        result.append(f"{indent}LOAD2_CMP_EQ {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'CMP_NE':
                        result.append(f"{indent}LOAD2_CMP_NE {var1} {var2}")
                        i += 2
                        continue

            # Pattern: Multiple LOAD src, STORE dst pairs -> FUSED_LOAD_STORE
            # Look for sequences of LOAD/STORE pairs
            if line.startswith('LOAD ') and i + 1 < len(lines) and lines[i+1].strip().startswith('STORE '):
                # Found first pair, look for more consecutive pairs
                pairs = []
                j = i
                while (j + 1 < len(lines) and 
                       lines[j].strip().startswith('LOAD ') and
                       lines[j+1].strip().startswith('STORE ')):
                    src = lines[j].strip().split()[1]
                    dst = lines[j+1].strip().split()[1]
                    pairs.append((src, dst))
                    j += 2

                # If we found 2 or more pairs, use FUSED_LOAD_STORE
                if len(pairs) >= 2:
                        # Build the fused instruction
                    args = ' '.join(f"{src} {dst}" for src, dst in pairs)
                    result.append(f"{indent}FUSED_LOAD_STORE {args}")
                    i = j
                    continue

            # Pattern: Multiple STORE dst, LOAD src pairs -> FUSED_STORE_LOAD
            # Look for sequences of STORE/LOAD pairs
            if line.startswith('STORE ') and i + 1 < len(lines) and lines[i+1].strip().startswith('LOAD '):
                # Found first pair, look for more consecutive pairs
                pairs = []
                j = i
                while (j + 1 < len(lines) and 
                       lines[j].strip().startswith('STORE ') and
                       lines[j+1].strip().startswith('LOAD ')):
                    dst = lines[j].strip().split()[1]
                    src = lines[j+1].strip().split()[1]
                    pairs.append((dst, src))
                    j += 2

                # If we found 2 or more pairs, use FUSED_STORE_LOAD
                if len(pairs) >= 2:
                    indent = lines[i][:len(lines[i]) - len(lines[i].lstrip())] if lines[i].startswith('STORE') else '  '
                    # Build the fused instruction
                    args = ' '.join(f"{dst} {src}" for dst, src in pairs)
                    result.append(f"{indent}FUSED_STORE_LOAD {args}")
                    i = j
                    continue

            # Pattern: LOAD a, LOAD b, SWAP -> LOAD b, LOAD a (reduce SWAP usage)
            # This is actually worse, so skip it

            # Pattern: DUP, LOAD N -> LOAD N, DUP (may enable other optimizations)
            # Skip for now as it's context-dependent

            result.append(lines[i])
            i += 1

        return result

    def remove_dead_stores(self, lines: List[str]) -> List[str]:
        """Remove stores to variables that are never loaded (basic pass)"""
        # Build label map first
        label_map = {}
        for i, line in enumerate(lines):
            if line.strip().startswith('LABEL '):
                label_name = line.strip().split()[1]
                label_map[label_name] = i

        # For now, just return as-is (dead store elimination is complex with control flow)
        # TODO: Implement proper liveness analysis
        return lines

