"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MathExpression = exports.Metric = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const alarm_1 = require("./alarm");
const metric_util_1 = require("./private/metric-util");
const statistic_1 = require("./private/statistic");
/**
 * A metric emitted by a service
 *
 * The metric is a combination of a metric identifier (namespace, name and dimensions)
 * and an aggregation function (statistic, period and unit).
 *
 * It also contains metadata which is used only in graphs, such as color and label.
 * It makes sense to embed this in here, so that compound constructs can attach
 * that metadata to metrics they expose.
 *
 * This class does not represent a resource, so hence is not a construct. Instead,
 * Metric is an abstraction that makes it easy to specify metrics for use in both
 * alarms and graphs.
 */
class Metric {
    constructor(props) {
        var _c;
        jsiiDeprecationWarnings._aws_cdk_aws_cloudwatch_MetricProps(props);
        this.period = props.period || cdk.Duration.minutes(5);
        const periodSec = this.period.toSeconds();
        if (periodSec !== 1 && periodSec !== 5 && periodSec !== 10 && periodSec !== 30 && periodSec % 60 !== 0) {
            throw new Error(`'period' must be 1, 5, 10, 30, or a multiple of 60 seconds, received ${periodSec}`);
        }
        this.dimensions = this.validateDimensions((_c = props.dimensionsMap) !== null && _c !== void 0 ? _c : props.dimensions);
        this.namespace = props.namespace;
        this.metricName = props.metricName;
        // Try parsing, this will throw if it's not a valid stat
        this.statistic = statistic_1.normalizeStatistic(props.statistic || 'Average');
        this.label = props.label;
        this.color = props.color;
        this.unit = props.unit;
        this.account = props.account;
        this.region = props.region;
    }
    /**
     * Grant permissions to the given identity to write metrics.
     *
     * @param grantee The IAM identity to give permissions to.
     */
    static grantPutMetricData(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['cloudwatch:PutMetricData'],
            resourceArns: ['*'],
        });
    }
    /**
     * Return a copy of Metric `with` properties changed.
     *
     * All properties except namespace and metricName can be changed.
     *
     * @param props The set of properties to change.
     */
    with(props) {
        var _c, _d;
        jsiiDeprecationWarnings._aws_cdk_aws_cloudwatch_MetricOptions(props);
        // Short-circuit creating a new object if there would be no effective change
        if ((props.label === undefined || props.label === this.label)
            && (props.color === undefined || props.color === this.color)
            && (props.statistic === undefined || props.statistic === this.statistic)
            && (props.unit === undefined || props.unit === this.unit)
            && (props.account === undefined || props.account === this.account)
            && (props.region === undefined || props.region === this.region)
            // For these we're not going to do deep equality, misses some opportunity for optimization
            // but that's okay.
            && (props.dimensions === undefined)
            && (props.dimensionsMap === undefined)
            && (props.period === undefined || props.period.toSeconds() === this.period.toSeconds())) {
            return this;
        }
        return new Metric({
            dimensionsMap: (_d = (_c = props.dimensionsMap) !== null && _c !== void 0 ? _c : props.dimensions) !== null && _d !== void 0 ? _d : this.dimensions,
            namespace: this.namespace,
            metricName: this.metricName,
            period: ifUndefined(props.period, this.period),
            statistic: ifUndefined(props.statistic, this.statistic),
            unit: ifUndefined(props.unit, this.unit),
            label: ifUndefined(props.label, this.label),
            color: ifUndefined(props.color, this.color),
            account: ifUndefined(props.account, this.account),
            region: ifUndefined(props.region, this.region),
        });
    }
    /**
     * Attach the metric object to the given construct scope
     *
     * Returns a Metric object that uses the account and region from the Stack
     * the given construct is defined in. If the metric is subsequently used
     * in a Dashboard or Alarm in a different Stack defined in a different
     * account or region, the appropriate 'region' and 'account' fields
     * will be added to it.
     *
     * If the scope we attach to is in an environment-agnostic stack,
     * nothing is done and the same Metric object is returned.
     */
    attachTo(scope) {
        const stack = cdk.Stack.of(scope);
        return this.with({
            region: cdk.Token.isUnresolved(stack.region) ? undefined : stack.region,
            account: cdk.Token.isUnresolved(stack.account) ? undefined : stack.account,
        });
    }
    toMetricConfig() {
        const dims = this.dimensionsAsList();
        return {
            metricStat: {
                dimensions: dims.length > 0 ? dims : undefined,
                namespace: this.namespace,
                metricName: this.metricName,
                period: this.period,
                statistic: this.statistic,
                unitFilter: this.unit,
                account: this.account,
                region: this.region,
            },
            renderingProperties: {
                color: this.color,
                label: this.label,
            },
        };
    }
    /** @deprecated use toMetricConfig() */
    toAlarmConfig() {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-cloudwatch.Metric#toAlarmConfig", "use toMetricConfig()");
        const metricConfig = this.toMetricConfig();
        if (metricConfig.metricStat === undefined) {
            throw new Error('Using a math expression is not supported here. Pass a \'Metric\' object instead');
        }
        const stat = statistic_1.parseStatistic(metricConfig.metricStat.statistic);
        return {
            dimensions: metricConfig.metricStat.dimensions,
            namespace: metricConfig.metricStat.namespace,
            metricName: metricConfig.metricStat.metricName,
            period: metricConfig.metricStat.period.toSeconds(),
            statistic: stat.type === 'simple' ? stat.statistic : undefined,
            extendedStatistic: stat.type === 'percentile' ? 'p' + stat.percentile : undefined,
            unit: this.unit,
        };
    }
    /**
     * @deprecated use toMetricConfig()
     */
    toGraphConfig() {
        var _c, _d, _e, _f;
        jsiiDeprecationWarnings.print("@aws-cdk/aws-cloudwatch.Metric#toGraphConfig", "use toMetricConfig()");
        const metricConfig = this.toMetricConfig();
        if (metricConfig.metricStat === undefined) {
            throw new Error('Using a math expression is not supported here. Pass a \'Metric\' object instead');
        }
        return {
            dimensions: metricConfig.metricStat.dimensions,
            namespace: metricConfig.metricStat.namespace,
            metricName: metricConfig.metricStat.metricName,
            renderingProperties: {
                period: metricConfig.metricStat.period.toSeconds(),
                stat: metricConfig.metricStat.statistic,
                color: asString((_c = metricConfig.renderingProperties) === null || _c === void 0 ? void 0 : _c.color),
                label: asString((_d = metricConfig.renderingProperties) === null || _d === void 0 ? void 0 : _d.label),
            },
            // deprecated properties for backwards compatibility
            period: metricConfig.metricStat.period.toSeconds(),
            statistic: metricConfig.metricStat.statistic,
            color: asString((_e = metricConfig.renderingProperties) === null || _e === void 0 ? void 0 : _e.color),
            label: asString((_f = metricConfig.renderingProperties) === null || _f === void 0 ? void 0 : _f.label),
            unit: this.unit,
        };
    }
    /**
     * Make a new Alarm for this metric
     *
     * Combines both properties that may adjust the metric (aggregation) as well
     * as alarm properties.
     */
    createAlarm(scope, id, props) {
        jsiiDeprecationWarnings._aws_cdk_aws_cloudwatch_CreateAlarmOptions(props);
        return new alarm_1.Alarm(scope, id, {
            metric: this.with({
                statistic: props.statistic,
                period: props.period,
            }),
            alarmName: props.alarmName,
            alarmDescription: props.alarmDescription,
            comparisonOperator: props.comparisonOperator,
            datapointsToAlarm: props.datapointsToAlarm,
            threshold: props.threshold,
            evaluationPeriods: props.evaluationPeriods,
            evaluateLowSampleCountPercentile: props.evaluateLowSampleCountPercentile,
            treatMissingData: props.treatMissingData,
            actionsEnabled: props.actionsEnabled,
        });
    }
    toString() {
        return this.label || this.metricName;
    }
    /**
     * Return the dimensions of this Metric as a list of Dimension.
     */
    dimensionsAsList() {
        const dims = this.dimensions;
        if (dims === undefined) {
            return [];
        }
        const list = Object.keys(dims).sort().map(key => ({ name: key, value: dims[key] }));
        return list;
    }
    validateDimensions(dims) {
        if (!dims) {
            return dims;
        }
        var dimsArray = Object.keys(dims);
        if ((dimsArray === null || dimsArray === void 0 ? void 0 : dimsArray.length) > 10) {
            throw new Error(`The maximum number of dimensions is 10, received ${dimsArray.length}`);
        }
        dimsArray.map(key => {
            if (dims[key] === undefined || dims[key] === null) {
                throw new Error(`Dimension value of '${dims[key]}' is invalid`);
            }
            ;
            if (key.length < 1 || key.length > 255) {
                throw new Error(`Dimension name must be at least 1 and no more than 255 characters; received ${key}`);
            }
            ;
            if (dims[key].length < 1 || dims[key].length > 255) {
                throw new Error(`Dimension value must be at least 1 and no more than 255 characters; received ${dims[key]}`);
            }
            ;
        });
        return dims;
    }
}
exports.Metric = Metric;
_a = JSII_RTTI_SYMBOL_1;
Metric[_a] = { fqn: "@aws-cdk/aws-cloudwatch.Metric", version: "1.147.0" };
function asString(x) {
    if (x === undefined) {
        return undefined;
    }
    if (typeof x !== 'string') {
        throw new Error(`Expected string, got ${x}`);
    }
    return x;
}
/**
 * A math expression built with metric(s) emitted by a service
 *
 * The math expression is a combination of an expression (x+y) and metrics to apply expression on.
 * It also contains metadata which is used only in graphs, such as color and label.
 * It makes sense to embed this in here, so that compound constructs can attach
 * that metadata to metrics they expose.
 *
 * MathExpression can also be used for search expressions. In this case,
 * it also optionally accepts a searchRegion and searchAccount property for cross-environment
 * search expressions.
 *
 * This class does not represent a resource, so hence is not a construct. Instead,
 * MathExpression is an abstraction that makes it easy to specify metrics for use in both
 * alarms and graphs.
 */
class MathExpression {
    constructor(props) {
        var _c;
        jsiiDeprecationWarnings._aws_cdk_aws_cloudwatch_MathExpressionProps(props);
        this.period = props.period || cdk.Duration.minutes(5);
        this.expression = props.expression;
        this.usingMetrics = changeAllPeriods((_c = props.usingMetrics) !== null && _c !== void 0 ? _c : {}, this.period);
        this.label = props.label;
        this.color = props.color;
        this.searchAccount = props.searchAccount;
        this.searchRegion = props.searchRegion;
        const invalidVariableNames = Object.keys(this.usingMetrics).filter(x => !validVariableName(x));
        if (invalidVariableNames.length > 0) {
            throw new Error(`Invalid variable names in expression: ${invalidVariableNames}. Must start with lowercase letter and only contain alphanumerics.`);
        }
        this.validateNoIdConflicts();
    }
    /**
     * Return a copy of Metric with properties changed.
     *
     * All properties except namespace and metricName can be changed.
     *
     * @param props The set of properties to change.
     */
    with(props) {
        jsiiDeprecationWarnings._aws_cdk_aws_cloudwatch_MathExpressionOptions(props);
        // Short-circuit creating a new object if there would be no effective change
        if ((props.label === undefined || props.label === this.label)
            && (props.color === undefined || props.color === this.color)
            && (props.period === undefined || props.period.toSeconds() === this.period.toSeconds())
            && (props.searchAccount === undefined || props.searchAccount === this.searchAccount)
            && (props.searchRegion === undefined || props.searchRegion === this.searchRegion)) {
            return this;
        }
        return new MathExpression({
            expression: this.expression,
            usingMetrics: this.usingMetrics,
            label: ifUndefined(props.label, this.label),
            color: ifUndefined(props.color, this.color),
            period: ifUndefined(props.period, this.period),
            searchAccount: ifUndefined(props.searchAccount, this.searchAccount),
            searchRegion: ifUndefined(props.searchRegion, this.searchRegion),
        });
    }
    /**
     * @deprecated use toMetricConfig()
     */
    toAlarmConfig() {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-cloudwatch.MathExpression#toAlarmConfig", "use toMetricConfig()");
        throw new Error('Using a math expression is not supported here. Pass a \'Metric\' object instead');
    }
    /**
     * @deprecated use toMetricConfig()
     */
    toGraphConfig() {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-cloudwatch.MathExpression#toGraphConfig", "use toMetricConfig()");
        throw new Error('Using a math expression is not supported here. Pass a \'Metric\' object instead');
    }
    toMetricConfig() {
        return {
            mathExpression: {
                period: this.period.toSeconds(),
                expression: this.expression,
                usingMetrics: this.usingMetrics,
                searchAccount: this.searchAccount,
                searchRegion: this.searchRegion,
            },
            renderingProperties: {
                label: this.label,
                color: this.color,
            },
        };
    }
    /**
     * Make a new Alarm for this metric
     *
     * Combines both properties that may adjust the metric (aggregation) as well
     * as alarm properties.
     */
    createAlarm(scope, id, props) {
        jsiiDeprecationWarnings._aws_cdk_aws_cloudwatch_CreateAlarmOptions(props);
        return new alarm_1.Alarm(scope, id, {
            metric: this.with({
                period: props.period,
            }),
            alarmName: props.alarmName,
            alarmDescription: props.alarmDescription,
            comparisonOperator: props.comparisonOperator,
            datapointsToAlarm: props.datapointsToAlarm,
            threshold: props.threshold,
            evaluationPeriods: props.evaluationPeriods,
            evaluateLowSampleCountPercentile: props.evaluateLowSampleCountPercentile,
            treatMissingData: props.treatMissingData,
            actionsEnabled: props.actionsEnabled,
        });
    }
    toString() {
        return this.label || this.expression;
    }
    validateNoIdConflicts() {
        const seen = new Map();
        visit(this);
        function visit(metric) {
            metric_util_1.dispatchMetric(metric, {
                withStat() {
                    // Nothing
                },
                withExpression(expr) {
                    for (const [id, subMetric] of Object.entries(expr.usingMetrics)) {
                        const existing = seen.get(id);
                        if (existing && metric_util_1.metricKey(existing) !== metric_util_1.metricKey(subMetric)) {
                            throw new Error(`The ID '${id}' used for two metrics in the expression: '${subMetric}' and '${existing}'. Rename one.`);
                        }
                        seen.set(id, subMetric);
                        visit(subMetric);
                    }
                },
            });
        }
    }
}
exports.MathExpression = MathExpression;
_b = JSII_RTTI_SYMBOL_1;
MathExpression[_b] = { fqn: "@aws-cdk/aws-cloudwatch.MathExpression", version: "1.147.0" };
const VALID_VARIABLE = new RegExp('^[a-z][a-zA-Z0-9_]*$');
function validVariableName(x) {
    return VALID_VARIABLE.test(x);
}
function ifUndefined(x, def) {
    if (x !== undefined) {
        return x;
    }
    return def;
}
/**
 * Change periods of all metrics in the map
 */
function changeAllPeriods(metrics, period) {
    const ret = {};
    for (const [id, metric] of Object.entries(metrics)) {
        ret[id] = changePeriod(metric, period);
    }
    return ret;
}
/**
 * Return a new metric object which is the same type as the input object, but with the period changed
 *
 * Relies on the fact that implementations of `IMetric` are also supposed to have
 * an implementation of `with` that accepts an argument called `period`. See `IModifiableMetric`.
 */
function changePeriod(metric, period) {
    if (isModifiableMetric(metric)) {
        return metric.with({ period });
    }
    throw new Error(`Metric object should also implement 'with': ${metric}`);
}
function isModifiableMetric(m) {
    return typeof m === 'object' && m !== null && !!m.with;
}
//# sourceMappingURL=data:application/json;base64,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