#include <math.h>
#include <units.h>

using namespace std;

const double GL3_slopes[] = {-551.69278079, -605.79514512, -636.15213299, -658.80300828, -676.55076599, -690.54472744,
            -701.54560761, -710.181026, -717.09646085, -722.45285959, -726.6320477 , -729.90778777, -732.59217876,
            -725.97247157, -723.14491661, -705.69099555, -829.68159132, -828.40600751, -673.37509311, -765.29784413,
            -783.61900867, -806.37356577, -796.88444573, -725.521963, -700.88928801, -712.71127905, -742.59453145,
            -881.92433156, -761.91923374, -698.0044229 , -797.16780191, -826.10619031, -794.99220856, -729.73677388,
            -749.30508543, -803.8395992, -813.77289573, -870.70443373, -777.1179954, -831.62791091, -859.59059759,
            -787.86666066, -747.37053658, -808.54076577, -775.94540373, -728.37695852, -761.61380387, -746.58469234,
            -838.38307728, -823.09200585, -872.21013695, -798.80327847, -674.32178971, -768.11989988, -906.73197075,
            -837.55426783, -832.6493757 , -785.64495464, -765.94446157, -858.19335572, -843.82943716, -872.54997042,
            -791.20929353, -822.42686599, -748.02404444, -862.70582236, -811.44058131, -847.96256712, -846.16667029,
            -966.59683409, -803.61974857, -868.60950793, -804.11206744, -775.71521448, -788.29683112, -767.28648949,
            -735.78070989, -785.82307626, -763.09430058, -858.16258687, -815.64937099, -822.81158087, -807.29157272,
            -815.47328989, -822.93690008, -894.41010931, -890.83977794, -778.0005308, -813.32971626, -762.72874948,
            -810.29593147, -817.86407375, -794.73918005, -761.82039903, -729.49376734, -718.50064607, -605.78046037,
            -690.41311223, -714.07738356, -744.89550143, -731.02258855, -733.05950597, -754.30998063, -726.03503714,
            -742.40721759, -731.21859804, -728.22428281, -749.48499624, -777.18451287, -782.00104766, -774.72674022,
            -763.21679755, -823.10274193, -857.27773072, -781.4437606, -782.01844129, -782.71559643, -770.98256986,
            -765.89852265, -786.71412, -744.58493825, -752.60788478,  -745.17262191, -725.9441846, -797.93979234,
            -757.84966773,  -678.02353815, -748.19117061, -725.67777841, -714.06881764,  -906.55763408, -896.4550209,
            -893.8407499, -830.40412699, -823.24665783, -819.867888, -792.43097521, -782.92688697,  -766.72665043,
            -804.17311735, -774.73832324, -845.74682321,  -781.75942348, -767.44128055, -861.79969379, -739.71867345,
            -737.93845321, -778.6664948, -717.82673969, -700.16601006,  -660.87994602, -710.47629246, -732.49634087,
            -729.62839332,  -706.68695301, -744.52048199, -746.7528528, -764.78097545, -718.57090887, -643.88985858,
            -698.02319414, -756.70636591, -1113.92027297, -1276.47888504, -1204.12909648, -1029.71465367, -862.00679549,
            -1193.78590842, -1049.30795299, -699.31614918, -1005.25400499, -939.4756907, -684.63882008, -777.76368979,
            -857.1727384, -1211.91406163, -1188.18856136, -1152.18285168, -1192.98989048, -1257.52329068, -1194.60671517,
            -1214.65582937, -1217.75472203, -1182.35355298, -1177.46261559, -1197.17831922, -1211.47494605, -1166.99939424,
            -1123.863428, -1102.35765406, -1091.05298182, -1120.75118944, -1106.72657604, -1130.8978579, -1090.22409832,
            -1085.27124033, -1061.11776271, -1056.50142, -1042.25771296, -1035.86258062, -1043.04156023, -1078.94958098,
            -1082.43150772, -1021.01384124, -1010.21066082, -1013.35757956, -1017.96026351, -1023.88344429, -1021.9535139,
            -998.4709459, -1019.8221382, -994.66890701, -1015.25033984, -960.42589713,  -972.2079184 ,  -999.68340867,
            -976.0282868, -847.87626765, -838.51548228, -788.59378055, -914.20044586, -900.58920201, -883.06365172,
            -855.10941921, -839.36419231, -809.5129923, -791.74084758, -782.70636602, -747.59843758, -753.44472587,
            -720.68779261, -734.52066631, -747.86050757, -737.78084395, -714.09184707, -717.90235421, -712.38984018,
            -698.66733344, -616.52890327, -611.78280685, -589.29877004, -592.69950957, -593.95281673, -615.48357408,
            -607.73966252, -586.40524506, -559.18479816, -591.03109061, -542.89888991, -526.59574008, -518.5336709,
            -510.32889606, -505.68790436, -506.12841242, -512.48858426, -495.21412805, -468.904347, -441.53200587,
            -471.30891518, -459.8169908, -443.89416953, -389.83475556, -368.3057916, -339.86757641,
            -348.53381862, -404.6146439, -391.86602506, -338.6375786, -370.8504577, -356.95038187,
            -345.37892961, -333.51047517, -295.73629813, -283.84396668, -282.13378594, -309.60356346, -314.76172496,
            -308.21392659, -287.04835248, -251.64953247, -265.30930328, -243.05464677, -265.10837899, -241.74880906,
            -220.59107451, -229.45116954, -264.87154965, -263.31928349, -253.21821248, -253.00343357, -247.49976462,
            -230.42122847, -229.49473522, -225.24223427, -224.0148991, -220.46867297, -210.47233955, -203.57927655,
            -204.35214706, -201.44543131};
    const double GL3_offsets[] = {793.70849244, 873.80992952, 917.83533672, 950.53274812, 976.15604571, 996.31540503,
            1012.10517972, 1024.47229945, 1034.3222566, 1042.00246523, 1048.06896605, 1053.11360075, 1057.14937049,
            1046.57512787, 1040.01098726, 1012.45039039, 1210.9578711, 1211.42014122, 961.67605517, 1103.54244724,
            1136.99133782, 1176.04094644, 1161.45073353, 1049.57448515, 1011.39082653, 1029.08797302, 1071.08817469,
            1282.4401864 , 1105.99632878, 1004.46522152, 1159.33279476, 1204.00754711, 1165.57872868, 1049.59984369,
            1087.19773043, 1172.12626444, 1186.19281845, 1268.6581024 , 1131.92711861, 1212.91485284, 1251.82894365,
            1150.36867455, 1085.74474178, 1177.2385728, 1127.62177707, 1048.42659364, 1106.27839173, 1085.7965831,
            1217.44834628, 1197.96881649, 1269.20601124, 1160.25043317, 970.41573727, 1120.46043477, 1317.36364997,
            1222.30566987, 1216.26334184, 1146.15828409, 1111.76183628, 1252.13667334, 1227.98109006, 1277.20727681,
            1155.10636658, 1201.9361431, 1088.8549693, 1257.77565851, 1187.86496111, 1235.84814265, 1234.39497592,
            1403.46466459, 1178.15685733, 1264.06754168, 1178.95693468, 1134.80835358, 1152.57871462, 1118.54116212,
            1061.04905887, 1146.14203749, 1110.0694841, 1254.14424325, 1196.14669523, 1208.75918245, 1183.44585397,
            1197.20047666, 1204.17016495, 1297.35207539, 1293.06431451, 1131.03123576, 1187.55616658, 1102.98924247,
            1185.9462695, 1194.29170956, 1159.18451674, 1103.88150141, 1057.06098506, 1031.64588258, 865.33600581,
            989.88110532, 1026.90774963, 1079.10821143, 1055.99159809, 1061.51842365, 1091.90478177, 1040.45328385,
            1073.13340971, 1048.84739087, 1042.52986462, 1085.1224431, 1130.60513972, 1137.39857734, 1120.96298628,
            1102.60326765, 1198.40074075, 1249.48285481, 1130.70263946, 1134.72515772, 1133.77426351, 1113.94580124,
            1104.93410777, 1140.26186848, 1067.09204577, 1085.16776851, 1070.02868827, 1036.68073764, 1155.45018225,
            1093.08583178, 961.73468995, 1080.81214076, 1037.63013693, 1021.37738648, 1314.60169324, 1303.11914732,
            1304.04789509, 1213.62646288, 1195.78051154, 1190.68341526, 1142.58395025, 1131.27964128, 1101.36398699,
            1169.13889587, 1123.30808075, 1234.17566981, 1134.78449725, 1105.48512318, 1251.79038192, 1056.86517101,
            1054.01795933, 1125.33683656, 1028.49821655,  998.93437997,  941.91815882, 1017.18998446, 1045.07584783,
            1045.98660017, 1011.32800714, 1070.75600222, 1067.12509504, 1100.20207483, 1022.23220189,  918.2822888,
            1002.01446739, 1094.018716, 1679.52835761, 1919.0889712, 1802.44561194, 1540.08372493, 1262.57328726,
            1809.05894492, 1541.53981364, 1008.71892731, 1477.56671864, 1378.77879518, 986.09962843, 1126.1001675,
            1249.86503491, 1825.29793378, 1799.91241062, 1733.54614372, 1808.06227033, 1905.29121823, 1813.47656778,
            1830.71798682, 1839.7175639, 1792.53512098, 1783.70804371, 1814.88607409, 1825.86015058, 1761.90182835,
            1692.29827315, 1655.17151347, 1640.01228094, 1693.45588323, 1663.79599186, 1696.89058204, 1634.01022311,
            1625.13174206, 1583.00059166, 1579.27927939, 1554.6543492, 1551.65228883, 1565.76998844, 1614.95070486,
            1613.11555339, 1534.57320307, 1501.81932208, 1511.70351344, 1514.66456537, 1515.27098633, 1510.94976038,
            1475.9685498, 1508.92512872, 1472.24098008, 1502.18974905, 1412.90458277, 1432.22527469, 1469.66565212,
            1435.75710536, 1257.20095078, 1238.58978573, 1164.25257236, 1344.29265522, 1321.64136631, 1297.63183771,
            1257.27859104, 1232.15851982, 1180.25402172, 1151.04233092, 1138.53639888, 1086.9626225, 1105.56039328,
            1048.54556185, 1068.49277732, 1096.90491032, 1073.52692961, 1043.58319982, 1050.81791719, 1042.1156963 ,
            1019.18303121, 904.23868185, 892.27481423,  855.77822801, 869.19908699, 872.85264783, 902.62808767,
            891.76639131,  849.82602224, 813.63668761, 862.04884093, 800.13735628, 776.81453142,  761.69067626,
            750.80218545, 748.20340828, 750.24748967, 754.0492277, 730.95182955, 693.7905928, 650.87516019,
            693.60444869, 680.51055921, 656.2089333, 570.98517811, 530.13188875, 492.27718313, 502.84692593,
            591.26110783, 572.97701107, 495.78155877, 543.15680859, 519.61034905, 502.92521147, 480.99952093,
            429.52649963, 409.14396087, 408.72017121,  448.77347866, 453.61191367, 445.62419799, 411.62620202,
            365.31495809,  381.45555374, 351.0943675, 380.85143466, 348.41601124, 318.99871196, 331.0600084,
            375.0829838, 374.74177316, 360.15419887, 358.70983821, 351.75263684, 328.59386721, 326.75874387,
            321.74856769, 320.71125213,  315.0484816, 303.65293855, 295.46765261, 295.06757621, 289.04276895};

double fit_GL1(double z, double frequency){
	// Model for Greenland. Taken from DOI: https://doi.org/10.3189/2015JoG15J057
	// Returns the attenuation length at 75 MHz as a function of depth
	double fit_values[] = {1.16052586e+03, 6.87257150e-02, -9.82378264e-05,
									-3.50628312e-07, -2.21040482e-10, -3.63912864e-14};

	double att_length = 0;
	for (int power = 0; power < 6; power++){
		att_length += fit_values[power] * pow(z, power);
	}

	double att_length_f = att_length - 0.55*utl::m * (frequency/utl::MHz - 75);

	const double min_length = 1 * utl::m;
	if ( att_length_f < min_length ){ att_length_f = min_length; }

	return att_length_f;
}

double fit_GL2(double z, double frequency){
    // Model from the 2021 measurements taken at Summit Station, Greenland: https://arxiv.org/abs/2201.07846
    const double fit_values_GL2[] = {1.20547286e+00, 1.58815679e-05, -2.58901767e-07, -5.16435542e-10, -2.89124473e-13, -4.58987344e-17};
    const double freq_slope = -0.54 * utl::m / utl::MHz;
    const double freq_inter = 852.0 * utl::m;
    double bulk_att_length_f = freq_inter + freq_slope * frequency;
    double att_length_f_GL2 = 0;
    for (int power = 0; power < 6; power++){
		att_length_f_GL2 += bulk_att_length_f * fit_values_GL2[power] * pow(z, power);
	}
	const double min_length = 1 * utl::m;
	if ( att_length_f_GL2 < min_length ){ att_length_f_GL2 = min_length; }

	return att_length_f_GL2;

}

double fit_GL3(double z, double frequency){
    double z_step = 10.0033;   // Spacing of depths in the CSV file
    double z_start = 4.5;  // Sarting depth of CSV file
    int i_row = int(floor((-z - z_start) / z_step));
    if (i_row > 298) {  // Prevent i_row from exceeding length of array
        return GL3_slopes[299] * frequency + GL3_offsets[299];
    }
    if (i_row < 0) {
    i_row = 0;
    }

    double slope_diff = GL3_slopes[i_row + 1] - GL3_slopes[i_row];
    double slope_interpolation = GL3_slopes[i_row] + slope_diff * (-z - i_row * z_step - z_start) / z_step;
    double offset_diff = GL3_offsets[i_row + 1] - GL3_offsets[i_row];
    double offset_interpolation = GL3_offsets[i_row] + offset_diff * (-z - i_row * z_step - z_start) / z_step;
    return frequency * slope_interpolation + offset_interpolation;



}

double get_temperature(double z){
	//return temperature as a function of depth
	// from https://icecube.wisc.edu/~araproject/radio/#icetemperature
	double z2 = abs(z/utl::m);
	return 1.83415e-09*z2*z2*z2 + (-1.59061e-08*z2*z2) + 0.00267687*z2 + (-51.0696 );
}

double get_attenuation_length(double z, double frequency, int model){
	if (z>0) {
		double inf = 1e130;
		return inf;
	}
	if(model == 1) {
		double t = get_temperature(z);
		double f0 = 0.0001;
		double f2 = 3.16;
		double w0 = log(f0);
		double w1 = 0.0;
		double w2 = log(f2);
		double w = log(frequency / utl::GHz);
		double b0 = -6.74890 + t * (0.026709 - t * 0.000884);
		double b1 = -6.22121 - t * (0.070927 + t * 0.001773);
		double b2 = -4.09468 - t * (0.002213 + t * 0.000332);
		double a, bb;
		if(frequency<1. * utl::GHz){
			a = (b1 * w0 - b0 * w1) / (w0 - w1);
			bb = (b1 - b0) / (w1 - w0);
		} else{
			a = (b2 * w1 - b1 * w2) / (w1 - w2);
			bb = (b2 - b1) / (w2 - w1);
		}
		return 1./exp(a +bb*w);
	} else if (model == 2) {

		return fit_GL1(z, frequency);

	} else if (model == 3) {
		double R = 0.82;
		double d_ice = 576 * utl::m;
		double att_length = 460 * utl::m - 180 * utl::m /utl::GHz * frequency;
		att_length *= 1./(1 + att_length / (2 * d_ice) * log(R));  // additional correction for reflection coefficient being less than 1.

		double d = -z * 420. * utl::m / d_ice;
        double L = (1250.*0.08886 * exp(-0.048827 * (225.6746 - 86.517596 * log10(848.870 - (d)))));
        // this differs from the equation published in F. Wu PhD thesis UCI.
        // 262m is supposed to be the depth averaged attenuation length but the
        // integral (int(1/L, 420, 0)/420) ^ -1 = 231.21m and NOT 262m.
        att_length *= L / (231.21 * utl::m);
		return att_length;
	} else if (model == 4) {
	    return fit_GL2(z, frequency);
	} else if (model == 5) {
	    return fit_GL3(z, frequency);
	} else {
		std::cout << "attenuation length model " << model << " unknown. Maybe you need to recompile the C++ ray tracer." << std::endl;
		throw 0;
	}
}
