import sys
import argparse
import json
import os
import numpy as np
from tqdm import tqdm
from tabulate import tabulate
import json
from collections import OrderedDict
import re

import eval_utils
from eval_utils import color, bold, underline
from row_classes import *

parser = argparse.ArgumentParser()
parser.add_argument("-config_file", default='eval_files/report_default',
                    type=str, help="Config file specifying which sheets to use.")
parser.add_argument("-pos", required=True,
                    type=str, help="Part-of-speech to perform the evaluation on.")
parser.add_argument("-db_baseline", default='eval_files/calima-msa-s31_0.4.2.utf8.db',
                    type=str, help="Path of the MSA baseline DB file we will be comparing against.")
parser.add_argument("-db_system", default='databases/camel-morph-msa/XYZ_msa_all_v1.0.db',
                    type=str, help="Path of the MSA baseline DB file we will be comparing against.")
parser.add_argument("-report_dir", default='eval_files/report_default',
                    type=str, help="Paths of the directory containing partial reports generated by the full generative evaluation.")
parser.add_argument("-baseline_feat_subcombs", default='',
                    type=str, help="Object containing feature sub-combinations that are causing a difference with the opposite system, and an explanation of why.")
parser.add_argument("-system_feat_subcombs", default='',
                    type=str, help="Object containing feature sub-combinations that are causing a difference with the opposite system, and an explanation of why.")
parser.add_argument("-baseline_name", default='Calima',
                    type=str, help="Name that will appear in the report for the baseline.")
parser.add_argument("-system_name", default='Camel',
                    type=str, help="Name that will appear in the report for the system.")
parser.add_argument("-camel_tools", default='local', choices=['local', 'official'],
                    type=str, help="Path of the directory containing the camel_tools modules.")
args = parser.parse_args()

with open('configs/config_default.json') as f:
    config = json.load(f)

if args.camel_tools == 'local':
    camel_tools_dir = config['global']['camel_tools']
    sys.path.insert(0, camel_tools_dir)

from camel_tools.morphology.database import MorphologyDB
from camel_tools.morphology.generator import Generator
from camel_tools.utils.charmap import CharMapper

bw2ar = CharMapper.builtin_mapper('bw2ar')
ar2bw = CharMapper.builtin_mapper('ar2bw')
        

def generate_examples(index2lemmas_pos, index2analysis,
                     match_comb_mask=None, match_comb_indexes=None):
    if match_comb_mask is not None:
        match_comb_indexes = np.where(match_comb_mask)
    example_coord = (match_comb_indexes[0][0], match_comb_indexes[1][0])
    lemma, pos = index2lemmas_pos[example_coord[0]]
    feats = index2analysis[example_coord[1]]
    try:
        example_forms_system = generator_system.generate(
            lemma, eval_utils.construct_feats(feats, pos))
        example_forms_system = ','.join(
            set(eval_utils.preprocess_lex_features(form, True)['diac']
                for form in example_forms_system))
        if example_forms_system:
            example_forms_system += bold(color('✓', 'green'))
        else:
            example_forms_system = bold(color('x', 'fail'))
    except:
        example_forms_system = ''
    try:
        example_forms_baseline = generator_baseline.generate(
            lemma, eval_utils.construct_feats(feats, pos), legacy=True)
        example_forms_baseline = ','.join(
            set(eval_utils.preprocess_lex_features(form, True)['diac']
                for form in example_forms_baseline))
        if example_forms_baseline:
            example_forms_baseline += bold(color('✓', 'green'))
        else:
            example_forms_baseline = bold(color('x', 'fail'))
    except:
        example_forms_baseline = ''

    examples_str = (bold(color('lex:', 'warning')) + ar2bw(lemma) + bold(color(' •', 'header')) + '\n' +
                   bold(color('feats:', 'warning')) + '+'.join(feats) + bold(color(' •', 'header')) + '\n' +
                   bold(color(f'{args.system_name.lower()}:', 'warning')) + ar2bw(example_forms_system) + '\n' +
                   bold(color(f'{args.baseline_name.lower()}:', 'warning')) + ar2bw(example_forms_baseline))
    
    return examples_str


def section_title():
    report_title = f"Evaluation Report - {' '.join(pos.upper() for pos in POS)}"
    try:
        terminal_size_col = os.get_terminal_size().columns
    except:
        terminal_size_col = len(report_title)
    print()
    print('=' * terminal_size_col)
    print(report_title)
    print('=' * terminal_size_col)
    print()
    print(bold(underline('DBs used for analysis')))
    print(f'{args.baseline_name}: ' + color(bold(args.db_baseline), 'cyan'))
    print(f'{args.system_name}: ' + color(bold(args.db_system), 'cyan'))
    print()


def section_lemma_overlap():
    print(bold(underline(f'Verb Lemmas overlap between {args.baseline_name} and {args.system_name}')))
    print()
    lemmas_pos_baseline = eval_utils.get_all_lemmas_from_db(db_baseline)
    lemmas_baseline = set([lemma_pos[0] for lemma_pos in lemmas_pos_baseline if lemma_pos[1] in POS])
    lemmas_pos_system = eval_utils.get_all_lemmas_from_db(MorphologyDB(args.db_system))
    lemmas_system = set([lemma_pos[0] for lemma_pos in lemmas_pos_system if lemma_pos[1] in POS])
    rows = []
    header = ['A . B', 'Result', '# lemmas', '(%)', 'Lemmas']
    lemmas_baseline_only = lemmas_baseline - lemmas_system
    lemmas_system_only = lemmas_system - lemmas_baseline
    lemmas_intersect = lemmas_system & lemmas_baseline
    lemmas_union = lemmas_system | lemmas_baseline
    rows.append([f'{args.baseline_name} - {args.system_name}',
                f'{len(lemmas_baseline_only):,}',
                f'{len(lemmas_baseline):,} in A',
                f'{len(lemmas_baseline_only) / len(lemmas_baseline):.2%}',
                ' '.join(sorted(map(ar2bw, lemmas_baseline_only)))])
    rows.append([f'{args.system_name} - {args.baseline_name}',
                f'{len(lemmas_system_only):,}',
                f'{len(lemmas_system):,} in A',
                f'{len(lemmas_system_only) / len(lemmas_system):.2%}',
                ' '.join(sorted(map(ar2bw, lemmas_system_only)))])
    rows.append([f'{args.system_name} ∩ {args.baseline_name}',
                bold(color(f'{len(lemmas_intersect):,}', 'green')),
                f'{len(lemmas_union):,} in A ∪ B',
                f'{len(lemmas_intersect) / len(lemmas_union):.2%}',
                '-'])
    print(tabulate(rows, tablefmt='fancy_grid', headers=header, maxcolwidths=[None, None, None, None, 100]))
    print()


def section_high_level_overlap_stats():
    print(bold(underline(f'Overlap statistics of generated diacs between {args.baseline_name} and {args.system_name}')))
    print()
    rows = []
    slots_total = num_valid_feats * num_valid_lemmas
    slots_filled_mask = mask_not_equal_0_system | mask_not_equal_0_baseline
    slots_filled_total = int(np.sum(slots_filled_mask))

    examples_str = generate_examples(index2lemmas_pos_inter, index2analysis_inter, slots_filled_mask)
    rows.append(['Number of slots filled by at least one of the systems (0-x, x-0, x-y)',
                bold(color(f'{slots_filled_total:,}', 'warning')),
                f'{slots_filled_total/slots_total:.0%}',
                examples_str])
    examples_str = generate_examples(index2lemmas_pos_inter, index2analysis_inter, ~slots_filled_mask)
    rows.append(['Number of slots filled by none of the systems (0-0)',
                f'{slots_total - slots_filled_total:,}',
                f'{(slots_total - slots_filled_total)/slots_total:.0%}',
                examples_str])
    rows.append(['Total number of slots per system',
                f'{slots_total:,}' + '\n(' + bold(color(f'{num_valid_lemmas:,} ', 'green')) + '× ' +
                bold(color(f'{num_valid_feats:,}', 'green')) + ')',
                f'{1:.0%}', '-'])
    assert len(diac_mat_baseline_inter) == len(diac_mat_system_inter)
    print('Number of lemmas evaluated on: ' + bold(color(f'{len(diac_mat_baseline_inter):,}', 'green')) +
        f' ({args.system_name} ∩ {args.baseline_name})')
    print('Total number of feature combinations across both systems: ' + bold(color(f'{num_valid_feats:,}', 'green')))
    print(tabulate(rows, tablefmt='fancy_grid', maxcolwidths=[40, None, None, 60]))
    print()


def section_shared_combinations_high_level_stats():
    print(bold(underline(f'Distribution of feature combination availability across systems for ')) +
      bold(underline(color('SHARED', 'orange'))) + bold(underline(f' feature combination pairs (0-x, x-0, x-y)')))

    notes = color(f"""
    Notes:
        - # diac here is the number of unique diacs generated, and not the number of analyses generated which could generally be more.
        - The meaning of the dash ("-") is anything but zero.
        - A "slot" is a matrix cell representing a lemma and a feature combinatinon from which one a more diacs were generated.
        - "Slot space" means we are counting number of slots while in diac space were are counting number of diacs. This effectively means that
        the # diac is taken as 1 for the purpose of recall analysis.
        - Slots column is the number of feature combinations that both systems were able to generate for listed lemmas with the specified number of diacs.
        - Lemmas column is the number of lemmas with which at least one feature combination generates with the specified number of diacs.
        - Top number in recall distribution (last columns) indicates recall (diac space) by {args.system_name} of {args.baseline_name} and bottom is in slot space
        (displayed only if different). Total recall in slot space basically represents the sum of all categories (in slot space) minus no_intersect.
        - All total recall values are micro-averaged.
    """, 'warning')
    print(notes)
    print()

    header = OrderedDict([
        ((f'# diac ({args.baseline_name})', 8), ('num_diac_baseline',)),
        ((f'# diac ({args.system_name})', 7), ('num_diac_system',)),
        (('Slots', None), ('match_comb_mask_str',)),
        (('Lemmas', None), ('num_lemmas_match_str',)),
        (('Feat combs', 7), ('num_feats_match_str',)),
        (('Example', 60), ('example_str',))
    ])

    combinations = [
        dict(baseline=0, system='≥1',
             match_comb_mask=(diac_mat_baseline_inter == 0) & (diac_mat_system_inter != 0)),
        dict(baseline='≥1', system=0,
             match_comb_mask=(diac_mat_baseline_inter != 0) & (diac_mat_system_inter == 0)),
        dict(baseline='≥1', system='≥1',
             match_comb_mask=(diac_mat_baseline_inter != 0) & (diac_mat_system_inter != 0),
             slots_color='cyan', lemmas_feats_color='orange'),
        dict(baseline='≥0', system='≥0',
             match_comb_mask=(diac_mat_baseline_inter != 0) | (diac_mat_system_inter != 0),
             slots_color='warning', lemmas_feats_color='green')
    ]

    match_total = int(np.sum(mask_not_equal_0_baseline | mask_not_equal_0_system))

    rows = []
    for combination in tqdm(combinations, ncols=100):
        row = DiacCombinationRow(combination, match_total,
                                num_valid_lemmas, num_valid_feats,
                                diac_mat_baseline=diac_mat_baseline_inter,
                                diac_mat_system=diac_mat_system_inter,
                                system_only_mat=system_only_mat_inter,
                                baseline_only_mat=baseline_only_mat_inter,
                                no_intersect_mat=no_intersect_mat_inter)
        if row.match_comb_mask_sum:
            row.example_str = generate_examples(
                index2lemmas_pos_inter, index2analysis_inter, combination['match_comb_mask'])
            rows.append(row)

    for row in rows:
        row.val_and_perc_str('match_comb_mask', row.slots_color)
        for attr in ['num_lemmas_match', 'num_feats_match']:
            row.val_and_perc_str(attr, row.lemmas_feats_color)

    rows = sorted(rows, key=lambda row: row.match_comb_mask_sum
                if row.num_diac_baseline != '≥0' else -1,
                reverse=True)
    rows = [[getattr(row, var[0]) for var in header.values()] for row in rows]
    rows[-1] = [bold(v) for v in rows[-1]]

    print(tabulate(rows, tablefmt='fancy_grid', headers=[h[0] for h in header],
                maxheadercolwidths=[h[1] for h in header]))
    print()


def section_x_y_breakdown():
    print(bold(underline(f'Summary breakdown of the x-y set (coverage of {args.baseline_name} by {args.system_name})')))
    notes = color(f"""
        Notes:
        - The five columns (1) No intersec (2) Exact match (3) Calima super (4) Camel Super (5) Intersec are a distribution of
        the recall from the system's side. So for example, 99% in Calima super and 49.5% in the last column (Recall baseline (micro))
        means that 99% of slots in Camel for that number of diacs combination match the baseline, while 49.5% is an aggregate (micro) as
        seen from the baseline's side because in this case the baseline diacs are a superset of the system's.
        """, 'warning')
    print(notes)

    combinations = [
        dict(baseline=1, system='>1',
            match_comb_mask=((diac_mat_baseline_inter == 1) &
                            (diac_mat_system_inter > 1)),
            slots_color='cyan', diacs_color='green'),
        dict(baseline='>1', system=1,
            match_comb_mask=((diac_mat_baseline_inter > 1) &
                            (diac_mat_system_inter == 1)),
            slots_color='cyan', diacs_color='green'),

        dict(baseline=1, system=1,
            match_comb_mask=((diac_mat_baseline_inter == 1) &
                            (diac_mat_system_inter == 1)),
            slots_color='cyan', diacs_color='green'),
        dict(baseline='x>1', system='y>1',
            match_comb_mask=((diac_mat_baseline_inter > 1) &
                            (diac_mat_system_inter > 1) &
                            (diac_mat_system_inter != diac_mat_baseline_inter)),
            slots_color='cyan', diacs_color='green'),

        dict(baseline='x>1', system='x>1',
            match_comb_mask=((diac_mat_baseline_inter > 1) &
                            (diac_mat_system_inter > 1) &
                            (diac_mat_system_inter == diac_mat_baseline_inter)),
            slots_color='cyan', diacs_color='green'),
        dict(baseline='≥1', system='≥1',
            match_comb_mask=((diac_mat_baseline_inter != 0) &
                            (diac_mat_system_inter != 0)),
            slots_color='cyan', diacs_color='green', lemmas_feats_color='orange')
    ]

    header = OrderedDict([
        ((f'# diac ({args.baseline_name})', 8), ('num_diac_baseline', 8)),
        ((f'# diac ({args.system_name})', 7), ('num_diac_system', 7)),
        (('Slots', None), ('match_comb_mask_str', None)),
        (('Lemmas', None), ('num_lemmas_match_str', None)),
        (('Feat combs', 9), ('num_feats_match_str', 9)),
        (('Example', 60), ('example_str', 60)),
        (('No intersec', 8), ('recall_no_intersect_str', 8)),
        (('Exact match', 7), ('recall_exact_match_str', 7)),
        ((f'{args.baseline_name} super', 8), ('recall_baseline_superset_str', 8)),
        ((f'{args.system_name} super', 7), ('recall_system_superset_str', 7)),
        (('Intersec', 8), ('recall_intersect_str', 8)),
        ((f'Recall {args.baseline_name} (micro)', 8), (f'recall_baseline_str', 8)),
        ((f'Recall {args.system_name} (micro)', 8), (f'recall_system_str', 8))
    ])

    match_x_y_total = int(np.sum(mask_not_equal_0_baseline & mask_not_equal_0_system))

    rows = []
    for i, combination in enumerate(tqdm(combinations, ncols=100)):
        row = DiacCombinationRow(combination, match_x_y_total,
                                num_valid_lemmas, num_valid_feats,
                                diac_mat_baseline=diac_mat_baseline_inter,
                                diac_mat_system=diac_mat_system_inter,
                                system_only_mat=system_only_mat_inter,
                                baseline_only_mat=baseline_only_mat_inter,
                                no_intersect_mat=no_intersect_mat_inter)
        if row.match_comb_mask_sum:
            row.example_str = generate_examples(
                index2lemmas_pos_inter, index2analysis_inter, combination['match_comb_mask'])
            row.val_and_perc_str('match_comb_mask', row.slots_color if i == len(combinations) - 1 else None)
            for attr in ['num_lemmas_match', 'num_feats_match']:
                row.val_and_perc_str(attr, row.lemmas_feats_color if i == len(combinations) - 1 else None)
            rows.append(row)

    rows = sorted(rows, key=lambda row: row.match_comb_mask_sum
                if row.num_diac_baseline != '≥1' else -1,
                reverse=True)
    rows = [[getattr(row, var[0]) for var in header.values()] for row in rows]
    rows[-1] = [bold(v) for v in rows[-1]]

    print(tabulate(rows, tablefmt='fancy_grid', headers=[h[0] for h in header],
                maxheadercolwidths=[h[1] for h in header]))
    print()


def section_unshared_combinations_high_level_stats():
    print(bold(underline(f'Distribution of feature combination availability across systems for ')) +
        bold(underline(color('UNSHARED', 'orange'))) + bold(underline(f' feature combination pairs (0-x, x-0)')))
    notes = color(f"""
    Notes:
    - The unshared feature combination pairs can be divided into two groups:
        1. The ones which contain individual feat:value pairs that are never seen in the opposite systems.
        2. The ones which contain sub-combinations of feat:value pairs that, while seen individually in the opposite system,
        are never seen together in the latter.
    - The last column of the below table lists the ones of the first group. Second group is dealt with in the next section.
    """, 'warning')
    print(notes)
    
    feat_value_pairs = {}
    for system_ in ['baseline', 'system']:
        feat_value_pairs[system_] = set(
            [tuple(feat_value_pair.split(':')) + (pos,)
             for pos, feat_value_pairs in POS2FEAT_VALUE_PAIRS[system_].items()
             for feat_value_pair in feat_value_pairs])
    # This is not intersection of the feat:value pairs of the intersction matrix feat_combs
    # but is the intersection of common feat:value pairs in both systems (whether they occur
    # in the above or not e.g., >a_ques does not occur in the 3828 combs intersection for verbs
    # but occurs in both systems individually, hence it will occur in the below variable).
    # Might need to change this if we want to enable an automatic discovery of mismatching feat:value
    # pairs.
    feat_value_pairs_intersect = feat_value_pairs['baseline'] & feat_value_pairs['system']
    feat_value_pairs = dict(
        intersect=feat_value_pairs['baseline'] & feat_value_pairs['system'],
        baseline_only=feat_value_pairs['baseline'] - feat_value_pairs_intersect,
        system_only=feat_value_pairs['system'] - feat_value_pairs_intersect
    )

    feat2index = {feat: i for i, feat in enumerate(eval_utils.essential_keys_no_lex_pos)}

    if args.system_feat_subcombs:
        feat_subcombs_path_system = os.path.join(
            args.report_dir, args.system_feat_subcombs)
    else:
        feat_subcombs_path_system = None
    if args.baseline_feat_subcombs:
        feat_subcombs_path_baseline = os.path.join(
            args.report_dir, args.baseline_feat_subcombs)
    else:
        feat_subcombs_path_baseline = None
    
    SYSTEM_INFO = [
         SystemInfoUnsharedFeats(feat_subcombs_path=feat_subcombs_path_system,
                                 combination=(0, '≥1'),
                                 system_='system',
                                 MATRICES=MATRICES,
                                 feat_value_pairs=feat_value_pairs,
                                 feat2index=feat2index),
         SystemInfoUnsharedFeats(feat_subcombs_path=feat_subcombs_path_baseline,
                                 combination=('≥1', 0),
                                 system_='baseline',
                                 MATRICES=MATRICES,
                                 feat_value_pairs=feat_value_pairs,
                                 feat2index=feat2index)
    ]
    header = OrderedDict([
        ((f'# diac ({args.baseline_name})', 13), ('num_diac_baseline_str', None)),
        ((f'# diac ({args.system_name})', 13), ('num_diac_system_str', None)),
        (('Slots', None), ('num_slots_str', None)),
        (('Lemmas', None), ('num_lemmas_valid_str', None)),
        (('Feats', None), ('num_feats_valid_str', None)),
        (('Example', 60), ('example_str', 60)),
        (('Unshared feats', 50), ('feat_value_pairs_str', 50))
    ])
    rows = []
    for system_info in SYSTEM_INFO:
        fns = [(system_info.compute_values_group_1_str, None),
               (system_info.compute_values_group_2_str, None),
               (system_info.compute_values_all_str, system_info.unshared_mask),
               (system_info.compute_values_0_0_str, ~system_info.unshared_mask)]
        for fn, mask in fns:
            fn()
            if mask is not None:
                setattr(system_info, 'example_str', generate_examples(
                    system_info.index2lemmas_pos, system_info.index2analysis, mask))
            else:
                setattr(system_info, 'example_str', '-')
            rows.append([getattr(system_info, var[0]) for var in header.values()])
        del system_info.unshared_mask
    
    print(tabulate(rows, tablefmt='fancy_grid', headers=[h[0] for h in header],
                   maxcolwidths=[v[1] for v in header.values()],
                   maxheadercolwidths=[h[1] for h in header]))
    print()

    def section_unshared_group1_stats():
        print(bold(underline(f'Breakdown of the individually unseen feat:value pairs in the x-0 and 0-x cases (GROUP 1)')))
        notes = color(f"""
        Notes:
        - The below table breaks down the first group of unshared features (described above).
        - Feats column below is used differently than previous Feats columns and is used to specify the number of feature combinations in which
        the respective sub-combination happens. There IS overlap between feat:value pairs and the total should NOT sum to the total number of feature combinations
        in group 1 (contrary to next section). The percentages in this column do sum to 100%.
        - Percentages here for Slots and Feats are based on proportion of the total GROUP 1 counts.
        """, 'warning')
        print(notes)
        print()
        rows = []
        for system_info in SYSTEM_INFO:
            rows_ = []
            for specific_feat_comb in system_info.group_1_pairs:
                row = SpecificFeatStatsRow(specific_feat_comb, system_info)
                row.example_str = generate_examples(
                    system_info.index2lemmas_pos, system_info.index2analysis,
                    match_comb_indexes=row.match_comb_indexes)
                rows_.append(row)
                
            rows += sorted(rows_, key=lambda row: row.num_slots_sum, reverse=True)

        rows = [[getattr(row, v[0]) for v in header.values()] for row in rows]
        print(tabulate(rows, tablefmt='fancy_grid', headers=[h[0] for h in header],
                       maxcolwidths=[v[1] for v in header.values()],
                       maxheadercolwidths=[h[1] for h in header]))
        print()


    def section_unshared_group2_stats():
        print(bold(underline(f'Breakdown of the (mappable, invalid, or missing) sub-combinations in the x-0 and 0-x cases (GROUP 2)')))
        notes = color(f"""
        Notes:
        - The below table breaks down the second group of unshared features (described above).
        - When all feature combinations that contain one of the unshared feat:value pairs are eliminated, we are left with three groups of feature sub-combinations:
            1. ones which can be mappable to feature combinations in the opposite system
            2. ones which should just not exist
            3. ones which are missing in the opposite system
        - Every single one of these feature combinations is accounted for automatically by elimination via human identification of the culprit feat:value
        pair(s) which make that feature combination not appear in the opposite system.
        - Feats column below is used differently than previous Feats columns and is used to specify the number of feature combinations in which
        the respective sub-combination happens. There is NO overlap between sub-combinations and the total SHOULD sum to the total number of feature combinations
        in group 2 (contrary to previous section).
        """, 'warning')
        print(notes)
        header[('Explanation', 60)] = ('explanation', 60)

        rows = []
        for system_info in SYSTEM_INFO:
            num_slots_total = system_info.group_2_slots_sum
            rows_ = []
            num_slots_total_ = 0
            for group in system_info.group_2_categorization:
                if group != 'UNSORTED (Total)':
                    feats_dict = {f_v.split(':')[0]: f_v.split(':')[1] for f_v in group.split()}
                    for i, info in enumerate(system_info.feat_subcombs):
                        if feats_dict == info['feats_dict']:
                            row_info = dict(feats_dict=feats_dict,
                                            explanation=system_info.feat_subcombs[i]['explanation'])
                            break
                    else:
                        row_info = dict(feats_dict=feats_dict, explanation='')
                else:
                    row_info = dict(feats_dict='UNSORTED (Total)', explanation='')

                row = SpecificFeatStatsRow(row_info, system_info)
                row.example_str = generate_examples(
                    system_info.index2lemmas_pos, system_info.index2analysis,
                    match_comb_indexes=row.match_comb_indexes)
                num_slots_total_ += row.num_slots_sum
                rows_.append(row)
            
            assert num_slots_total_ == num_slots_total
            rows_ += SpecificFeatStatsRow.compute_total_row_group_2(rows_)
            rows_ = sorted(
                rows_, key=lambda row: row.num_slots_sum if 'Total' not in row.feat_value else (
                num_slots_total - row.num_slots_sum), reverse=True)
            rows_ = [[getattr(row, v[0]) for v in header.values()] for row in rows_]
            rows += rows_

        print(tabulate(rows, tablefmt='fancy_grid', headers=[h[0] for h in header],
                       maxcolwidths=[v[1] for v in header.values()],
                       maxheadercolwidths=[h[1] for h in header]))
        print()

    header = OrderedDict([
        ((f'# diac ({args.baseline_name})', 8), ('num_diac_baseline', 8)),
        ((f'# diac ({args.system_name})', 7), ('num_diac_system', 7)),
        (('Features', None), ('feat_value', None)),
        (('Slots', None), ('num_slots_str', None)),
        (('Lemmas', None), ('num_lemmas_valid_str', None)),
        (('Feats', None), ('num_feats_valid_str', None)),
        (('Example', 60), ('example_str', 60))
    ])
    section_unshared_group1_stats()
    section_unshared_group2_stats()


db_baseline_gen = MorphologyDB(args.db_baseline, flags='g')
generator_baseline = Generator(db_baseline_gen)

db_system_gen = MorphologyDB(args.db_system, flags='g')
generator_system = Generator(db_system_gen)

POS = eval_utils.get_pos(args.pos, db_baseline_gen, db_system_gen)
eval_utils.harmonize_defaults(generator_baseline._db, generator_system._db, POS)

try:
    results_debug_eval = eval_utils.load_results_debug_eval(args.report_dir)
except:
    pass

try:
    MATRICES = eval_utils.load_matrices(args.report_dir)

    info = MATRICES['intersection']
    diac_mat_baseline_inter = info['diac_mat_baseline']
    diac_mat_system_inter = info['diac_mat_system']
    system_only_mat_inter = info['system_only_mat']
    baseline_only_mat_inter = info['baseline_only_mat']
    no_intersect_mat_inter = info['no_intersect_mat']
    index2analysis_inter, analysis2index_inter = info['index2analysis'], info['analysis2index']
    index2lemmas_pos_inter = {index: lemma_pos for index, lemma_pos in info['lemmas_pos']}
    failed = info['failed'] if 'failed' in info else None

    info_bo = MATRICES['baseline_only']
    diac_mat_baseline_only = info_bo['diac_mat_baseline']
    index2analysis_bo, analysis2index_bo = info_bo['index2analysis'], info_bo['analysis2index']
    index2lemmas_pos_bo = {index: lemma_pos for index, lemma_pos in info_bo['lemmas_pos']}
    MATRICES['baseline_only']['index2lemmas_pos'] = index2lemmas_pos_bo
    failed_bo = info_bo['failed'] if 'failed' in info_bo else None

    info_so = MATRICES['system_only']
    diac_mat_system_only = info_so['diac_mat_system']
    index2analysis_so, analysis2index_so = info_so['index2analysis'], info_so['analysis2index']
    index2lemmas_pos_so = {index: lemma_pos for index, lemma_pos in info_so['lemmas_pos']}
    MATRICES['system_only']['index2lemmas_pos'] = index2lemmas_pos_so
    failed_so = info_so['failed'] if 'failed' in info_so else None
    POS2FEAT_VALUE_PAIRS = eval_utils.load_pos2feat_value_pairs(args.report_dir)
except:
    print('WARNING: Script will not work as intended and might raise errors if all three splits are not present.')

db_baseline = MorphologyDB(args.db_baseline)
db_system = MorphologyDB(args.db_system)


if __name__ == '__main__':
    mask_not_equal_0_baseline = diac_mat_baseline_inter != 0
    mask_not_equal_0_system = diac_mat_system_inter != 0
    num_valid_feats = int(np.sum(np.any(diac_mat_system_inter|diac_mat_baseline_inter, axis=0)))
    num_valid_lemmas = int(np.sum(np.any(diac_mat_system_inter|diac_mat_baseline_inter, axis=1)))

    section_title()
    section_lemma_overlap()
    del db_baseline, db_system
    section_high_level_overlap_stats()
    section_shared_combinations_high_level_stats()
    section_x_y_breakdown()
    section_unshared_combinations_high_level_stats()