import sys
import os
import argparse
import json
from tqdm import tqdm
import pickle
import random
import multiprocessing
import cProfile, pstats
import itertools
from types import ModuleType, FunctionType
import numpy as np

import eval_utils

# Custom objects know their class.
# Function objects seem to know way too much, including modules.
# Exclude modules as well.
BLACKLIST = type, ModuleType, FunctionType

feat2index = {k: i for i, k in enumerate(eval_utils.essential_keys_no_lex_pos)}

parser = argparse.ArgumentParser()
parser.add_argument("-config_file", default='configs/config.json',
                    type=str, help="Config file specifying which sheets to use.")
parser.add_argument("-db_dir", default='databases',
                        type=str, help="Path of the directory to load the DB from.")
parser.add_argument("-msa_config_name", default='all_aspects_msa',
                    type=str, help="Config name which specifies the path of the MSA Camel DB.")
parser.add_argument("-msa_baseline_db", default='eval_files/calima-msa-s31_0.4.2.utf8.db',
                    type=str, help="Path of the MSA baseline DB file we will be comparing against.")
parser.add_argument("-report_dir", default='eval_files/report_default',
                    type=str, help="Paths of the directory containing partial reports generated by the full generative evaluation.")
parser.add_argument("-multiprocessing", default=False, action='store_true',
                    help="Whether or not to use multiprocessing.")
parser.add_argument("-test_mode", default=False, action='store_true',
                    help="Only test mode.")
parser.add_argument("-profiling", default=False, action='store_true',
                    help="Run profiling.")
parser.add_argument("-sample", default=False, action='store_true',
                    help="Take sample of lemmas for validating and not all.")
parser.add_argument("-feats_debug", default=[], action="extend", nargs="+",
                    type=str, help="Features to debug.")
parser.add_argument("-feat_combs", default=1,
                    type=int, help="Number of features to use in different combinations for debugging.")
parser.add_argument("-n_cpu", default=8,
                    type=int, help="Number of cores to use.")
parser.add_argument("-n", default=100,
                    type=int, help="Number of inputs to the two compared systems.")
parser.add_argument("-camel_tools", default='local', choices=['local', 'official'],
                    type=str, help="Path of the directory containing the camel_tools modules.")
args = parser.parse_args()

random.seed(42)

with open(args.config_file) as f:
    config = json.load(f)
    config_local = config['local']
    config_msa = config_local[args.msa_config_name]

if args.camel_tools == 'local':
    camel_tools_dir = config['global']['camel_tools']
    sys.path.insert(0, camel_tools_dir)

from camel_tools.morphology.database import MorphologyDB
from camel_tools.morphology.generator import Generator
from camel_tools.utils.charmap import CharMapper

bw2ar = CharMapper.builtin_mapper('bw2ar')
ar2bw = CharMapper.builtin_mapper('ar2bw')

FEATS_DEBUG = [set(itertools.combinations(args.feats_debug, i))
               for i in range(1, args.feat_combs + 1)]

def _generate_test_keys(analysis_key):
    test_keys = []
    for FEATS_DEBUG_ in FEATS_DEBUG:
        for feats in FEATS_DEBUG_:
            # Products involving values in different positions from the same feature should not be allowed
            if len(feats) > 1 and len(set([feat.split(':')[0] for feat in feats])) != len(feats):
                continue
            # Fully specified, e.g., gen:m>u (both source and target are specified)
            # Partially specified, e.g., prc0:0 (only target value is specified)
            # Unspecified, e.g., vox (neither source not target are specified, all values allowed)
            specification = {}
            for feat in feats:
                if ':' in feat:
                    specification.setdefault(feat.split(':')[0], []).append(feat)
                else:
                    for v in DEFINES[feat]:
                        specification.setdefault(feat, []).append(f'{feat}:{v}')
            
            values_combinations = list(itertools.product(*specification.values()))
            for values_comb in values_combinations:
                test_key_ = list(analysis_key)
                test_key_.insert(0, [])
                for feat_value in values_comb:
                    feat, value = feat_value.split(':')
                    index = feat2index[feat] + 1
                    if '>' in value:
                        value_before, value_after = value.split('>')
                        if value_before != test_key_[index]:
                            break
                        diff, value = f'{feat}:{value}', value_after
                    else:
                        diff = f'{feat}:{test_key_[index]}>{value}'
                    if len(set(diff.split(':')[1].split('>'))) == 1:
                        break
                    test_key_[0].append(diff)
                    test_key_[index] = value
                else:
                    test_key_[0] = '+'.join(test_key_[0])
                    test_key = tuple(test_key_)
                    if test_key[1:] != analysis_key:
                        test_keys.append(test_key)
    return tuple(sorted(set(test_keys), key=lambda x: len(x[0].split('+'))))


def _validate_analysis(analysis_key, index):
    test_keys = _generate_test_keys(tuple(analysis_key))
    explanation = set()
    for key_feats in test_keys:
        key_feats_ = '+'.join(key_feats[1:])
        if key_feats_ not in eval_with_clitics['both']:
            if key_feats_ in eval_with_clitics['baseline']:
                if accuracy_matrix_baseline[index][analysis2index[key_feats_]]:
                    explanation.add(('baseline', key_feats[0]))
            elif key_feats_ in eval_with_clitics['camel']:
                if accuracy_matrix_camel[index][analysis2index[key_feats_]]:
                    return 'camel', key_feats[0]
            else:
                explanation.add(('invalid', key_feats[0]))
        else:
            if accuracy_matrix_baseline is not None:
                analyses_exist_for_lemma = bool(
                    accuracy_matrix_baseline[index][analysis2index[key_feats_]] and
                    accuracy_matrix_camel[index][analysis2index[key_feats_]])
            else:
                analyses_exist_for_lemma = True
            
            if analyses_exist_for_lemma:
                if all(index not in eval_with_clitics[x][key_feats_]
                       if key_feats_ in eval_with_clitics[x] else True
                       for x in ['camel', 'baseline']):
                    return 'matching', key_feats[0]
    
    for e in explanation:
        if e[0] == 'baseline':
            return 'baseline', e[1]
    
    return '', 'invalid'

def debug_eval(analysis_key_indexes):
    analysis_key, indexes = analysis_key_indexes
    analysis_key_split = analysis_key.split('+')
    if args.sample:
        indexes_ = random.sample(indexes, min(10, len(indexes)))
    else:
        indexes_ = indexes
    results = []
    for index in indexes_:
        diff = _validate_analysis(analysis_key_split, index)
        results.append((diff, analysis_key, index))
    return results

path_db_baseline = args.msa_baseline_db

db_baseline_gen = MorphologyDB(path_db_baseline, flags='g')
generator_baseline = Generator(db_baseline_gen)

path_db_camel = os.path.join(args.db_dir, config_msa['db'])
db_camel_gen = MorphologyDB(path_db_camel, flags='g')
generator_camel = Generator(db_camel_gen)

DEFINES = {k: v if v is None else [vv for vv in v if vv != 'na']
           for k, v in generator_baseline._db.defines.items()}

index2lemmas_pos = eval_utils.load_index2lemmas_pos(args.report_dir)
eval_with_clitics = eval_utils.load_full_report(args.report_dir)

info = eval_utils.load_accuracy_matrices(args.report_dir)
accuracy_matrix_baseline = info['accuracy_matrix_baseline']
accuracy_matrix_camel = info['accuracy_matrix_camel']
index2analysis, analysis2index = info['index2analysis'], info['analysis2index']

if args.profiling:
    profiler = cProfile.Profile()
    profiler.enable()

if __name__ == "__main__":
    iterable = eval_with_clitics['baseline'].items()
    if args.multiprocessing:
        with multiprocessing.Pool(args.n_cpu) as p:
            results_debug_eval = list(
                tqdm(p.imap(debug_eval, iterable), total=len(iterable), smoothing=0.2))
    else:
        results_debug_eval = []
        for analysis_key_indexes in tqdm(iterable, smoothing=0.2):
            results_debug_eval.append(debug_eval(analysis_key_indexes))

    if args.profiling:
        profiler.disable()
        stats = pstats.Stats(profiler).sort_stats('cumtime')
        stats.print_stats()
    
    eval_with_clitics_ = {}
    for results_ in results_debug_eval:
        for diff, analysis_key, index in results_:
            if diff[0]:
                eval_with_clitics_.setdefault('baseline', {}).setdefault(
                    diff[0], {}).setdefault(diff[1], {}).setdefault(analysis_key, []).append(index)
            else:
                eval_with_clitics_.setdefault('baseline', {}).setdefault(
                    'not_validated', {}).setdefault(diff[1], {}).setdefault(analysis_key, []).append(index)

    with open(os.path.join(args.report_dir, 'results_debug_eval.pkl'), 'wb') as f:
        pickle.dump(eval_with_clitics_, f)
    
    print('Done.')