# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['thresult']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'thresult',
    'version': '0.9.25',
    'description': 'Tangled Result',
    'long_description': '[![Build][build-image]]()\n[![Status][status-image]][pypi-project-url]\n[![Stable Version][stable-ver-image]][pypi-project-url]\n[![Coverage][coverage-image]]()\n[![Python][python-ver-image]][pypi-project-url]\n[![License][bsd3-image]][bsd3-url]\n\n\n# thresult\n\n*Python* `Result` library for handling returned values (`Result`, `Ok`, `Err`) from functions/methods and handling errors. It is *error handling* library which is alternative to `try/except` style of programming.\n\nIt is inspired by great *Rust* `Result`, `Ok`, `Err` types.\n\n\n## Installation\n\n```bash\npip install thresult\n```\n\n\n## Simple Usage\n\n### Traditional `try-except` example\n\n```python\ndef div(x: float, y: float) -> float:\n    z: float = x / y\n    return z\n\n\nz0: float = div(1.0, 2.0) # 0.5\nz1: float = div(1.0, 0.0) # raises "ZeroDivisionError: division by zero" exception\n```\n\n\n### Unwrapping `Result` as Context Manager\n\n```python\nfrom thresult import Result\n\n\n@Result\ndef div(x: float, y: float) -> float:\n    z: float = x / y\n    return z\n\n\ntry:\n    with div(1.0, 0.0) as z:\n        # unreachable\n        r: float = z\nexcept ZeroDivisionError as e:\n    # exception happened\n    pass\n```\n\n\n### Unwrapping `Result` in `unwrap` Context Manager\n\n```python\nfrom thresult import Result, unwrap\n\n\n@Result\ndef f(x: float, y: float) -> float:\n    z: float = x / y\n    return z\n\n\ntry:\n    with unwrap():\n        # unreachable\n        r: float = f(1, 0)\nexcept ZeroDivisionError as e:\n    # exception happened\n    raise e\n```\n\n\n### Pattern-matching `Result` using `match-case`\n```python\nfrom thresult import Result, Ok, Err\n\n\n@Result\ndef div(x: float, y: float) -> float:\n    # can raise "ZeroDivisionError: division by zero" exception\n    z: float = x / y\n    return z\n\n\nr: Result = div(1.0, 2.0) # Ok[float](0.5)\n\nmatch r:\n    case Ok(v):\n        assert v == 0.5\n    case Err(e):\n        # unreachable\n        # "ZeroDivisionError: division by zero"\n        assert isinstance(e, ZeroDivisionError)\n```\n\n\n### In-place pattern-matching `Result` using `walrus operator` and `match-case`\n\n```python\n@Result\ndef div(x: float, y: float) -> float:\n    # can raise "ZeroDivisionError: division by zero" exception\n    z: float = x / y\n    return z\n\n\nmatch r := div(1.0, 2.0): # Ok[float](0.5)\n    case Ok(v):\n        assert v == 0.5\n    case Err(e):\n        # unreachable\n        # "ZeroDivisionError: division by zero"\n        assert isinstance(e, ZeroDivisionError)\n```\n\n\n## Advanced Usage\n\n\n### Unwrapping `Result[float, ZeroDivisionError]` as Context Manager\n\n```python\nfrom thresult import Result, Ok, Err\n\n\n@Result[float, ZeroDivisionError]\ndef div(x: float, y: float) -> float:\n    z: float = x / y\n    return z\n\n\ntry:\n    with div(1.0, 0.0) as z:\n        # unreachable \n        pass\nexcept ZeroDivisionError as e:\n    # exception happened\n    pass\n```\n\n\n### Manually create Result value, and Structural Pattern Matching\n\n```python\nfrom thresult import Result, Ok, Err\n\n\ndef div(x: float, y: float) -> Result[float, ZeroDivisionError]:\n    res: Result[float, ZeroDivisionError]\n\n    try:\n        # can raise "ZeroDivisionError: division by zero" exception\n        z: float = x / y\n        res = Ok[float](z)\n    except ZeroDivisionError as e:\n        res = Err[ZeroDivisionError](e)\n\n    return res\n\n\nr0: Result = div(1.0, 2.0) # Ok\nr1: Result = div(1.0, 0.0) # Err\n\nmatch r0:\n    case Ok(v):\n        print(\'Ok, value:\', v)\n    case Err(e):\n        print(\'Err, error:\', e) # "ZeroDivisionError: division by zero"\n\nmatch r1:\n    case Ok(v):\n        print(\'Ok, value:\', v)\n    case Err(e):\n        print(\'Err, error:\', e) # "ZeroDivisionError: division by zero"\n\nz0: float = r0.unwrap() # 0.5\nz1: float = r1.unwrap_or(float(\'inf\')) # inf\nz1: float = r1.unwrap() # raises "ZeroDivisionError: division by zero" exception\n```\n\n\n### Decorate function with Result, and Structural Pattern Matching\n\n```python\nfrom thresult import Result, Ok, Err\n\n\n@Result[float, ZeroDivisionError]\ndef div(x: float, y: float) -> float:\n    # can raise "ZeroDivisionError: division by zero" exception\n    z: float = x / y\n    return z\n\n\nr0: Result = div(1.0, 2.0) # Ok\nr1: Result = div(1.0, 0.0) # Err\n\nmatch r0:\n    case Ok(v):\n        print(\'Ok, value:\', v)\n    case Err(e):\n        print(\'Err, error:\', e) # "ZeroDivisionError: division by zero"\n\nmatch r1:\n    case Ok(v):\n        print(\'Ok, value:\', v)\n    case Err(e):\n        print(\'Err, error:\', e) # "ZeroDivisionError: division by zero"\n\nz0: float = r0.unwrap() # 0.5\nz1: float = r1.unwrap_or(float(\'inf\')) # inf\nz1: float = r1.unwrap() # raises "ZeroDivisionError: division by zero" exception\n```\n\n\n## Run / Develop Cycle\n```bash\ndocker run --rm -it -v $PWD/thresult:/code -w /code python:3.11-alpine python -B result.py\n```\n\n\n## Testing\n\n```bash\ndocker-compose build thresult-test ; docker-compose run --rm -v $PWD:/test thresult-test\n```\n\n\n## Coverage\n\n```bash\ndocker-compose build thresult-coverage ; docker-compose run --rm -v $PWD:/test thresult-coverage\n```\n\n\n## Building\n\n```bash\ndocker-compose build thresult-build ; docker-compose run --rm thresult-build\n```\n\n\n## Licensing\n\n`thresult` is licensed under the BSD 3 license.\n\nCheck the [LICENSE](https://opensource.org/licenses/BSD-3-Clause) for details.\n\n\n<!-- Badges -->\n[bsd3-image]: https://img.shields.io/badge/License-BSD_3--Clause-blue.svg\n[bsd3-url]: https://opensource.org/licenses/BSD-3-Clause\n[build-image]: https://img.shields.io/gitlab/pipeline-status/tangledlabs/thresult?branch=main\n[coverage-image]: https://img.shields.io/gitlab/pipeline-coverage/tangledlabs/thresult?branch=main\n\n[pypi-project-url]: https://pypi.org/project/thresult/\n[stable-ver-image]: https://img.shields.io/pypi/v/thresult?label=stable\n[python-ver-image]: https://img.shields.io/pypi/pyversions/thresult.svg?logo=python&logoColor=FBE072\n[status-image]: https://img.shields.io/pypi/status/thresult.svg\n',
    'author': 'Tangled',
    'author_email': 'info@tangledgroup.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://gitlab.com/tangledlabs/thresult',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.10,<4.0',
}


setup(**setup_kwargs)
