# Copyright 2016 Criteo
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Expose python thread names."""

# From: https://bugs.python.org/issue15500
import ctypes
import ctypes.util
import threading

import six


def set_thread_names():
    """Expose python thread names."""
    libpthread_path = ctypes.util.find_library("pthread")
    if not libpthread_path:
        return
    libpthread = ctypes.CDLL(libpthread_path)
    if hasattr(libpthread, "pthread_setname_np"):
        pthread_setname_np = libpthread.pthread_setname_np
        pthread_setname_np.argtypes = [ctypes.c_void_p, ctypes.c_char_p]
        pthread_setname_np.restype = ctypes.c_int
        orig_start = threading.Thread.start

        def new_start(self):
            orig_start(self)
            try:
                name = self.name
                if not name or name.startswith("Thread-"):
                    name = self.__class__.__name__
                    if name == "Thread":
                        name = self.name
                if name:
                    if isinstance(name, six.text_type):
                        name = name.encode("ascii", "replace")
                    ident = getattr(self, "ident", None)
                    if ident is not None:
                        pthread_setname_np(ident, name[:15])
            except Exception:
                pass  # Don't care about failure to set name

        threading.Thread.start = new_start
