"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const assert_1 = require("@aws-cdk/assert");
const lambda = require("@aws-cdk/aws-lambda");
const kinesis = require("@aws-cdk/aws-kinesis");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Pattern minimal deployment
// --------------------------------------------------------------
test('Pattern minimal deployment', () => {
    // Initial setup
    const stack = new core_1.Stack();
    const props = {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        }
    };
    new lib_1.KinesisStreamsToLambda(stack, 'test-kinesis-streams-lambda', props);
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Test getter methods
// --------------------------------------------------------------
test('Test properties', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        },
        kinesisEventSourceProps: {
            startingPosition: lambda.StartingPosition.TRIM_HORIZON,
            batchSize: 1
        }
    };
    const app = new lib_1.KinesisStreamsToLambda(stack, 'test-kinesis-streams-lambda', props);
    // Assertion 1
    expect(app.lambdaFunction !== null);
    // Assertion 2
    expect(app.kinesisStream !== null);
});
// --------------------------------------------------------------
// Test existing resources
// --------------------------------------------------------------
test('Test existing resources', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const fn = new lambda.Function(stack, 'test-fn', {
        runtime: lambda.Runtime.NODEJS_10_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    });
    const stream = new kinesis.Stream(stack, 'test-stream', {
        streamName: 'existing-stream',
        shardCount: 5,
        retentionPeriod: core_1.Duration.hours(48),
        encryption: kinesis.StreamEncryption.UNENCRYPTED
    });
    new lib_1.KinesisStreamsToLambda(stack, 'test-kinesis-streams-lambda', {
        existingLambdaObj: fn,
        existingStreamObj: stream,
        // These properties will be ignored as existing objects were provided
        lambdaFunctionProps: {
            runtime: lambda.Runtime.PYTHON_3_8,
            handler: 'lambda_function.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        },
        kinesisStreamProps: {
            streamName: 'other-name-stream',
            shardCount: 1,
            retentionPeriod: core_1.Duration.hours(24)
        }
    });
    // Assertions
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    expect(stack).toHaveResource('AWS::Kinesis::Stream', {
        Name: 'existing-stream',
        ShardCount: 5,
        RetentionPeriodHours: 48,
    });
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Handler: 'index.handler',
        Runtime: 'nodejs10.x',
    });
});
test('test sqsDlqQueueProps override', () => {
    const stack = new core_1.Stack();
    const props = {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        },
        sqsDlqQueueProps: {
            queueName: 'hello-world',
            visibilityTimeout: core_1.Duration.seconds(50)
        }
    };
    new lib_1.KinesisStreamsToLambda(stack, 'test-kinesis-streams-lambda', props);
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "hello-world",
        VisibilityTimeout: 50
    });
});
//# sourceMappingURL=data:application/json;base64,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