"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Database = exports.WordPress = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const efs = require("@aws-cdk/aws-efs");
const logs = require("@aws-cdk/aws-logs");
const rds = require("@aws-cdk/aws-rds");
const cdk = require("@aws-cdk/core");
const patterns = require("./index");
class WordPress extends cdk.Construct {
    constructor(scope, id, props = {}) {
        var _a, _b;
        super(scope, id);
        this.vpc = (_a = props.vpc) !== null && _a !== void 0 ? _a : getOrCreateVpc(this);
        this.db = this.addDatabase({
            vpc: this.vpc,
            databaseSubnets: props.databaseSubnets,
            instanceType: props.databaseInstanceType,
            instanceEngine: props.instanceEngine,
            clusterEngine: props.clusterEngine,
            auroraServerless: props.auroraServerless,
            singleDbInstance: props.singleDbInstance,
            backupRetention: props.backupRetention,
        });
        const logGroup = new logs.LogGroup(this, 'LogGroup', {
            retention: logs.RetentionDays.ONE_MONTH,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        const task = new ecs.FargateTaskDefinition(this, 'Task', {
            cpu: 256,
            memoryLimitMiB: 512,
        });
        task.addContainer('wordpress', {
            image: ecs.ContainerImage.fromRegistry('wordpress:latest'),
            portMappings: [{ containerPort: 80 }],
            environment: {
                WORDPRESS_DB_NAME: 'wordpress',
            },
            logging: new ecs.AwsLogDriver({
                streamPrefix: 'wordpress-fargate',
                logGroup,
            }),
            secrets: {
                WORDPRESS_DB_HOST: ecs.Secret.fromSecretsManager(this.db.secret, 'host'),
                WORDPRESS_DB_USER: ecs.Secret.fromSecretsManager(this.db.secret, 'username'),
                WORDPRESS_DB_PASSWORD: ecs.Secret.fromSecretsManager(this.db.secret, 'password'),
            },
        });
        const healthCheck = {
            path: '/wp-includes/images/blank.gif',
            interval: cdk.Duration.minutes(1),
        };
        this.svc = new patterns.DualAlbFargateService(this, 'ALBFargateService', {
            vpc: this.vpc,
            spot: props.spot,
            enableExecuteCommand: props.enableExecuteCommand,
            tasks: props.serviceProps ? [props.serviceProps] : [
                {
                    listenerPort: 80,
                    accessibility: patterns.LoadBalancerAccessibility.EXTERNAL_ONLY,
                    task,
                    healthCheck,
                },
            ],
            route53Ops: {
                enableLoadBalancerAlias: false,
            },
        });
        // EFS volume
        const filesystem = new efs.FileSystem(this, 'FileSystem', {
            vpc: this.vpc,
            encrypted: true,
        });
        const volumeName = 'efs';
        this.svc.service[0].taskDefinition.addVolume({
            name: volumeName,
            efsVolumeConfiguration: {
                fileSystemId: filesystem.fileSystemId,
            },
        });
        (_b = this.svc.service[0].taskDefinition.defaultContainer) === null || _b === void 0 ? void 0 : _b.addMountPoints({
            containerPath: '/var/www/html',
            readOnly: false,
            sourceVolume: volumeName,
        });
        filesystem.connections.allowFrom(new ec2.Connections({ securityGroups: this.svc.service[0].connections.securityGroups }), ec2.Port.tcp(2049), 'allow wordpress to connect efs');
        this.db.connections.allowFrom(this.svc.service[0], this.db.connections.defaultPort, `allow ${this.svc.service[0].serviceName} to connect db`);
    }
    addDatabase(props) {
        return new Database(this, 'Database', {
            ...props,
        });
    }
}
exports.WordPress = WordPress;
/**
 * Represents the database instance or database cluster
 */
class Database extends cdk.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this._mysqlListenerPort = 3306;
        this.vpc = props.vpc;
        const config = props.auroraServerless ? this._createServerlessCluster(props)
            : props.singleDbInstance ? this._createRdsInstance(props) : this._createRdsCluster(props);
        this.secret = config.secret;
        // allow internally from the same security group
        config.connections.allowInternally(ec2.Port.tcp(this._mysqlListenerPort));
        // allow from the fargate service or the whole vpc cidr
        config.connections.allowFrom((_a = props.allowFrom) !== null && _a !== void 0 ? _a : ec2.Peer.ipv4(props.vpc.vpcCidrBlock), ec2.Port.tcp(this._mysqlListenerPort));
        this.clusterEndpointHostname = config.endpoint;
        this.clusterIdentifier = config.identifier;
        this.connections = config.connections;
        printOutput(this, 'DBSecretArn', config.secret.secretArn);
        printOutput(this, 'clusterEndpointHostname', this.clusterEndpointHostname);
        printOutput(this, 'clusterIdentifier', this.clusterIdentifier);
    }
    _createRdsInstance(props) {
        var _a, _b, _c;
        const dbInstance = new rds.DatabaseInstance(this, 'DBInstance', {
            vpc: props.vpc,
            vpcSubnets: props.databaseSubnets,
            engine: (_a = props.instanceEngine) !== null && _a !== void 0 ? _a : rds.DatabaseInstanceEngine.mysql({
                version: rds.MysqlEngineVersion.VER_8_0_23,
            }),
            storageEncrypted: true,
            backupRetention: (_b = props.backupRetention) !== null && _b !== void 0 ? _b : cdk.Duration.days(7),
            credentials: rds.Credentials.fromGeneratedSecret('admin'),
            instanceType: (_c = props.instanceType) !== null && _c !== void 0 ? _c : new ec2.InstanceType('r5.large'),
            parameterGroup: rds.ParameterGroup.fromParameterGroupName(this, 'ParameterGroup', 'default.mysql8.0'),
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        return {
            connections: dbInstance.connections,
            endpoint: dbInstance.dbInstanceEndpointAddress,
            identifier: dbInstance.instanceIdentifier,
            secret: dbInstance.secret,
        };
    }
    // create a RDS for MySQL DB cluster
    _createRdsCluster(props) {
        var _a, _b, _c;
        const dbCluster = new rds.DatabaseCluster(this, 'DBCluster', {
            engine: (_a = props.clusterEngine) !== null && _a !== void 0 ? _a : rds.DatabaseClusterEngine.auroraMysql({
                version: rds.AuroraMysqlEngineVersion.VER_2_09_1,
            }),
            credentials: rds.Credentials.fromGeneratedSecret('admin'),
            defaultDatabaseName: 'wordpress',
            instanceProps: {
                vpc: props.vpc,
                vpcSubnets: props.databaseSubnets,
                instanceType: (_b = props.instanceType) !== null && _b !== void 0 ? _b : new ec2.InstanceType('r5.large'),
            },
            parameterGroup: rds.ParameterGroup.fromParameterGroupName(this, 'ParameterGroup', 'default.aurora-mysql5.7'),
            backup: {
                retention: (_c = props.backupRetention) !== null && _c !== void 0 ? _c : cdk.Duration.days(7),
            },
            storageEncrypted: true,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        return {
            connections: dbCluster.connections,
            endpoint: dbCluster.clusterEndpoint.hostname,
            identifier: dbCluster.clusterIdentifier,
            secret: dbCluster.secret,
        };
    }
    _createServerlessCluster(props) {
        var _a;
        const dbCluster = new rds.ServerlessCluster(this, 'AuroraServerlessCluster', {
            engine: rds.DatabaseClusterEngine.AURORA_MYSQL,
            vpc: props.vpc,
            vpcSubnets: props.databaseSubnets,
            credentials: rds.Credentials.fromGeneratedSecret('admin'),
            backupRetention: (_a = props.backupRetention) !== null && _a !== void 0 ? _a : cdk.Duration.days(7),
            removalPolicy: cdk.RemovalPolicy.DESTROY,
            parameterGroup: rds.ParameterGroup.fromParameterGroupName(this, 'ParameterGroup', 'default.aurora-mysql5.7'),
            defaultDatabaseName: 'wordpress',
        });
        return {
            connections: dbCluster.connections,
            endpoint: dbCluster.clusterEndpoint.hostname,
            identifier: dbCluster.clusterIdentifier,
            secret: dbCluster.secret,
        };
    }
}
exports.Database = Database;
function getOrCreateVpc(scope) {
    // use an existing vpc or create a new one
    return scope.node.tryGetContext('use_default_vpc') === '1'
        || process.env.CDK_USE_DEFAULT_VPC === '1' ? ec2.Vpc.fromLookup(scope, 'Vpc', { isDefault: true }) :
        scope.node.tryGetContext('use_vpc_id') ?
            ec2.Vpc.fromLookup(scope, 'Vpc', { vpcId: scope.node.tryGetContext('use_vpc_id') }) :
            new ec2.Vpc(scope, 'Vpc', { maxAzs: 3, natGateways: 1 });
}
function printOutput(scope, id, key) {
    new cdk.CfnOutput(scope, id, { value: String(key) });
}
//# sourceMappingURL=data:application/json;base64,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