"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const policy_1 = require("./policy");
const rotation_schedule_1 = require("./rotation-schedule");
const secretsmanager = require("./secretsmanager.generated");
/**
 * The common behavior of Secrets. Users should not use this class directly, and instead use ``Secret``.
 */
class SecretBase extends core_1.Resource {
    grantRead(grantee, versionStages) {
        // @see https://docs.aws.amazon.com/fr_fr/secretsmanager/latest/userguide/auth-and-access_identity-based-policies.html
        const result = iam.Grant.addToPrincipal({
            grantee,
            actions: ['secretsmanager:GetSecretValue'],
            resourceArns: [this.secretArn],
            scope: this,
        });
        if (versionStages != null && result.principalStatement) {
            result.principalStatement.addCondition('ForAnyValue:StringEquals', {
                'secretsmanager:VersionStage': versionStages,
            });
        }
        if (this.encryptionKey) {
            // @see https://docs.aws.amazon.com/fr_fr/kms/latest/developerguide/services-secrets-manager.html
            this.encryptionKey.grantDecrypt(new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, grantee.grantPrincipal));
        }
        return result;
    }
    get secretValue() {
        return this.secretValueFromJson('');
    }
    secretValueFromJson(jsonField) {
        return core_1.SecretValue.secretsManager(this.secretArn, { jsonField });
    }
    addRotationSchedule(id, options) {
        return new rotation_schedule_1.RotationSchedule(this, id, {
            secret: this,
            ...options,
        });
    }
    addToResourcePolicy(statement) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new policy_1.ResourcePolicy(this, 'Policy', { secret: this });
        }
        if (this.policy) {
            this.policy.document.addStatements(statement);
        }
    }
    denyAccountRootDelete() {
        this.addToResourcePolicy(new iam.PolicyStatement({
            actions: ['secretsmanager:DeleteSecret'],
            effect: iam.Effect.DENY,
            resources: ['*'],
            principals: [new iam.AccountRootPrincipal()],
        }));
    }
}
/**
 * Creates a new secret in AWS SecretsManager.
 */
class Secret extends SecretBase {
    constructor(scope, id, props = {}) {
        var _a, _b;
        super(scope, id, {
            physicalName: props.secretName,
        });
        this.autoCreatePolicy = true;
        if (props.generateSecretString &&
            (props.generateSecretString.secretStringTemplate || props.generateSecretString.generateStringKey) &&
            !(props.generateSecretString.secretStringTemplate && props.generateSecretString.generateStringKey)) {
            throw new Error('`secretStringTemplate` and `generateStringKey` must be specified together.');
        }
        const resource = new secretsmanager.CfnSecret(this, 'Resource', {
            description: props.description,
            kmsKeyId: props.encryptionKey && props.encryptionKey.keyArn,
            generateSecretString: props.generateSecretString || {},
            name: this.physicalName,
        });
        this.secretArn = this.getResourceArnAttribute(resource.ref, {
            service: 'secretsmanager',
            resource: 'secret',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.encryptionKey = props.encryptionKey;
        // @see https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html#asm-authz
        const principle = new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, new iam.AccountPrincipal(core_1.Stack.of(this).account));
        (_a = this.encryptionKey) === null || _a === void 0 ? void 0 : _a.grantEncryptDecrypt(principle);
        (_b = this.encryptionKey) === null || _b === void 0 ? void 0 : _b.grant(principle, 'kms:CreateGrant', 'kms:DescribeKey');
    }
    static fromSecretArn(scope, id, secretArn) {
        return Secret.fromSecretAttributes(scope, id, { secretArn });
    }
    /**
     * Import an existing secret into the Stack.
     *
     * @param scope the scope of the import.
     * @param id    the ID of the imported Secret in the construct tree.
     * @param attrs the attributes of the imported secret.
     */
    static fromSecretAttributes(scope, id, attrs) {
        class Import extends SecretBase {
            constructor() {
                super(...arguments);
                this.encryptionKey = attrs.encryptionKey;
                this.secretArn = attrs.secretArn;
                this.autoCreatePolicy = false;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds a target attachment to the secret.
     *
     * @returns an AttachedSecret
     *
     * @deprecated use `attach()` instead
     */
    addTargetAttachment(id, options) {
        return new SecretTargetAttachment(this, id, {
            secret: this,
            ...options,
        });
    }
    /**
     * Attach a target to this secret
     *
     * @param target The target to attach
     * @returns An attached secret
     */
    attach(target) {
        const id = 'Attachment';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('Secret is already attached to a target.');
        }
        return new SecretTargetAttachment(this, id, {
            secret: this,
            target,
        });
    }
}
exports.Secret = Secret;
/**
 * The type of service or database that's being associated with the secret.
 */
var AttachmentTargetType;
(function (AttachmentTargetType) {
    /**
     * A database instance
     *
     * @deprecated use RDS_DB_INSTANCE instead
     */
    AttachmentTargetType["INSTANCE"] = "AWS::RDS::DBInstance";
    /**
     * A database cluster
     *
     * @deprecated use RDS_DB_CLUSTER instead
     */
    AttachmentTargetType["CLUSTER"] = "AWS::RDS::DBCluster";
    /**
     * AWS::RDS::DBInstance
     */
    AttachmentTargetType["RDS_DB_INSTANCE"] = "AWS::RDS::DBInstance";
    /**
     * AWS::RDS::DBCluster
     */
    AttachmentTargetType["RDS_DB_CLUSTER"] = "AWS::RDS::DBCluster";
    /**
     * AWS::Redshift::Cluster
     */
    AttachmentTargetType["REDSHIFT_CLUSTER"] = "AWS::Redshift::Cluster";
    /**
     * AWS::DocDB::DBInstance
     */
    AttachmentTargetType["DOCDB_DB_INSTANCE"] = "AWS::DocDB::DBInstance";
    /**
     * AWS::DocDB::DBCluster
     */
    AttachmentTargetType["DOCDB_DB_CLUSTER"] = "AWS::DocDB::DBCluster";
})(AttachmentTargetType = exports.AttachmentTargetType || (exports.AttachmentTargetType = {}));
/**
 * An attached secret.
 */
class SecretTargetAttachment extends SecretBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.autoCreatePolicy = true;
        const attachment = new secretsmanager.CfnSecretTargetAttachment(this, 'Resource', {
            secretId: props.secret.secretArn,
            targetId: props.target.asSecretAttachmentTarget().targetId,
            targetType: props.target.asSecretAttachmentTarget().targetType,
        });
        this.encryptionKey = props.secret.encryptionKey;
        // This allows to reference the secret after attachment (dependency).
        this.secretArn = attachment.ref;
        this.secretTargetAttachmentSecretArn = attachment.ref;
    }
    static fromSecretTargetAttachmentSecretArn(scope, id, secretTargetAttachmentSecretArn) {
        class Import extends SecretBase {
            constructor() {
                super(...arguments);
                this.secretArn = secretTargetAttachmentSecretArn;
                this.secretTargetAttachmentSecretArn = secretTargetAttachmentSecretArn;
                this.autoCreatePolicy = false;
            }
        }
        return new Import(scope, id);
    }
}
exports.SecretTargetAttachment = SecretTargetAttachment;
//# sourceMappingURL=data:application/json;base64,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