import numpy as np
import pywt
from scipy.ndimage import zoom
import matplotlib.pyplot as plt


def calculate_wavelet_dataset(
    dataset, scales=np.arange(1, 33), mother_wavelet:str='mexh', preprocess:bool=True, 
    downsampling_ratio:float=0.25, remove_last_row_column:bool=True):
    """Deploy CWT on dataset, downsample, process and reshape it. 
    Every element or index of the dataset should be a 2D tensor of C x L where
    - **L** is the number of samples (sequence length)
    - **C** is the number of channels (features).
    The dataset can be an N x C x L tensor, where **N** is the batch size.
    
    The function first performs CWT on the data, whose output is an H x L matrix of CWT coefficients for every channel.
    If the preprocessing is active (True), the following take place:
    - The matrix is downsampled by a factor of `downsampling_ratio` (default 0.25) across both dimensions 
      (because we are downsampling a 2D image not a 1D signal)
    - The last row and column of the CWT are ignored to account for baseline drift and motion artifacts, 
      if `remove_last_row_column` is True.
    - The dataset is returned as an N x C x H' x L' tensor (default input format for PyTorch Conv2D), where
        - **N** is the batch size
        - **H** is the height of the image generated by CWT, equal to the number of elements in the scales array.
        - **H'** is the height after downsampling and clipping, if any.
        - **L'** is the width after downsampling and clipping, if any.
    If the dataset is a NumPy array, the new dataset is returned as a NumPy array. Otherwise, it is returned as a list.

    ### Args:
        `dataset` (iterator or generator-type dataset): Dataset including examples to be classified.
        `scales` (Numpy array, optional): Array of scales. Defaults to np.arange(1, 33). Input to pywt.cwt function.
        `mother_wavelet` (str, optional): Mother wavelet. Defaults to 'mexh' for Mexican Hat Wavelet. 
        Input to `pywt.cwt` function.
        `preprocess` (bool, optional): Whether to preprocess the data. Defaults to True.
        `downsampling_ratio` (float, optional): Downsampling ratio. Defaults to 0.25. Only active if preprocess is True.
        `remove_last_row_column` (bool, optional): Whether to remove the last row and column of the CWT coefficients.
        Only active if `preprocess` is True. Defaults to True.

    ### Returns:
        CWT-transformed dataset: Dataset whose elements are 3D tensors of C x H' x L'
    """
    dataset_spectrogram = []
    mother_wavelet = 'mexh'
    for examples in dataset: # Every example should be C x L
        # print("Shape of example: ", np.shape(examples))
        channels = []
        for electrode_vector in examples: # Every electrode vector is an (L,) vector.
            # print("Shape of electrode vector: ", np.shape(electrode_vector))
            coefs = calculate_wavelet_vector(
                np.abs(electrode_vector), mother_wavelet=mother_wavelet, scales=scales) # H x L
            # print("Shape of CWT coefficients: ", np.shape(coefs))
            # print(np.shape(coefs))
            # show_wavelet(coef=coefs)
            if preprocess:
                coefs = zoom(coefs, downsampling_ratio, order=0) # H/4 x L/4
                # print("Shape of downsampled CWT coefficients: ", np.shape(coefs))
                if remove_last_row_column:
                    coefs = np.delete(coefs, axis=0, obj=len(coefs)-1) # (H/4 - 1) x L/4
                    coefs = np.delete(coefs, axis=1, obj=np.shape(coefs)[1]-1) # (H/4 - 1) x (L/4 - 1) = H' x L'
            # channels.append(np.swapaxes(coefs, 0, 1)) # C x L' x H'
            channels.append(coefs) # C x H' x L'
        example_to_classify = np.stack(channels, axis=0) # C x H' x L'
        # print("Shape of new example to add to the new dataset: ", np.shape(example_to_classify))
        dataset_spectrogram.append(example_to_classify) # N x C x H' x L' (what pytorch needs)
    if 'numpy' in str(type(dataset)):
        dataset_spectrogram = np.stack(dataset_spectrogram, axis=0)
    return dataset_spectrogram



def calculate_wavelet_vector(vector, mother_wavelet='mexh', scales=np.arange(1, 33)):
    """Calculate CWT of data matrix
    
    L: number of samples (sequence length)
    C: number of data channels (features)
    
    ### Args:
        `vector`: C x L matrix of data 
        `mother_wavelet` (str, optional): Mother wavelet. Defaults to 'mexh' for Mexican Hat Wavelet.
        `scales` (Numpy array, optional): Array of scales. Defaults to np.arange(1, 33).

    ### Returns:
        coef: H x C x L matrix of CWT coefficients, where H is the height of the image generated, and L is the width.
    """
    coef, _ = pywt.cwt(vector, scales=scales, wavelet=mother_wavelet)
    return coef



def show_wavelet(coef):
    """Plots the image generated by CWT for a single-dimensional data.

    ### Args:
        `coef` (Numpy array): H x C x L Matrix of coefficients returned directly by the CWT function.
        H is the height (size of the scales array passed to the CWT function)
        L is the width (sequence length)
        C is the number of channels (features)
    """
    # print(np.shape(coef))
    # plt.rcParams.update({'font.size': 36})
    ndim = len(np.shape(coef))
    if ndim == 2:
        plt.matshow(coef)
        plt.ylabel('Scale')
        plt.xlabel('Samples')
        plt.show()
    else:
        C = np.shape(coef)[1]
        for c in range(C):
            plt.subplot(C, 1, c+1)
            plt.imshow(coef[:, c, :], aspect='auto')
            # plt.matshow(coef[:, :, c])
            plt.title('Channel {}'.format(c+1))
            plt.ylabel('Scales')
        plt.xlabel('Samples')
        plt.subplots_adjust(hspace=0.5)
        plt.show()



def test_calculate_wavelet_vector():
    print("---------------------------------------")
    print("Testing calculate_wavelet_vector()...")
    L = 512
    C = 2
    H = 32
    t = np.arange(L)
    x1 = np.sin(2 * np.pi * 32 * t / L)
    x2 = np.sin(2 * np.pi * 64 * t / L)
    x = np.array([x1, x2])
    print("L: ", L)
    print("C: ", C)
    print("H: ", H)
    print("Shape of t: ", t.shape)
    print("Shape of x: ", x.shape)
    coeff = calculate_wavelet_vector(x)
    print("Shape of coeff: ", coeff.shape)
    show_wavelet(coeff)



def test_calculate_wavelet_dataset():
    print("---------------------------------------")
    print("Testing calculate_wavelet_dataset()...")
    N = 10
    L = 512
    C = 2
    H = 32
    dataset = np.random.rand(N, C, L)
    print("Shape of dataset: ", dataset.shape)
    new_dataset = calculate_wavelet_dataset(dataset)
    print("Shape of new_dataset with preprocessing:    ", new_dataset.shape)
    new_dataset = calculate_wavelet_dataset(dataset, preprocess=False)
    print("Shape of new_dataset without preprocessing: ", new_dataset.shape)



if __name__ == '__main__':
    test_calculate_wavelet_vector()
    test_calculate_wavelet_dataset()