import json

from django.contrib.gis.geos import Point
from django.test import TestCase
from django.urls import reverse

from ...config.models import Device
from ..models import DeviceLocation, Location
from . import TestGeoMixin


class TestApi(TestGeoMixin, TestCase):
    object_model = Device
    location_model = Location
    object_location_model = DeviceLocation
    url_name = 'geo:api_device_location'

    def test_permission_404(self):
        url = reverse(self.url_name, args=[self.object_model().pk])
        r = self.client.get(url)
        self.assertEqual(r.status_code, 404)

    def test_permission_403(self):
        dl = self._create_object_location()
        url = reverse(self.url_name, args=[dl.device.pk])
        r = self.client.get(url)
        self.assertEqual(r.status_code, 403)

    def test_method_not_allowed(self):
        device = self._create_object()
        url = reverse(self.url_name, args=[device.pk])
        r = self.client.post(url, {'key': device.key})
        self.assertEqual(r.status_code, 405)

    def test_get_existing_location(self):
        self.assertEqual(self.location_model.objects.count(), 0)
        dl = self._create_object_location()
        url = reverse(self.url_name, args=[dl.device.pk])
        self.assertEqual(self.location_model.objects.count(), 1)
        r = self.client.get(url, {'key': dl.device.key})
        self.assertEqual(r.status_code, 200)
        self.assertDictEqual(r.json(), {
            'type': 'Feature',
            'geometry': json.loads(dl.location.geometry.geojson),
            'properties': {'name': dl.location.name}
        })
        self.assertEqual(self.location_model.objects.count(), 1)

    def test_get_create_location(self):
        self.assertEqual(self.location_model.objects.count(), 0)
        device = self._create_object()
        url = reverse(self.url_name, args=[device.pk])
        r = self.client.get(url, {'key': device.key})
        self.assertEqual(r.status_code, 200)
        self.assertDictEqual(r.json(), {
            'type': 'Feature',
            'geometry': None,
            'properties': {'name': device.name}
        })
        self.assertEqual(self.location_model.objects.count(), 1)

    def test_put_update_coordinates(self):
        self.assertEqual(self.location_model.objects.count(), 0)
        dl = self._create_object_location()
        url = reverse(self.url_name, args=[dl.device.pk])
        url = '{0}?key={1}'.format(url, dl.device.key)
        self.assertEqual(self.location_model.objects.count(), 1)
        coords = json.loads(Point(2, 23).geojson)
        feature = json.dumps({'type': 'Feature', 'geometry': coords})
        r = self.client.put(url, feature, content_type='application/json')
        self.assertEqual(r.status_code, 200)
        self.assertDictEqual(r.json(), {
            'type': 'Feature',
            'geometry': coords,
            'properties': {'name': dl.location.name}
        })
        self.assertEqual(self.location_model.objects.count(), 1)
