# AUTOGENERATED! DO NOT EDIT! File to edit: ../notebooks/api/03_ctx.ipynb.

# %% auto 0
__all__ = ['baseurl', 'storage_root', 'edrindex', 'CTXEDR', 'CTX', 'ctx_calib', 'CTXCollection']

# %% ../notebooks/api/03_ctx.ipynb 3
import warnings
from multiprocessing import Pool
from pathlib import Path

import hvplot.xarray  # noqa
import rasterio
import rioxarray as rxr
from dask import compute, delayed
from fastcore.basics import store_attr
from fastcore.script import call_parse
from .config import config
from .pds.apps import get_index
from .utils import catch_isis_error, file_variations, url_retrieve
from tqdm.auto import tqdm
from tqdm.contrib.concurrent import process_map
from yarl import URL

try:
    from kalasiris.pysis import (
        ProcessError,
        ctxcal,
        ctxevenodd,
        getkey,
        mroctx2isis,
        spiceinit,
    )
except KeyError:
    warnings.warn("kalasiris has a problem initializing ISIS")

# %% ../notebooks/api/03_ctx.ipynb 4
warnings.filterwarnings("ignore", category=rasterio.errors.NotGeoreferencedWarning)
baseurl = URL(config.get_value("mro.ctx.datalevels.edr.url"))

# %% ../notebooks/api/03_ctx.ipynb 5
storage_root = config.storage_root / "missions/mro/ctx"
edrindex = get_index("mro.ctx", "edr")
edrindex["short_pid"] = edrindex.PRODUCT_ID.map(lambda x: x[:15])
edrindex["month_col"] = edrindex.PRODUCT_ID.map(lambda x: x[:3])

# %% ../notebooks/api/03_ctx.ipynb 9
class CTXEDR:
    """Manage access to EDR data"""

    root = config.get_value("mro.ctx.datalevels.edr.root") or storage_root / "edr"
    with_pid_folder = config.get_value("mro.ctx.datalevels.edr.with_pid_folder")
    with_volume = config.get_value("mro.ctx.datalevels.edr.with_volume")

    def __init__(
        self,
        id_: str,  # CTX product id (pid)
        root: str = "",  # alternative root folder for EDR data
        with_volume=None,  # does the storage path include the volume folder
        with_pid_folder=None,  # control if stuff is stored inside PID folders
    ):
        self.pid = id_
        self.root = Path(root) if root else Path(self.root)
        self.with_volume = with_volume if with_volume is not None else self.with_volume
        self.with_pid_folder = (
            with_pid_folder if with_pid_folder is not None else self.with_pid_folder
        )

    @property
    def pid(self):
        return self.id_

    @pid.setter
    def pid(self, value):
        if len(value) == 15:
            value = edrindex.query(f"short_pid=='{value}'").PRODUCT_ID.iloc[0]
        self.id_ = value

    @property
    def meta(self):
        "get the metadata from the index table"
        s = edrindex.query("PRODUCT_ID == @self.pid").squeeze()
        s.index = s.index.str.lower()
        return s

    @property
    def volume(self):
        "get the PDS volume number for the current product id"
        return self.meta.volume_id.lower()

    @property
    def source_folder(self):
        base = self.root
        if self.with_volume:
            base = self.root / self.volume
        if self.with_pid_folder:
            base = base / self.pid
        return base

    @property
    def source_path(self):
        return self.source_folder / f"{self.pid}.IMG"

    @property
    def url(self):
        "Calculate URL from input dataframe row."
        url = baseurl / self.meta.volume_id.lower() / "data" / (self.pid + ".IMG")
        return url

    def download(self, overwrite=False):
        if self.source_path.exists() and not overwrite:
            print("File exists. Use `overwrite=True` to download fresh.")
            return
        self.source_folder.mkdir(parents=True, exist_ok=True)
        url_retrieve(self.url, self.source_path)

    def __str__(self):
        s = f"PRODUCT_ID: {self.pid}\n"
        s += f"URL: {self.url}\n"
        s += f"source_path: {self.source_path}\n"
        return s

    def __repr__(self):
        return self.__str__()

# %% ../notebooks/api/03_ctx.ipynb 19
class CTX:
    """Class to manage dealing with CTX data.

    HAS a CTXEDR attribute as defined above.
    """

    proc_dir = Path(config.get_value("mro.ctx.root")) or storage_root / "edr"
    calib_extension = config.get_value("mro.ctx.calib_extension") or ".cal.cub"
    proc_with_pid_folder = config.get_value("mro.ctx.with_pid_folder")
    proc_with_volume = config.get_value("mro.ctx.with_volume")

    def __init__(
        self,
        id_: str,  # CTX product id
        source_dir: str = "",  # where the raw EDR data is stored, if not coming from plpy
        proc_dir: str = "",  # where to store processed, if not plpy
        with_volume=None,  # store with extra volume subfolder?
        with_id_dir=None,  # store with extra product_id subfolder?
    ):
        self.edr = CTXEDR(id_, root=source_dir, with_volume=with_volume)
        store_attr(but="source_dir,proc_dir")
        self.proc_dir = Path(proc_dir) if proc_dir else self.proc_dir

        (self.cub_name, self.cal_name, self.destripe_name) = file_variations(
            self.edr.source_path.name, [".cub", self.calib_extension, ".dst.cal.cub"]
        )
        self.is_read = False
        self.is_calib_read = False
        self.checked_destripe = False

    def __getattr__(self, attr):
        return getattr(self.edr, attr)

    @property
    def proc_folder(self):
        "the folder for all processed data. could be same as source_dir"
        base = self.proc_dir
        if self.proc_with_volume:
            base = base / self.volume
        if self.proc_with_pid_folder:
            base = base / self.pid
        return base

    @property
    def cub_path(self):
        return self.proc_folder / self.cub_name

    @property
    def cal_path(self):
        return self.proc_folder / self.cal_name

    @property
    def destripe_path(self):
        return self.proc_folder / self.destripe_name

    @catch_isis_error
    def isis_import(self):
        mroctx2isis(from_=self.source_path, to=self.cub_path)

    @catch_isis_error
    def spice_init(self, web="yes"):
        spiceinit(from_=self.cub_path, web=web)

    @catch_isis_error
    def calibrate(self):
        ctxcal(from_=self.cub_path, to=self.cal_path)
        self.is_calib_read = False

    @catch_isis_error
    def destripe(self, do_rename=True):
        if self.spatial_summing != 2:
            ctxevenodd(from_=self.cal_path, to=self.destripe_path)
            if do_rename:
                self.destripe_path.rename(self.cal_path)

    @property
    def spatial_summing(self):
        return self.meta["spatial_summing"]

    @property
    def data_quality(self):
        return self.meta.data_quality_desc

    def calib_pipeline(self, overwrite=False):
        if self.cal_path.exists() and not overwrite:
            return
        pbar = tqdm("isis_import spice_init calibrate destripe".split())
        for name in pbar:
            pbar.set_description(name)
            getattr(self, name)()
        pbar.set_description("Done.")

    def read_edr(self):
        "`da` stands for dataarray, standard abbr. within xarray."
        if not self.source_path.exists():
            raise FileNotFoundError("EDR not downloaded yet.")
        if not self.is_read:
            self.edr_da = rxr.open_rasterio(self.source_path)
            self.edr_da.name = "ctx_edr"
            self.is_read = True
        return self.edr_da.sel(band=1, drop=True)

    def read_calibrated(self, path=None, **kwargs):
        "`da` stands for dataarray, standard abbr. within xarray."
        if path is None:
            path = self.cal_path
        if not self.is_calib_read:
            self.cal_da = rxr.open_rasterio(path, **kwargs)
            self.cal_da.name = "ctx_calibrated"
            self.is_calibd_read = True
        return self.cal_da.sel(band=1, drop=True).drop_vars("spatial_ref")

    def plot_da(self, data=None):
        data = self.edr_da if data is None else data
        return data.hvplot(
            x="y",
            y="x",
            rasterize=True,
            cmap="gray",
            width=1000,
            height=400,
            title=self.pid[:15],
        )

    def plot_calibrated(self):
        return self.plot_da(self.read_calibrated())

    def __str__(self):
        s = self.edr.__str__()
        try:
            s += f"Shape: {self.read_edr().shape}"
        except FileNotFoundError:
            s += f"Not downloaded yet."
        return s

    def __repr__(self):
        return self.__str__()

# %% ../notebooks/api/03_ctx.ipynb 41
@call_parse
def ctx_calib(
    id_: str,  # CTX product_id
    source: str = "",  # path to where EDRs are stored if not from plpy
    proc_dir: str = "",  # path to where processed data is to be stored
    overwrite: bool = False,  # overwrite processed data
):
    ctx = CTX(id_, source_dir=source, proc_dir=proc_dir)
    ctx.calib_pipeline(overwrite=overwrite)
    print("Produced\n", ctx.cal_path)

# %% ../notebooks/api/03_ctx.ipynb 43
class CTXCollection:
    """Class to deal with a set of CTX products."""

    @classmethod
    def by_volume(cls, vol_id, full_width=False):
        if not str(vol_id).startswith("MROX_"):
            vol_id = "MROX_" + str(vol_id)
        query = f"VOLUME_ID=='{vol_id}'"
        if full_width:
            query += " and LINE_SAMPLES>5000"
        return cls(edrindex.query(query).PRODUCT_ID.values)

    @classmethod
    def by_month(cls, month_letters, full_width=False):
        df = edrindex[edrindex.PRODUCT_ID.str.startswith(month_letters)]
        if full_width:
            df = df.query("LINE_SAMPLES>5000")
        return cls(df.PRODUCT_ID.values)

    @classmethod
    def volume_from_pid(cls, pid, **kwargs):
        vol = edrindex.query(f"PRODUCT_ID=='{pid}'").VOLUME_ID.iat[0]
        return CTXCollection.by_volume(vol, **kwargs)

    def __init__(self, product_ids, filter_error=False):
        self.product_ids = product_ids
        if filter_error:
            self.filter_for_ok()

    @property
    def pids(self):
        "Alias on product_id"
        return self.product_ids

    def get_urls(self):
        """Get URLs for list of product_ids.

        Returns
        -------
        List[yarl.URL]
            List of URL objects with the respective PDS URL for download.
        """
        urls = []
        for p_id in self.product_ids:
            ctx = CTXEDR(p_id)
            urls.append(ctx.url)
        self.urls = urls
        return urls

    def _do_download(self, p_id, **kwargs):
        ctx = CTX(p_id)
        ctx.download(**kwargs)

    def download_collection(self, **kwargs):
        r = process_map(self._do_download, self.product_ids, max_workers=6)

    def calibrate_collection(self, overwrite=False):
        lazys = []
        for p_id in self.product_ids:
            ctx = CTX(p_id)
            lazys.append(delayed(ctx.calib_pipeline)(overwrite=overwrite))
        print("Launching parallel calibration...")
        compute(*lazys)
        print("Done.")

    def edr_exist_check(self):
        return [(p_id, CTX(p_id).source_path.exists()) for p_id in self.product_ids]

    def calib_exist_check(self):
        return [(p_id, CTX(p_id).cal_path.exists()) for p_id in self.product_ids]

    def only_full_width(self):
        new_pids = [
            pid for pid in self.product_ids if CTX(pid).meta.line_samples > 5000
        ]
        self.product_ids = new_pids

    def get_ctx_n(self, n):
        return CTX(self.product_ids[n])

    def get_pid_n(self, n):
        return self.product_ids[n]

    @property
    def n_items(self):
        return len(self.pids)

    @property
    def meta(self):
        return edrindex[edrindex.PRODUCT_ID.isin(self.pids)]

    @property
    def image_times(self):
        return self.meta.IMAGE_TIME

    def get_corrupted(self):
        return [pid for pid in self.pids if CTX(pid).data_quality == "ERROR"]

    def filter_error(self):
        self.product_ids = [
            pid for pid in self.pids if CTX(pid).data_quality != "ERROR"
        ]
