# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/template.ipynb.

# %% auto 0
__all__ = ['match_template', 'cu_match_template']

# %% ../nbs/template.ipynb 4
import numpy as np
from scipy.signal import fftconvolve
import cupy as cp
from cupyx.scipy.signal import fftconvolve as cu_fftconvolve
from typing import Union

# %% ../nbs/template.ipynb 5
def _window_sum_2d(image:np.ndarray, # reference image
                   window_shape:Union[list,tuple], # Two element list or tuple describe the shape of template image
                  ):

    window_sum = np.cumsum(image, axis=-2)
    window_sum = (window_sum[...,window_shape[0]:-1,:]
                  - window_sum[...,:-window_shape[0] - 1,:])

    window_sum = np.cumsum(window_sum, axis=-1)
    window_sum = (window_sum[..., window_shape[1]:-1]
                  - window_sum[..., :-window_shape[1] - 1])

    return window_sum

# %% ../nbs/template.ipynb 6
def _cu_window_sum_2d(image:cp.ndarray, # reference image(s) in cupy array
                      window_shape:Union[list,tuple], # Two element list or tuple describe the shape of template image
                     ):

    window_sum = cp.cumsum(image, axis=-2)
    window_sum = (window_sum[...,window_shape[0]:-1,:]
                  - window_sum[...,:-window_shape[0] - 1,:])

    window_sum = cp.cumsum(window_sum, axis=-1)
    window_sum = (window_sum[..., window_shape[1]:-1]
                  - window_sum[..., :-window_shape[1] - 1])

    return window_sum

# %% ../nbs/template.ipynb 7
def match_template(image:np.ndarray, # reference image(s)
                   template:np.ndarray, # template image(s)
                  ):
    template_shape = np.array(template.shape[-2:])
    image_shape = np.array(image.shape[-2:])

    pad_width = tuple((width, width) for width in template_shape)
    pad_width = ((0,0),)*(image.ndim-2)+pad_width
    image = np.pad(image, pad_width=pad_width, mode='constant')

    image_window_sum = _window_sum_2d(image, template_shape)
    image_window_sum2 = _window_sum_2d(image ** 2, template_shape)

    template_mean = np.expand_dims(template.mean(axis=(-2,-1)),axis=(-2,-1))
    template_volume = np.prod(template_shape) # scalar
    template_ssd = np.expand_dims(np.sum((template - template_mean) ** 2,axis=(-2,-1)),axis=(-2,-1))

    xcorr = fftconvolve(image, template[...,::-1, ::-1],
                            mode="valid",axes=(-2,-1))[...,1:-1, 1:-1]

    numerator = xcorr - image_window_sum * template_mean

    denominator = image_window_sum2
    np.multiply(image_window_sum, image_window_sum, out=image_window_sum)
    np.divide(image_window_sum, template_volume, out=image_window_sum)
    denominator -= image_window_sum
    denominator *= template_ssd
    np.maximum(denominator, 0, out=denominator)  # sqrt of negative number not allowed
    np.sqrt(denominator, out=denominator)

    response = np.zeros_like(xcorr, dtype=image.dtype)

    # avoid zero-division
    mask = denominator > np.finfo(image.dtype).eps

    response[mask] = numerator[mask] / denominator[mask]

    d0 = template_shape - 1
    d1 = d0 + image_shape - template_shape + 1

    return response[...,d0[0]:d1[0],d0[1]:d1[1]]

# %% ../nbs/template.ipynb 8
def cu_match_template(image:cp.ndarray, # reference image(s) in cupy array
                      template:cp.ndarray,# template image(s) in cupy array
                     ):
    template_shape = np.array(template.shape[-2:])
    image_shape = np.array(image.shape[-2:])

    pad_width = tuple((width, width) for width in template_shape)
    pad_width = ((0,0),)*(image.ndim-2)+pad_width
    image = cp.pad(image, pad_width=pad_width, mode='constant')

    image_window_sum = _window_sum_2d(image, template_shape)
    image_window_sum2 = _window_sum_2d(image ** 2, template_shape)

    template_mean = cp.expand_dims(template.mean(axis=(-2,-1)),axis=(-2,-1))
    template_volume = cp.prod(template_shape) # scalar
    template_ssd = cp.expand_dims(cp.sum((template - template_mean) ** 2,axis=(-2,-1)),axis=(-2,-1))

    xcorr = cu_fftconvolve(image, template[...,::-1, ::-1],
                            mode="valid",axes=(-2,-1))[...,1:-1, 1:-1]

    numerator = xcorr - image_window_sum * template_mean

    denominator = image_window_sum2
    cp.multiply(image_window_sum, image_window_sum, out=image_window_sum)
    cp.divide(image_window_sum, template_volume, out=image_window_sum)
    denominator -= image_window_sum
    denominator *= template_ssd
    cp.maximum(denominator, 0, out=denominator)  # sqrt of negative number not allowed
    cp.sqrt(denominator, out=denominator)

    response = cp.zeros_like(xcorr, dtype=image.dtype)

    # avoid zero-division
    mask = denominator > cp.finfo(image.dtype).eps

    response[mask] = numerator[mask] / denominator[mask]

    d0 = template_shape - 1
    d1 = d0 + image_shape - template_shape + 1

    return response[...,d0[0]:d1[0],d0[1]:d1[1]]
