# ---------------------------------------------------------
# Copyright (c) Microsoft Corporation. All rights reserved.
# ---------------------------------------------------------


class ModuleCliError(Exception):
    """All the errors from module related commands will be wrapped as ModuleCliError."""
    def __init__(self, ex):
        message = ModuleCliError._readable_error_message(ex)
        super(ModuleCliError, self).__init__(message)

    @staticmethod
    def _readable_error_message(ex):
        """
        This function extracts human readable error message (that to be displayed on the CLI)
        from an Exception object.

        * For the exceptions have only one str argument, respect the argument as the readable message.
          which is most of the use case, such as:

            >>> ex = FileNotFoundError("File /path/to/the/file not found.")
            >>> _readable_error_message(ex)
            'File /path/to/the/file not found.'

            >>> from azureml.exceptions import UserErrorException
            >>> ex = UserErrorException("Workspace bad_workspace does not exist.")
            >>> _readable_error_message(ex)
            'Workspace bad_workspace does not exist.'

        * For other cases, respect the __repr__ of the ex.

            >>> ex = ZeroDivisionError()
            >>> _readable_error_message(ex)
            'ZeroDivisionError()'

            >>> import os, errno
            >>> ex = OSError(errno.EPERM, os.strerror(errno.EPERM))
            >>> _readable_error_message(ex)
            "PermissionError(1, 'Operation not permitted')"

            >>> ex = OSError(errno.EIO, os.strerror(errno.EIO))
            >>> _readable_error_message(ex)
            "OSError(5, 'Input/output error')"
        """
        if len(ex.args) == 1 and isinstance(ex.args[0], str):
            message = ex.args[0]
        else:
            message = ex.__repr__()
        return message
