"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const defaults = require("@aws-solutions-constructs/core");
const cdk = require("aws-cdk-lib");
const lib_1 = require("../lib");
const kms = require("aws-cdk-lib/aws-kms");
const sqs = require("aws-cdk-lib/aws-sqs");
const ecs = require("aws-cdk-lib/aws-ecs");
test('New service/new queue, dlq, public API, new VPC', () => {
    // An environment with region is required to enable logging on an ALB
    const stack = new cdk.Stack();
    const publicApi = true;
    const clusterName = "custom-cluster-name";
    const containerName = "custom-container-name";
    const serviceName = "custom-service-name";
    const queueName = "custom-queue-name";
    const familyName = "family-name";
    const deadLetterQueueName = "dlqQueue";
    const construct = new lib_1.FargateToSqs(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { cidr: '172.0.0.0/16' },
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        queueProps: { queueName },
        deadLetterQueueProps: {
            queueName: deadLetterQueueName
        },
        queuePermissions: ['Read', 'Write']
    });
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        LaunchType: 'FARGATE',
        DesiredCount: 2,
        DeploymentConfiguration: {
            MaximumPercent: 150,
            MinimumHealthyPercent: 75
        },
        PlatformVersion: ecs.FargatePlatformVersion.LATEST,
    });
    expect(construct.vpc !== null);
    expect(construct.service !== null);
    expect(construct.container !== null);
    expect(construct.sqsQueue !== null);
    expect(construct.deadLetterQueue !== null);
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        ServiceName: serviceName
    });
    expect(stack).toHaveResourceLike("AWS::ECS::TaskDefinition", {
        Family: familyName
    });
    expect(stack).toHaveResourceLike("AWS::ECS::Cluster", {
        ClusterName: clusterName
    });
    expect(stack).toHaveResourceLike("AWS::SQS::Queue", {
        QueueName: queueName
    });
    expect(stack).toHaveResourceLike("AWS::SQS::Queue", {
        QueueName: deadLetterQueueName
    });
    expect(stack).toHaveResourceLike("AWS::ECS::TaskDefinition", {
        ContainerDefinitions: [
            {
                Essential: true,
                Image: {
                    "Fn::Join": [
                        "",
                        [
                            "123456789012.dkr.ecr.us-east-1.",
                            {
                                Ref: "AWS::URLSuffix"
                            },
                            "/fake-repo:latest"
                        ]
                    ]
                },
                MemoryReservation: 512,
                Name: containerName,
                PortMappings: [
                    {
                        ContainerPort: 8080,
                        Protocol: "tcp"
                    }
                ]
            }
        ]
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        CidrBlock: '172.0.0.0/16'
    });
    expect(stack).toHaveResourceLike("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "sqs:ReceiveMessage",
                        "sqs:ChangeMessageVisibility",
                        "sqs:GetQueueUrl",
                        "sqs:DeleteMessage",
                        "sqs:GetQueueAttributes"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testconstructtestconstructqueue6D12C99B",
                            "Arn"
                        ]
                    }
                },
                {
                    Action: [
                        "sqs:SendMessage",
                        "sqs:GetQueueAttributes",
                        "sqs:GetQueueUrl"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testconstructtestconstructqueue6D12C99B",
                            "Arn"
                        ]
                    }
                },
            ],
        }
    });
    // Confirm we created a Public/Private VPC
    expect(stack).toHaveResourceLike('AWS::EC2::InternetGateway', {});
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toCountResources('AWS::SQS::Queue', 2);
    expect(stack).toCountResources('AWS::ECS::Service', 1);
});
test('New service/new queue, private API, new VPC', () => {
    // An environment with region is required to enable logging on an ALB
    const stack = new cdk.Stack();
    const publicApi = false;
    new lib_1.FargateToSqs(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { cidr: '172.0.0.0/16' },
        deployDeadLetterQueue: false,
        queuePermissions: ['Read']
    });
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        LaunchType: 'FARGATE',
        DesiredCount: 2,
        DeploymentConfiguration: {
            MaximumPercent: 150,
            MinimumHealthyPercent: 75
        },
        PlatformVersion: ecs.FargatePlatformVersion.LATEST,
    });
    expect(stack).toHaveResourceLike("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        CidrBlock: '172.0.0.0/16'
    });
    expect(stack).toHaveResourceLike("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "sqs:ReceiveMessage",
                        "sqs:ChangeMessageVisibility",
                        "sqs:GetQueueUrl",
                        "sqs:DeleteMessage",
                        "sqs:GetQueueAttributes"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testconstructtestconstructqueue6D12C99B",
                            "Arn"
                        ]
                    }
                }
            ],
        }
    });
    // Confirm we created an Isolated VPC
    expect(stack).not.toHaveResourceLike('AWS::EC2::InternetGateway', {});
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toCountResources('AWS::SQS::Queue', 1);
    expect(stack).toCountResources('AWS::ECS::Service', 1);
});
test('New service/existing fifo queue, private API, existing VPC', () => {
    // An environment with region is required to enable logging on an ALB
    const stack = new cdk.Stack();
    const publicApi = false;
    const queueName = 'custom-queue-name.fifo';
    const existingVpc = defaults.getTestVpc(stack, publicApi);
    const existingQueue = new sqs.Queue(stack, 'MyQueue', {
        queueName,
        fifo: true
    });
    new lib_1.FargateToSqs(stack, 'test-construct', {
        publicApi,
        existingVpc,
        existingQueueObj: existingQueue,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
    });
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        LaunchType: 'FARGATE',
        DesiredCount: 2,
        DeploymentConfiguration: {
            MaximumPercent: 150,
            MinimumHealthyPercent: 75
        },
        PlatformVersion: ecs.FargatePlatformVersion.LATEST,
    });
    expect(stack).toHaveResourceLike("AWS::SQS::Queue", {
        QueueName: queueName,
        FifoQueue: true
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        CidrBlock: '172.168.0.0/16'
    });
    expect(stack).toHaveResourceLike("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "sqs:SendMessage",
                        "sqs:GetQueueAttributes",
                        "sqs:GetQueueUrl"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "MyQueueE6CA6235",
                            "Arn"
                        ]
                    }
                },
            ],
        }
    });
    // Confirm we created an Isolated VPC
    expect(stack).not.toHaveResourceLike('AWS::EC2::InternetGateway', {});
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toCountResources('AWS::SQS::Queue', 1);
    expect(stack).toCountResources('AWS::ECS::Service', 1);
});
test('Existing service/new queue, public API, existing VPC', () => {
    // An environment with region is required to enable logging on an ALB
    const stack = new cdk.Stack();
    const publicApi = true;
    const serviceName = 'custom-name';
    const customVariableName = 'CUSTOM_NAME';
    const customArnName = 'CUSTOM_ARN_NAME';
    const queueName = 'testQueue';
    const dlqName = 'dlqQueue';
    const existingVpc = defaults.getTestVpc(stack);
    const [testService, testContainer] = defaults.CreateFargateService(stack, 'test', existingVpc, undefined, defaults.fakeEcrRepoArn, undefined, undefined, undefined, { serviceName });
    new lib_1.FargateToSqs(stack, 'test-construct', {
        publicApi,
        existingFargateServiceObject: testService,
        existingContainerDefinitionObject: testContainer,
        existingVpc,
        queueUrlEnvironmentVariableName: customVariableName,
        queueArnEnvironmentVariableName: customArnName,
        queueProps: {
            queueName
        },
        deadLetterQueueProps: {
            queueName: dlqName
        }
    });
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        ServiceName: serviceName
    });
    expect(stack).toHaveResourceLike("AWS::ECS::TaskDefinition", {
        ContainerDefinitions: [
            {
                Environment: [
                    {
                        Name: customArnName,
                        Value: {
                            "Fn::GetAtt": [
                                "testconstructtestconstructqueue6D12C99B",
                                "Arn"
                            ]
                        }
                    },
                    {
                        Name: customVariableName,
                        Value: {
                            Ref: "testconstructtestconstructqueue6D12C99B"
                        }
                    },
                ],
                Essential: true,
                Image: {
                    "Fn::Join": [
                        "",
                        [
                            "123456789012.dkr.ecr.us-east-1.",
                            {
                                Ref: "AWS::URLSuffix"
                            },
                            "/fake-repo:latest"
                        ]
                    ]
                },
                MemoryReservation: 512,
                Name: "test-container",
                PortMappings: [
                    {
                        ContainerPort: 8080,
                        Protocol: "tcp"
                    }
                ]
            }
        ]
    });
    expect(stack).toHaveResourceLike("AWS::SQS::Queue", {
        QueueName: queueName
    });
    expect(stack).toHaveResourceLike("AWS::SQS::Queue", {
        QueueName: dlqName
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        CidrBlock: '172.168.0.0/16'
    });
    // Confirm we created a Public/Private VPC
    expect(stack).toHaveResourceLike('AWS::EC2::InternetGateway', {});
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toCountResources('AWS::SQS::Queue', 2);
    expect(stack).toCountResources('AWS::ECS::Service', 1);
});
// Test existing service/existing queue, private API, new VPC
test('Existing service/existing queue, private API, existing VPC', () => {
    // An environment with region is required to enable logging on an ALB
    const stack = new cdk.Stack();
    const publicApi = false;
    const serviceName = 'custom-name';
    const queueName = 'custom-queue-name';
    const existingVpc = defaults.getTestVpc(stack, publicApi);
    const [testService, testContainer] = defaults.CreateFargateService(stack, 'test', existingVpc, undefined, defaults.fakeEcrRepoArn, undefined, undefined, undefined, { serviceName });
    const existingQueue = new sqs.Queue(stack, 'MyQueue', {
        queueName
    });
    new lib_1.FargateToSqs(stack, 'test-construct', {
        publicApi,
        existingFargateServiceObject: testService,
        existingContainerDefinitionObject: testContainer,
        existingVpc,
        existingQueueObj: existingQueue
    });
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        ServiceName: serviceName,
    });
    expect(stack).toHaveResourceLike("AWS::ECS::TaskDefinition", {
        ContainerDefinitions: [
            {
                Environment: [
                    {
                        Name: "SQS_QUEUE_ARN",
                        Value: {
                            "Fn::GetAtt": [
                                "MyQueueE6CA6235",
                                "Arn"
                            ]
                        }
                    },
                    {
                        Name: "SQS_QUEUE_URL",
                        Value: {
                            Ref: "MyQueueE6CA6235"
                        }
                    }
                ],
                Essential: true,
                Image: {
                    "Fn::Join": [
                        "",
                        [
                            "123456789012.dkr.ecr.us-east-1.",
                            {
                                Ref: "AWS::URLSuffix"
                            },
                            "/fake-repo:latest"
                        ]
                    ]
                },
                MemoryReservation: 512,
                Name: "test-container",
                PortMappings: [
                    {
                        ContainerPort: 8080,
                        Protocol: "tcp"
                    }
                ]
            }
        ]
    });
    expect(stack).toHaveResourceLike("AWS::SQS::Queue", {
        QueueName: queueName
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        CidrBlock: '172.168.0.0/16'
    });
    // Confirm we created an Isolated VPC
    expect(stack).not.toHaveResourceLike('AWS::EC2::InternetGateway', {});
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toCountResources('AWS::SQS::Queue', 1);
    expect(stack).toCountResources('AWS::ECS::Service', 1);
});
test('Test bad queuePermissions', () => {
    // An environment with region is required to enable logging on an ALB
    const stack = new cdk.Stack();
    const publicApi = false;
    const props = {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { cidr: '172.0.0.0/16' },
        deployDeadLetterQueue: false,
        queuePermissions: ['Reed'],
    };
    const app = () => {
        new lib_1.FargateToSqs(stack, 'test-construct', props);
    };
    expect(app).toThrowError('Invalid queue permission submitted - Reed');
});
test('Queue is encrypted with imported CMK when set on encryptionKey prop', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const cmk = new kms.Key(stack, 'cmk');
    new lib_1.FargateToSqs(stack, 'test-construct', {
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        encryptionKey: cmk
    });
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "cmk01DE03DA",
                "Arn"
            ]
        }
    });
});
test('Queue is encrypted with imported CMK when set on queueProps.encryptionMasterKey prop', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const cmk = new kms.Key(stack, 'cmk');
    new lib_1.FargateToSqs(stack, 'test-construct', {
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        queueProps: {
            encryptionMasterKey: cmk
        }
    });
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "cmk01DE03DA",
                "Arn"
            ]
        }
    });
});
test('Queue is encrypted with provided encrytionKeyProps', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    new lib_1.FargateToSqs(stack, 'test-construct', {
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        encryptionKeyProps: {
            alias: 'new-key-alias-from-props'
        }
    });
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
    expect(stack).toHaveResource('AWS::KMS::Alias', {
        AliasName: 'alias/new-key-alias-from-props',
        TargetKeyId: {
            'Fn::GetAtt': [
                'testconstructEncryptionKey6153B053',
                'Arn'
            ]
        }
    });
});
test('Queue is encrypted with SQS-managed KMS key when no other encryption propreties are set', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    new lib_1.FargateToSqs(stack, 'test-construct', {
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
    });
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
});
test('Queue is encrypted with customer managed KMS Key when enable encryption flag is true', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    new lib_1.FargateToSqs(stack, 'test-construct', {
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        enableEncryptionWithCustomerManagedKey: true
    });
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "testconstructEncryptionKey6153B053",
                "Arn"
            ]
        }
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZmFyZ2F0ZS1zcXMudGVzdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImZhcmdhdGUtc3FzLnRlc3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7Ozs7Ozs7OztHQVdHOztBQUVILGdDQUE4QjtBQUM5QiwyREFBMkQ7QUFDM0QsbUNBQW1DO0FBQ25DLGdDQUFzQztBQUN0QywyQ0FBMkM7QUFDM0MsMkNBQTJDO0FBQzNDLDJDQUEyQztBQUUzQyxJQUFJLENBQUMsaURBQWlELEVBQUUsR0FBRyxFQUFFO0lBRTNELHFFQUFxRTtJQUNyRSxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUM5QixNQUFNLFNBQVMsR0FBRyxJQUFJLENBQUM7SUFDdkIsTUFBTSxXQUFXLEdBQUcscUJBQXFCLENBQUM7SUFDMUMsTUFBTSxhQUFhLEdBQUcsdUJBQXVCLENBQUM7SUFDOUMsTUFBTSxXQUFXLEdBQUcscUJBQXFCLENBQUM7SUFDMUMsTUFBTSxTQUFTLEdBQUcsbUJBQW1CLENBQUM7SUFDdEMsTUFBTSxVQUFVLEdBQUcsYUFBYSxDQUFDO0lBQ2pDLE1BQU0sbUJBQW1CLEdBQUcsVUFBVSxDQUFDO0lBRXZDLE1BQU0sU0FBUyxHQUFHLElBQUksa0JBQVksQ0FBQyxLQUFLLEVBQUUsZ0JBQWdCLEVBQUU7UUFDMUQsU0FBUztRQUNULGdCQUFnQixFQUFFLFFBQVEsQ0FBQyxjQUFjO1FBQ3pDLFFBQVEsRUFBRSxFQUFFLElBQUksRUFBRSxjQUFjLEVBQUU7UUFDbEMsWUFBWSxFQUFFLEVBQUUsV0FBVyxFQUFFO1FBQzdCLHdCQUF3QixFQUFFLEVBQUUsYUFBYSxFQUFFO1FBQzNDLDBCQUEwQixFQUFFLEVBQUUsTUFBTSxFQUFFLFVBQVUsRUFBRTtRQUNsRCxtQkFBbUIsRUFBRSxFQUFFLFdBQVcsRUFBRTtRQUNwQyxVQUFVLEVBQUUsRUFBRSxTQUFTLEVBQUU7UUFDekIsb0JBQW9CLEVBQUU7WUFDcEIsU0FBUyxFQUFFLG1CQUFtQjtTQUMvQjtRQUNELGdCQUFnQixFQUFFLENBQUMsTUFBTSxFQUFFLE9BQU8sQ0FBQztLQUNwQyxDQUFDLENBQUM7SUFFSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsbUJBQW1CLEVBQUU7UUFDcEQsVUFBVSxFQUFFLFNBQVM7UUFDckIsWUFBWSxFQUFFLENBQUM7UUFDZix1QkFBdUIsRUFBRTtZQUN2QixjQUFjLEVBQUUsR0FBRztZQUNuQixxQkFBcUIsRUFBRSxFQUFFO1NBQzFCO1FBQ0QsZUFBZSxFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxNQUFNO0tBQ25ELENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxTQUFTLENBQUMsR0FBRyxLQUFLLElBQUksQ0FBQyxDQUFDO0lBQy9CLE1BQU0sQ0FBQyxTQUFTLENBQUMsT0FBTyxLQUFLLElBQUksQ0FBQyxDQUFDO0lBQ25DLE1BQU0sQ0FBQyxTQUFTLENBQUMsU0FBUyxLQUFLLElBQUksQ0FBQyxDQUFDO0lBQ3JDLE1BQU0sQ0FBQyxTQUFTLENBQUMsUUFBUSxLQUFLLElBQUksQ0FBQyxDQUFDO0lBQ3BDLE1BQU0sQ0FBQyxTQUFTLENBQUMsZUFBZSxLQUFLLElBQUksQ0FBQyxDQUFDO0lBRTNDLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxtQkFBbUIsRUFBRTtRQUNwRCxXQUFXLEVBQUUsV0FBVztLQUN6QixDQUFDLENBQUM7SUFDSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsMEJBQTBCLEVBQUU7UUFDM0QsTUFBTSxFQUFFLFVBQVU7S0FDbkIsQ0FBQyxDQUFDO0lBRUgsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLG1CQUFtQixFQUFFO1FBQ3BELFdBQVcsRUFBRSxXQUFXO0tBQ3pCLENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxpQkFBaUIsRUFBRTtRQUNsRCxTQUFTLEVBQUUsU0FBUztLQUNyQixDQUFDLENBQUM7SUFFSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsaUJBQWlCLEVBQUU7UUFDbEQsU0FBUyxFQUFFLG1CQUFtQjtLQUMvQixDQUFDLENBQUM7SUFFSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsMEJBQTBCLEVBQUU7UUFDM0Qsb0JBQW9CLEVBQUU7WUFDcEI7Z0JBQ0UsU0FBUyxFQUFFLElBQUk7Z0JBQ2YsS0FBSyxFQUFFO29CQUNMLFVBQVUsRUFBRTt3QkFDVixFQUFFO3dCQUNGOzRCQUNFLGlDQUFpQzs0QkFDakM7Z0NBQ0UsR0FBRyxFQUFFLGdCQUFnQjs2QkFDdEI7NEJBQ0QsbUJBQW1CO3lCQUNwQjtxQkFDRjtpQkFDRjtnQkFDRCxpQkFBaUIsRUFBRSxHQUFHO2dCQUN0QixJQUFJLEVBQUUsYUFBYTtnQkFDbkIsWUFBWSxFQUFFO29CQUNaO3dCQUNFLGFBQWEsRUFBRSxJQUFJO3dCQUNuQixRQUFRLEVBQUUsS0FBSztxQkFDaEI7aUJBQ0Y7YUFDRjtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLGVBQWUsRUFBRTtRQUNoRCxTQUFTLEVBQUUsY0FBYztLQUMxQixDQUFDLENBQUM7SUFFSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsa0JBQWtCLEVBQUU7UUFDbkQsY0FBYyxFQUFFO1lBQ2QsU0FBUyxFQUFFO2dCQUNUO29CQUNFLE1BQU0sRUFBRTt3QkFDTixvQkFBb0I7d0JBQ3BCLDZCQUE2Qjt3QkFDN0IsaUJBQWlCO3dCQUNqQixtQkFBbUI7d0JBQ25CLHdCQUF3QjtxQkFDekI7b0JBQ0QsTUFBTSxFQUFFLE9BQU87b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFlBQVksRUFBRTs0QkFDWix5Q0FBeUM7NEJBQ3pDLEtBQUs7eUJBQ047cUJBQ0Y7aUJBQ0Y7Z0JBQ0Q7b0JBQ0UsTUFBTSxFQUFFO3dCQUNOLGlCQUFpQjt3QkFDakIsd0JBQXdCO3dCQUN4QixpQkFBaUI7cUJBQ2xCO29CQUNELE1BQU0sRUFBRSxPQUFPO29CQUNmLFFBQVEsRUFBRTt3QkFDUixZQUFZLEVBQUU7NEJBQ1oseUNBQXlDOzRCQUN6QyxLQUFLO3lCQUNOO3FCQUNGO2lCQUNGO2FBQ0Y7U0FDRjtLQUNGLENBQUMsQ0FBQztJQUVILDBDQUEwQztJQUMxQyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsMkJBQTJCLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDbEUsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGdCQUFnQixDQUFDLGVBQWUsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUNuRCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsZ0JBQWdCLENBQUMsaUJBQWlCLEVBQUUsQ0FBQyxDQUFDLENBQUM7SUFDckQsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGdCQUFnQixDQUFDLG1CQUFtQixFQUFFLENBQUMsQ0FBQyxDQUFDO0FBQ3pELENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDZDQUE2QyxFQUFFLEdBQUcsRUFBRTtJQUV2RCxxRUFBcUU7SUFDckUsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7SUFDOUIsTUFBTSxTQUFTLEdBQUcsS0FBSyxDQUFDO0lBRXhCLElBQUksa0JBQVksQ0FBQyxLQUFLLEVBQUUsZ0JBQWdCLEVBQUU7UUFDeEMsU0FBUztRQUNULGdCQUFnQixFQUFFLFFBQVEsQ0FBQyxjQUFjO1FBQ3pDLFFBQVEsRUFBRSxFQUFFLElBQUksRUFBRSxjQUFjLEVBQUU7UUFDbEMscUJBQXFCLEVBQUUsS0FBSztRQUM1QixnQkFBZ0IsRUFBRSxDQUFDLE1BQU0sQ0FBQztLQUMzQixDQUFDLENBQUM7SUFFSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsbUJBQW1CLEVBQUU7UUFDcEQsVUFBVSxFQUFFLFNBQVM7UUFDckIsWUFBWSxFQUFFLENBQUM7UUFDZix1QkFBdUIsRUFBRTtZQUN2QixjQUFjLEVBQUUsR0FBRztZQUNuQixxQkFBcUIsRUFBRSxFQUFFO1NBQzFCO1FBQ0QsZUFBZSxFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxNQUFNO0tBQ25ELENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxpQkFBaUIsRUFBRTtRQUNsRCxjQUFjLEVBQUUsZUFBZTtLQUNoQyxDQUFDLENBQUM7SUFFSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsZUFBZSxFQUFFO1FBQ2hELFNBQVMsRUFBRSxjQUFjO0tBQzFCLENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxrQkFBa0IsRUFBRTtRQUNuRCxjQUFjLEVBQUU7WUFDZCxTQUFTLEVBQUU7Z0JBQ1Q7b0JBQ0UsTUFBTSxFQUFFO3dCQUNOLG9CQUFvQjt3QkFDcEIsNkJBQTZCO3dCQUM3QixpQkFBaUI7d0JBQ2pCLG1CQUFtQjt3QkFDbkIsd0JBQXdCO3FCQUN6QjtvQkFDRCxNQUFNLEVBQUUsT0FBTztvQkFDZixRQUFRLEVBQUU7d0JBQ1IsWUFBWSxFQUFFOzRCQUNaLHlDQUF5Qzs0QkFDekMsS0FBSzt5QkFDTjtxQkFDRjtpQkFDRjthQUNGO1NBQ0Y7S0FDRixDQUFDLENBQUM7SUFFSCxxQ0FBcUM7SUFDckMsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQywyQkFBMkIsRUFBRSxFQUFFLENBQUMsQ0FBQztJQUN0RSxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsZ0JBQWdCLENBQUMsZUFBZSxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQ25ELE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxnQkFBZ0IsQ0FBQyxpQkFBaUIsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUNyRCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsZ0JBQWdCLENBQUMsbUJBQW1CLEVBQUUsQ0FBQyxDQUFDLENBQUM7QUFDekQsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsNERBQTRELEVBQUUsR0FBRyxFQUFFO0lBQ3RFLHFFQUFxRTtJQUNyRSxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUM5QixNQUFNLFNBQVMsR0FBRyxLQUFLLENBQUM7SUFDeEIsTUFBTSxTQUFTLEdBQUcsd0JBQXdCLENBQUM7SUFFM0MsTUFBTSxXQUFXLEdBQUcsUUFBUSxDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDLENBQUM7SUFFMUQsTUFBTSxhQUFhLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUU7UUFDcEQsU0FBUztRQUNULElBQUksRUFBRSxJQUFJO0tBQ1gsQ0FBQyxDQUFDO0lBRUgsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxnQkFBZ0IsRUFBRTtRQUN4QyxTQUFTO1FBQ1QsV0FBVztRQUNYLGdCQUFnQixFQUFFLGFBQWE7UUFDL0IsZ0JBQWdCLEVBQUUsUUFBUSxDQUFDLGNBQWM7S0FDMUMsQ0FBQyxDQUFDO0lBRUgsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLG1CQUFtQixFQUFFO1FBQ3BELFVBQVUsRUFBRSxTQUFTO1FBQ3JCLFlBQVksRUFBRSxDQUFDO1FBQ2YsdUJBQXVCLEVBQUU7WUFDdkIsY0FBYyxFQUFFLEdBQUc7WUFDbkIscUJBQXFCLEVBQUUsRUFBRTtTQUMxQjtRQUNELGVBQWUsRUFBRSxHQUFHLENBQUMsc0JBQXNCLENBQUMsTUFBTTtLQUNuRCxDQUFDLENBQUM7SUFFSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsaUJBQWlCLEVBQUU7UUFDbEQsU0FBUyxFQUFFLFNBQVM7UUFDcEIsU0FBUyxFQUFFLElBQUk7S0FDaEIsQ0FBQyxDQUFDO0lBRUgsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLGVBQWUsRUFBRTtRQUNoRCxTQUFTLEVBQUUsZ0JBQWdCO0tBQzVCLENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxrQkFBa0IsRUFBRTtRQUNuRCxjQUFjLEVBQUU7WUFDZCxTQUFTLEVBQUU7Z0JBQ1Q7b0JBQ0UsTUFBTSxFQUFFO3dCQUNOLGlCQUFpQjt3QkFDakIsd0JBQXdCO3dCQUN4QixpQkFBaUI7cUJBQ2xCO29CQUNELE1BQU0sRUFBRSxPQUFPO29CQUNmLFFBQVEsRUFBRTt3QkFDUixZQUFZLEVBQUU7NEJBQ1osaUJBQWlCOzRCQUNqQixLQUFLO3lCQUNOO3FCQUNGO2lCQUNGO2FBQ0Y7U0FDRjtLQUNGLENBQUMsQ0FBQztJQUVILHFDQUFxQztJQUNyQyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsR0FBRyxDQUFDLGtCQUFrQixDQUFDLDJCQUEyQixFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQ3RFLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxnQkFBZ0IsQ0FBQyxlQUFlLEVBQUUsQ0FBQyxDQUFDLENBQUM7SUFDbkQsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGdCQUFnQixDQUFDLGlCQUFpQixFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQ3JELE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxnQkFBZ0IsQ0FBQyxtQkFBbUIsRUFBRSxDQUFDLENBQUMsQ0FBQztBQUN6RCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxzREFBc0QsRUFBRSxHQUFHLEVBQUU7SUFDaEUscUVBQXFFO0lBQ3JFLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBQzlCLE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQztJQUN2QixNQUFNLFdBQVcsR0FBRyxhQUFhLENBQUM7SUFDbEMsTUFBTSxrQkFBa0IsR0FBRyxhQUFhLENBQUM7SUFDekMsTUFBTSxhQUFhLEdBQUcsaUJBQWlCLENBQUM7SUFDeEMsTUFBTSxTQUFTLEdBQUcsV0FBVyxDQUFDO0lBQzlCLE1BQU0sT0FBTyxHQUFHLFVBQVUsQ0FBQztJQUUzQixNQUFNLFdBQVcsR0FBRyxRQUFRLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBRS9DLE1BQU0sQ0FBQyxXQUFXLEVBQUUsYUFBYSxDQUFDLEdBQUcsUUFBUSxDQUFDLG9CQUFvQixDQUFDLEtBQUssRUFDdEUsTUFBTSxFQUNOLFdBQVcsRUFDWCxTQUFTLEVBQ1QsUUFBUSxDQUFDLGNBQWMsRUFDdkIsU0FBUyxFQUNULFNBQVMsRUFDVCxTQUFTLEVBQ1QsRUFBRSxXQUFXLEVBQUUsQ0FBQyxDQUFDO0lBRW5CLElBQUksa0JBQVksQ0FBQyxLQUFLLEVBQUUsZ0JBQWdCLEVBQUU7UUFDeEMsU0FBUztRQUNULDRCQUE0QixFQUFFLFdBQVc7UUFDekMsaUNBQWlDLEVBQUUsYUFBYTtRQUNoRCxXQUFXO1FBQ1gsK0JBQStCLEVBQUUsa0JBQWtCO1FBQ25ELCtCQUErQixFQUFFLGFBQWE7UUFDOUMsVUFBVSxFQUFFO1lBQ1YsU0FBUztTQUNWO1FBQ0Qsb0JBQW9CLEVBQUU7WUFDcEIsU0FBUyxFQUFFLE9BQU87U0FDbkI7S0FDRixDQUFDLENBQUM7SUFFSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsbUJBQW1CLEVBQUU7UUFDcEQsV0FBVyxFQUFFLFdBQVc7S0FDekIsQ0FBQyxDQUFDO0lBRUgsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLDBCQUEwQixFQUFFO1FBQzNELG9CQUFvQixFQUFFO1lBQ3BCO2dCQUNFLFdBQVcsRUFBRTtvQkFDWDt3QkFDRSxJQUFJLEVBQUUsYUFBYTt3QkFDbkIsS0FBSyxFQUFFOzRCQUNMLFlBQVksRUFBRTtnQ0FDWix5Q0FBeUM7Z0NBQ3pDLEtBQUs7NkJBQ047eUJBQ0Y7cUJBQ0Y7b0JBQ0Q7d0JBQ0UsSUFBSSxFQUFFLGtCQUFrQjt3QkFDeEIsS0FBSyxFQUFFOzRCQUNMLEdBQUcsRUFBRSx5Q0FBeUM7eUJBQy9DO3FCQUNGO2lCQUNGO2dCQUNELFNBQVMsRUFBRSxJQUFJO2dCQUNmLEtBQUssRUFBRTtvQkFDTCxVQUFVLEVBQUU7d0JBQ1YsRUFBRTt3QkFDRjs0QkFDRSxpQ0FBaUM7NEJBQ2pDO2dDQUNFLEdBQUcsRUFBRSxnQkFBZ0I7NkJBQ3RCOzRCQUNELG1CQUFtQjt5QkFDcEI7cUJBQ0Y7aUJBQ0Y7Z0JBQ0QsaUJBQWlCLEVBQUUsR0FBRztnQkFDdEIsSUFBSSxFQUFFLGdCQUFnQjtnQkFDdEIsWUFBWSxFQUFFO29CQUNaO3dCQUNFLGFBQWEsRUFBRSxJQUFJO3dCQUNuQixRQUFRLEVBQUUsS0FBSztxQkFDaEI7aUJBQ0Y7YUFDRjtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0lBQ0gsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLGlCQUFpQixFQUFFO1FBQ2xELFNBQVMsRUFBRSxTQUFTO0tBQ3JCLENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxpQkFBaUIsRUFBRTtRQUNsRCxTQUFTLEVBQUUsT0FBTztLQUNuQixDQUFDLENBQUM7SUFFSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsZUFBZSxFQUFFO1FBQ2hELFNBQVMsRUFBRSxnQkFBZ0I7S0FDNUIsQ0FBQyxDQUFDO0lBRUgsMENBQTBDO0lBQzFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQywyQkFBMkIsRUFBRSxFQUFFLENBQUMsQ0FBQztJQUNsRSxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsZ0JBQWdCLENBQUMsZUFBZSxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQ25ELE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxnQkFBZ0IsQ0FBQyxpQkFBaUIsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUNyRCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsZ0JBQWdCLENBQUMsbUJBQW1CLEVBQUUsQ0FBQyxDQUFDLENBQUM7QUFDekQsQ0FBQyxDQUFDLENBQUM7QUFFSCw2REFBNkQ7QUFDN0QsSUFBSSxDQUFDLDREQUE0RCxFQUFFLEdBQUcsRUFBRTtJQUN0RSxxRUFBcUU7SUFDckUsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7SUFDOUIsTUFBTSxTQUFTLEdBQUcsS0FBSyxDQUFDO0lBQ3hCLE1BQU0sV0FBVyxHQUFHLGFBQWEsQ0FBQztJQUNsQyxNQUFNLFNBQVMsR0FBRyxtQkFBbUIsQ0FBQztJQUV0QyxNQUFNLFdBQVcsR0FBRyxRQUFRLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRSxTQUFTLENBQUMsQ0FBQztJQUUxRCxNQUFNLENBQUMsV0FBVyxFQUFFLGFBQWEsQ0FBQyxHQUFHLFFBQVEsQ0FBQyxvQkFBb0IsQ0FBQyxLQUFLLEVBQ3RFLE1BQU0sRUFDTixXQUFXLEVBQ1gsU0FBUyxFQUNULFFBQVEsQ0FBQyxjQUFjLEVBQ3ZCLFNBQVMsRUFDVCxTQUFTLEVBQ1QsU0FBUyxFQUNULEVBQUUsV0FBVyxFQUFFLENBQUMsQ0FBQztJQUVuQixNQUFNLGFBQWEsR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRTtRQUNwRCxTQUFTO0tBQ1YsQ0FBQyxDQUFDO0lBRUgsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxnQkFBZ0IsRUFBRTtRQUN4QyxTQUFTO1FBQ1QsNEJBQTRCLEVBQUUsV0FBVztRQUN6QyxpQ0FBaUMsRUFBRSxhQUFhO1FBQ2hELFdBQVc7UUFDWCxnQkFBZ0IsRUFBRSxhQUFhO0tBQ2hDLENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxtQkFBbUIsRUFBRTtRQUNwRCxXQUFXLEVBQUUsV0FBVztLQUN6QixDQUFDLENBQUM7SUFFSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsMEJBQTBCLEVBQUU7UUFDM0Qsb0JBQW9CLEVBQUU7WUFDcEI7Z0JBQ0UsV0FBVyxFQUFFO29CQUNYO3dCQUNFLElBQUksRUFBRSxlQUFlO3dCQUNyQixLQUFLLEVBQUU7NEJBQ0wsWUFBWSxFQUFFO2dDQUNaLGlCQUFpQjtnQ0FDakIsS0FBSzs2QkFDTjt5QkFDRjtxQkFDRjtvQkFDRDt3QkFDRSxJQUFJLEVBQUUsZUFBZTt3QkFDckIsS0FBSyxFQUFFOzRCQUNMLEdBQUcsRUFBRSxpQkFBaUI7eUJBQ3ZCO3FCQUNGO2lCQUNGO2dCQUNELFNBQVMsRUFBRSxJQUFJO2dCQUNmLEtBQUssRUFBRTtvQkFDTCxVQUFVLEVBQUU7d0JBQ1YsRUFBRTt3QkFDRjs0QkFDRSxpQ0FBaUM7NEJBQ2pDO2dDQUNFLEdBQUcsRUFBRSxnQkFBZ0I7NkJBQ3RCOzRCQUNELG1CQUFtQjt5QkFDcEI7cUJBQ0Y7aUJBQ0Y7Z0JBQ0QsaUJBQWlCLEVBQUUsR0FBRztnQkFDdEIsSUFBSSxFQUFFLGdCQUFnQjtnQkFDdEIsWUFBWSxFQUFFO29CQUNaO3dCQUNFLGFBQWEsRUFBRSxJQUFJO3dCQUNuQixRQUFRLEVBQUUsS0FBSztxQkFDaEI7aUJBQ0Y7YUFDRjtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLGlCQUFpQixFQUFFO1FBQ2xELFNBQVMsRUFBRSxTQUFTO0tBQ3JCLENBQUMsQ0FBQztJQUNILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxlQUFlLEVBQUU7UUFDaEQsU0FBUyxFQUFFLGdCQUFnQjtLQUM1QixDQUFDLENBQUM7SUFFSCxxQ0FBcUM7SUFDckMsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQywyQkFBMkIsRUFBRSxFQUFFLENBQUMsQ0FBQztJQUN0RSxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsZ0JBQWdCLENBQUMsZUFBZSxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQ25ELE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxnQkFBZ0IsQ0FBQyxpQkFBaUIsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUNyRCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsZ0JBQWdCLENBQUMsbUJBQW1CLEVBQUUsQ0FBQyxDQUFDLENBQUM7QUFDekQsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsMkJBQTJCLEVBQUUsR0FBRyxFQUFFO0lBRXJDLHFFQUFxRTtJQUNyRSxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUM5QixNQUFNLFNBQVMsR0FBRyxLQUFLLENBQUM7SUFFeEIsTUFBTSxLQUFLLEdBQUc7UUFDWixTQUFTO1FBQ1QsZ0JBQWdCLEVBQUUsUUFBUSxDQUFDLGNBQWM7UUFDekMsUUFBUSxFQUFFLEVBQUUsSUFBSSxFQUFFLGNBQWMsRUFBRTtRQUNsQyxxQkFBcUIsRUFBRSxLQUFLO1FBQzVCLGdCQUFnQixFQUFFLENBQUMsTUFBTSxDQUFDO0tBQzNCLENBQUM7SUFFRixNQUFNLEdBQUcsR0FBRyxHQUFHLEVBQUU7UUFDZixJQUFJLGtCQUFZLENBQUMsS0FBSyxFQUFFLGdCQUFnQixFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ25ELENBQUMsQ0FBQztJQUVGLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxZQUFZLENBQUMsMkNBQTJDLENBQUMsQ0FBQztBQUN4RSxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxxRUFBcUUsRUFBRSxHQUFHLEVBQUU7SUFDL0UsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLFNBQVMsRUFBRSxTQUFTLEVBQUU7UUFDaEQsR0FBRyxFQUFFLEVBQUUsT0FBTyxFQUFFLGNBQWMsRUFBRSxNQUFNLEVBQUUsV0FBVyxFQUFFO0tBQ3RELENBQUMsQ0FBQztJQUVILE1BQU0sR0FBRyxHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDdEMsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxnQkFBZ0IsRUFBRTtRQUN4QyxTQUFTLEVBQUUsSUFBSTtRQUNmLGdCQUFnQixFQUFFLFFBQVEsQ0FBQyxjQUFjO1FBQ3pDLGFBQWEsRUFBRSxHQUFHO0tBQ25CLENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsaUJBQWlCLEVBQUU7UUFDOUMsY0FBYyxFQUFFO1lBQ2QsWUFBWSxFQUFFO2dCQUNaLGFBQWE7Z0JBQ2IsS0FBSzthQUNOO1NBQ0Y7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxzRkFBc0YsRUFBRSxHQUFHLEVBQUU7SUFDaEcsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLFNBQVMsRUFBRSxTQUFTLEVBQUU7UUFDaEQsR0FBRyxFQUFFLEVBQUUsT0FBTyxFQUFFLGNBQWMsRUFBRSxNQUFNLEVBQUUsV0FBVyxFQUFFO0tBQ3RELENBQUMsQ0FBQztJQUVILE1BQU0sR0FBRyxHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDdEMsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxnQkFBZ0IsRUFBRTtRQUN4QyxTQUFTLEVBQUUsSUFBSTtRQUNmLGdCQUFnQixFQUFFLFFBQVEsQ0FBQyxjQUFjO1FBQ3pDLFVBQVUsRUFBRTtZQUNWLG1CQUFtQixFQUFFLEdBQUc7U0FDekI7S0FDRixDQUFDLENBQUM7SUFFSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLGlCQUFpQixFQUFFO1FBQzlDLGNBQWMsRUFBRTtZQUNkLFlBQVksRUFBRTtnQkFDWixhQUFhO2dCQUNiLEtBQUs7YUFDTjtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsb0RBQW9ELEVBQUUsR0FBRyxFQUFFO0lBQzlELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxTQUFTLEVBQUUsU0FBUyxFQUFFO1FBQ2hELEdBQUcsRUFBRSxFQUFFLE9BQU8sRUFBRSxjQUFjLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBRTtLQUN0RCxDQUFDLENBQUM7SUFFSCxJQUFJLGtCQUFZLENBQUMsS0FBSyxFQUFFLGdCQUFnQixFQUFFO1FBQ3hDLFNBQVMsRUFBRSxJQUFJO1FBQ2YsZ0JBQWdCLEVBQUUsUUFBUSxDQUFDLGNBQWM7UUFDekMsa0JBQWtCLEVBQUU7WUFDbEIsS0FBSyxFQUFFLDBCQUEwQjtTQUNsQztLQUNGLENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsaUJBQWlCLEVBQUU7UUFDOUMsY0FBYyxFQUFFLGVBQWU7S0FDaEMsQ0FBQyxDQUFDO0lBRUgsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyxpQkFBaUIsRUFBRTtRQUM5QyxTQUFTLEVBQUUsZ0NBQWdDO1FBQzNDLFdBQVcsRUFBRTtZQUNYLFlBQVksRUFBRTtnQkFDWixvQ0FBb0M7Z0JBQ3BDLEtBQUs7YUFDTjtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMseUZBQXlGLEVBQUUsR0FBRyxFQUFFO0lBQ25HLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxTQUFTLEVBQUUsU0FBUyxFQUFFO1FBQ2hELEdBQUcsRUFBRSxFQUFFLE9BQU8sRUFBRSxjQUFjLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBRTtLQUN0RCxDQUFDLENBQUM7SUFFSCxJQUFJLGtCQUFZLENBQUMsS0FBSyxFQUFFLGdCQUFnQixFQUFFO1FBQ3hDLFNBQVMsRUFBRSxJQUFJO1FBQ2YsZ0JBQWdCLEVBQUUsUUFBUSxDQUFDLGNBQWM7S0FDMUMsQ0FBQyxDQUFDO0lBRUgsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyxpQkFBaUIsRUFBRTtRQUM5QyxjQUFjLEVBQUUsZUFBZTtLQUNoQyxDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxzRkFBc0YsRUFBRSxHQUFHLEVBQUU7SUFDaEcsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLFNBQVMsRUFBRSxTQUFTLEVBQUU7UUFDaEQsR0FBRyxFQUFFLEVBQUUsT0FBTyxFQUFFLGNBQWMsRUFBRSxNQUFNLEVBQUUsV0FBVyxFQUFFO0tBQ3RELENBQUMsQ0FBQztJQUVILElBQUksa0JBQVksQ0FBQyxLQUFLLEVBQUUsZ0JBQWdCLEVBQUU7UUFDeEMsU0FBUyxFQUFFLElBQUk7UUFDZixnQkFBZ0IsRUFBRSxRQUFRLENBQUMsY0FBYztRQUN6QyxzQ0FBc0MsRUFBRSxJQUFJO0tBQzdDLENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsaUJBQWlCLEVBQUU7UUFDOUMsY0FBYyxFQUFFO1lBQ2QsWUFBWSxFQUFFO2dCQUNaLG9DQUFvQztnQkFDcEMsS0FBSzthQUNOO1NBQ0Y7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogIENvcHlyaWdodCAyMDIyIEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIikuIFlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2VcbiAqICB3aXRoIHRoZSBMaWNlbnNlLiBBIGNvcHkgb2YgdGhlIExpY2Vuc2UgaXMgbG9jYXRlZCBhdFxuICpcbiAqICAgICAgaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogIG9yIGluIHRoZSAnbGljZW5zZScgZmlsZSBhY2NvbXBhbnlpbmcgdGhpcyBmaWxlLiBUaGlzIGZpbGUgaXMgZGlzdHJpYnV0ZWQgb24gYW4gJ0FTIElTJyBCQVNJUywgV0lUSE9VVCBXQVJSQU5USUVTXG4gKiAgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZXhwcmVzcyBvciBpbXBsaWVkLiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnNcbiAqICBhbmQgbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKi9cblxuaW1wb3J0ICdAYXdzLWNkay9hc3NlcnQvamVzdCc7XG5pbXBvcnQgKiBhcyBkZWZhdWx0cyBmcm9tICdAYXdzLXNvbHV0aW9ucy1jb25zdHJ1Y3RzL2NvcmUnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gXCJhd3MtY2RrLWxpYlwiO1xuaW1wb3J0IHsgRmFyZ2F0ZVRvU3FzIH0gZnJvbSBcIi4uL2xpYlwiO1xuaW1wb3J0ICogYXMga21zIGZyb20gJ2F3cy1jZGstbGliL2F3cy1rbXMnO1xuaW1wb3J0ICogYXMgc3FzIGZyb20gJ2F3cy1jZGstbGliL2F3cy1zcXMnO1xuaW1wb3J0ICogYXMgZWNzIGZyb20gJ2F3cy1jZGstbGliL2F3cy1lY3MnO1xuXG50ZXN0KCdOZXcgc2VydmljZS9uZXcgcXVldWUsIGRscSwgcHVibGljIEFQSSwgbmV3IFZQQycsICgpID0+IHtcblxuICAvLyBBbiBlbnZpcm9ubWVudCB3aXRoIHJlZ2lvbiBpcyByZXF1aXJlZCB0byBlbmFibGUgbG9nZ2luZyBvbiBhbiBBTEJcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gIGNvbnN0IHB1YmxpY0FwaSA9IHRydWU7XG4gIGNvbnN0IGNsdXN0ZXJOYW1lID0gXCJjdXN0b20tY2x1c3Rlci1uYW1lXCI7XG4gIGNvbnN0IGNvbnRhaW5lck5hbWUgPSBcImN1c3RvbS1jb250YWluZXItbmFtZVwiO1xuICBjb25zdCBzZXJ2aWNlTmFtZSA9IFwiY3VzdG9tLXNlcnZpY2UtbmFtZVwiO1xuICBjb25zdCBxdWV1ZU5hbWUgPSBcImN1c3RvbS1xdWV1ZS1uYW1lXCI7XG4gIGNvbnN0IGZhbWlseU5hbWUgPSBcImZhbWlseS1uYW1lXCI7XG4gIGNvbnN0IGRlYWRMZXR0ZXJRdWV1ZU5hbWUgPSBcImRscVF1ZXVlXCI7XG5cbiAgY29uc3QgY29uc3RydWN0ID0gbmV3IEZhcmdhdGVUb1NxcyhzdGFjaywgJ3Rlc3QtY29uc3RydWN0Jywge1xuICAgIHB1YmxpY0FwaSxcbiAgICBlY3JSZXBvc2l0b3J5QXJuOiBkZWZhdWx0cy5mYWtlRWNyUmVwb0FybixcbiAgICB2cGNQcm9wczogeyBjaWRyOiAnMTcyLjAuMC4wLzE2JyB9LFxuICAgIGNsdXN0ZXJQcm9wczogeyBjbHVzdGVyTmFtZSB9LFxuICAgIGNvbnRhaW5lckRlZmluaXRpb25Qcm9wczogeyBjb250YWluZXJOYW1lIH0sXG4gICAgZmFyZ2F0ZVRhc2tEZWZpbml0aW9uUHJvcHM6IHsgZmFtaWx5OiBmYW1pbHlOYW1lIH0sXG4gICAgZmFyZ2F0ZVNlcnZpY2VQcm9wczogeyBzZXJ2aWNlTmFtZSB9LFxuICAgIHF1ZXVlUHJvcHM6IHsgcXVldWVOYW1lIH0sXG4gICAgZGVhZExldHRlclF1ZXVlUHJvcHM6IHtcbiAgICAgIHF1ZXVlTmFtZTogZGVhZExldHRlclF1ZXVlTmFtZVxuICAgIH0sXG4gICAgcXVldWVQZXJtaXNzaW9uczogWydSZWFkJywgJ1dyaXRlJ11cbiAgfSk7XG5cbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZUxpa2UoXCJBV1M6OkVDUzo6U2VydmljZVwiLCB7XG4gICAgTGF1bmNoVHlwZTogJ0ZBUkdBVEUnLFxuICAgIERlc2lyZWRDb3VudDogMixcbiAgICBEZXBsb3ltZW50Q29uZmlndXJhdGlvbjoge1xuICAgICAgTWF4aW11bVBlcmNlbnQ6IDE1MCxcbiAgICAgIE1pbmltdW1IZWFsdGh5UGVyY2VudDogNzVcbiAgICB9LFxuICAgIFBsYXRmb3JtVmVyc2lvbjogZWNzLkZhcmdhdGVQbGF0Zm9ybVZlcnNpb24uTEFURVNULFxuICB9KTtcblxuICBleHBlY3QoY29uc3RydWN0LnZwYyAhPT0gbnVsbCk7XG4gIGV4cGVjdChjb25zdHJ1Y3Quc2VydmljZSAhPT0gbnVsbCk7XG4gIGV4cGVjdChjb25zdHJ1Y3QuY29udGFpbmVyICE9PSBudWxsKTtcbiAgZXhwZWN0KGNvbnN0cnVjdC5zcXNRdWV1ZSAhPT0gbnVsbCk7XG4gIGV4cGVjdChjb25zdHJ1Y3QuZGVhZExldHRlclF1ZXVlICE9PSBudWxsKTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZShcIkFXUzo6RUNTOjpTZXJ2aWNlXCIsIHtcbiAgICBTZXJ2aWNlTmFtZTogc2VydmljZU5hbWVcbiAgfSk7XG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKFwiQVdTOjpFQ1M6OlRhc2tEZWZpbml0aW9uXCIsIHtcbiAgICBGYW1pbHk6IGZhbWlseU5hbWVcbiAgfSk7XG5cbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZUxpa2UoXCJBV1M6OkVDUzo6Q2x1c3RlclwiLCB7XG4gICAgQ2x1c3Rlck5hbWU6IGNsdXN0ZXJOYW1lXG4gIH0pO1xuXG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKFwiQVdTOjpTUVM6OlF1ZXVlXCIsIHtcbiAgICBRdWV1ZU5hbWU6IHF1ZXVlTmFtZVxuICB9KTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZShcIkFXUzo6U1FTOjpRdWV1ZVwiLCB7XG4gICAgUXVldWVOYW1lOiBkZWFkTGV0dGVyUXVldWVOYW1lXG4gIH0pO1xuXG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKFwiQVdTOjpFQ1M6OlRhc2tEZWZpbml0aW9uXCIsIHtcbiAgICBDb250YWluZXJEZWZpbml0aW9uczogW1xuICAgICAge1xuICAgICAgICBFc3NlbnRpYWw6IHRydWUsXG4gICAgICAgIEltYWdlOiB7XG4gICAgICAgICAgXCJGbjo6Sm9pblwiOiBbXG4gICAgICAgICAgICBcIlwiLFxuICAgICAgICAgICAgW1xuICAgICAgICAgICAgICBcIjEyMzQ1Njc4OTAxMi5ka3IuZWNyLnVzLWVhc3QtMS5cIixcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIFJlZjogXCJBV1M6OlVSTFN1ZmZpeFwiXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIFwiL2Zha2UtcmVwbzpsYXRlc3RcIlxuICAgICAgICAgICAgXVxuICAgICAgICAgIF1cbiAgICAgICAgfSxcbiAgICAgICAgTWVtb3J5UmVzZXJ2YXRpb246IDUxMixcbiAgICAgICAgTmFtZTogY29udGFpbmVyTmFtZSxcbiAgICAgICAgUG9ydE1hcHBpbmdzOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgQ29udGFpbmVyUG9ydDogODA4MCxcbiAgICAgICAgICAgIFByb3RvY29sOiBcInRjcFwiXG4gICAgICAgICAgfVxuICAgICAgICBdXG4gICAgICB9XG4gICAgXVxuICB9KTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZShcIkFXUzo6RUMyOjpWUENcIiwge1xuICAgIENpZHJCbG9jazogJzE3Mi4wLjAuMC8xNidcbiAgfSk7XG5cbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZUxpa2UoXCJBV1M6OklBTTo6UG9saWN5XCIsIHtcbiAgICBQb2xpY3lEb2N1bWVudDoge1xuICAgICAgU3RhdGVtZW50OiBbXG4gICAgICAgIHtcbiAgICAgICAgICBBY3Rpb246IFtcbiAgICAgICAgICAgIFwic3FzOlJlY2VpdmVNZXNzYWdlXCIsXG4gICAgICAgICAgICBcInNxczpDaGFuZ2VNZXNzYWdlVmlzaWJpbGl0eVwiLFxuICAgICAgICAgICAgXCJzcXM6R2V0UXVldWVVcmxcIixcbiAgICAgICAgICAgIFwic3FzOkRlbGV0ZU1lc3NhZ2VcIixcbiAgICAgICAgICAgIFwic3FzOkdldFF1ZXVlQXR0cmlidXRlc1wiXG4gICAgICAgICAgXSxcbiAgICAgICAgICBFZmZlY3Q6IFwiQWxsb3dcIixcbiAgICAgICAgICBSZXNvdXJjZToge1xuICAgICAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICAgICAgXCJ0ZXN0Y29uc3RydWN0dGVzdGNvbnN0cnVjdHF1ZXVlNkQxMkM5OUJcIixcbiAgICAgICAgICAgICAgXCJBcm5cIlxuICAgICAgICAgICAgXVxuICAgICAgICAgIH1cbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgXCJzcXM6U2VuZE1lc3NhZ2VcIixcbiAgICAgICAgICAgIFwic3FzOkdldFF1ZXVlQXR0cmlidXRlc1wiLFxuICAgICAgICAgICAgXCJzcXM6R2V0UXVldWVVcmxcIlxuICAgICAgICAgIF0sXG4gICAgICAgICAgRWZmZWN0OiBcIkFsbG93XCIsXG4gICAgICAgICAgUmVzb3VyY2U6IHtcbiAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgIFwidGVzdGNvbnN0cnVjdHRlc3Rjb25zdHJ1Y3RxdWV1ZTZEMTJDOTlCXCIsXG4gICAgICAgICAgICAgIFwiQXJuXCJcbiAgICAgICAgICAgIF1cbiAgICAgICAgICB9XG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH1cbiAgfSk7XG5cbiAgLy8gQ29uZmlybSB3ZSBjcmVhdGVkIGEgUHVibGljL1ByaXZhdGUgVlBDXG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVDMjo6SW50ZXJuZXRHYXRld2F5Jywge30pO1xuICBleHBlY3Qoc3RhY2spLnRvQ291bnRSZXNvdXJjZXMoJ0FXUzo6RUMyOjpWUEMnLCAxKTtcbiAgZXhwZWN0KHN0YWNrKS50b0NvdW50UmVzb3VyY2VzKCdBV1M6OlNRUzo6UXVldWUnLCAyKTtcbiAgZXhwZWN0KHN0YWNrKS50b0NvdW50UmVzb3VyY2VzKCdBV1M6OkVDUzo6U2VydmljZScsIDEpO1xufSk7XG5cbnRlc3QoJ05ldyBzZXJ2aWNlL25ldyBxdWV1ZSwgcHJpdmF0ZSBBUEksIG5ldyBWUEMnLCAoKSA9PiB7XG5cbiAgLy8gQW4gZW52aXJvbm1lbnQgd2l0aCByZWdpb24gaXMgcmVxdWlyZWQgdG8gZW5hYmxlIGxvZ2dpbmcgb24gYW4gQUxCXG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICBjb25zdCBwdWJsaWNBcGkgPSBmYWxzZTtcblxuICBuZXcgRmFyZ2F0ZVRvU3FzKHN0YWNrLCAndGVzdC1jb25zdHJ1Y3QnLCB7XG4gICAgcHVibGljQXBpLFxuICAgIGVjclJlcG9zaXRvcnlBcm46IGRlZmF1bHRzLmZha2VFY3JSZXBvQXJuLFxuICAgIHZwY1Byb3BzOiB7IGNpZHI6ICcxNzIuMC4wLjAvMTYnIH0sXG4gICAgZGVwbG95RGVhZExldHRlclF1ZXVlOiBmYWxzZSxcbiAgICBxdWV1ZVBlcm1pc3Npb25zOiBbJ1JlYWQnXVxuICB9KTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZShcIkFXUzo6RUNTOjpTZXJ2aWNlXCIsIHtcbiAgICBMYXVuY2hUeXBlOiAnRkFSR0FURScsXG4gICAgRGVzaXJlZENvdW50OiAyLFxuICAgIERlcGxveW1lbnRDb25maWd1cmF0aW9uOiB7XG4gICAgICBNYXhpbXVtUGVyY2VudDogMTUwLFxuICAgICAgTWluaW11bUhlYWx0aHlQZXJjZW50OiA3NVxuICAgIH0sXG4gICAgUGxhdGZvcm1WZXJzaW9uOiBlY3MuRmFyZ2F0ZVBsYXRmb3JtVmVyc2lvbi5MQVRFU1QsXG4gIH0pO1xuXG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKFwiQVdTOjpTUVM6OlF1ZXVlXCIsIHtcbiAgICBLbXNNYXN0ZXJLZXlJZDogXCJhbGlhcy9hd3Mvc3FzXCJcbiAgfSk7XG5cbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZUxpa2UoXCJBV1M6OkVDMjo6VlBDXCIsIHtcbiAgICBDaWRyQmxvY2s6ICcxNzIuMC4wLjAvMTYnXG4gIH0pO1xuXG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKFwiQVdTOjpJQU06OlBvbGljeVwiLCB7XG4gICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICB7XG4gICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICBcInNxczpSZWNlaXZlTWVzc2FnZVwiLFxuICAgICAgICAgICAgXCJzcXM6Q2hhbmdlTWVzc2FnZVZpc2liaWxpdHlcIixcbiAgICAgICAgICAgIFwic3FzOkdldFF1ZXVlVXJsXCIsXG4gICAgICAgICAgICBcInNxczpEZWxldGVNZXNzYWdlXCIsXG4gICAgICAgICAgICBcInNxczpHZXRRdWV1ZUF0dHJpYnV0ZXNcIlxuICAgICAgICAgIF0sXG4gICAgICAgICAgRWZmZWN0OiBcIkFsbG93XCIsXG4gICAgICAgICAgUmVzb3VyY2U6IHtcbiAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgIFwidGVzdGNvbnN0cnVjdHRlc3Rjb25zdHJ1Y3RxdWV1ZTZEMTJDOTlCXCIsXG4gICAgICAgICAgICAgIFwiQXJuXCJcbiAgICAgICAgICAgIF1cbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIF0sXG4gICAgfVxuICB9KTtcblxuICAvLyBDb25maXJtIHdlIGNyZWF0ZWQgYW4gSXNvbGF0ZWQgVlBDXG4gIGV4cGVjdChzdGFjaykubm90LnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpFQzI6OkludGVybmV0R2F0ZXdheScsIHt9KTtcbiAgZXhwZWN0KHN0YWNrKS50b0NvdW50UmVzb3VyY2VzKCdBV1M6OkVDMjo6VlBDJywgMSk7XG4gIGV4cGVjdChzdGFjaykudG9Db3VudFJlc291cmNlcygnQVdTOjpTUVM6OlF1ZXVlJywgMSk7XG4gIGV4cGVjdChzdGFjaykudG9Db3VudFJlc291cmNlcygnQVdTOjpFQ1M6OlNlcnZpY2UnLCAxKTtcbn0pO1xuXG50ZXN0KCdOZXcgc2VydmljZS9leGlzdGluZyBmaWZvIHF1ZXVlLCBwcml2YXRlIEFQSSwgZXhpc3RpbmcgVlBDJywgKCkgPT4ge1xuICAvLyBBbiBlbnZpcm9ubWVudCB3aXRoIHJlZ2lvbiBpcyByZXF1aXJlZCB0byBlbmFibGUgbG9nZ2luZyBvbiBhbiBBTEJcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gIGNvbnN0IHB1YmxpY0FwaSA9IGZhbHNlO1xuICBjb25zdCBxdWV1ZU5hbWUgPSAnY3VzdG9tLXF1ZXVlLW5hbWUuZmlmbyc7XG5cbiAgY29uc3QgZXhpc3RpbmdWcGMgPSBkZWZhdWx0cy5nZXRUZXN0VnBjKHN0YWNrLCBwdWJsaWNBcGkpO1xuXG4gIGNvbnN0IGV4aXN0aW5nUXVldWUgPSBuZXcgc3FzLlF1ZXVlKHN0YWNrLCAnTXlRdWV1ZScsIHtcbiAgICBxdWV1ZU5hbWUsXG4gICAgZmlmbzogdHJ1ZVxuICB9KTtcblxuICBuZXcgRmFyZ2F0ZVRvU3FzKHN0YWNrLCAndGVzdC1jb25zdHJ1Y3QnLCB7XG4gICAgcHVibGljQXBpLFxuICAgIGV4aXN0aW5nVnBjLFxuICAgIGV4aXN0aW5nUXVldWVPYmo6IGV4aXN0aW5nUXVldWUsXG4gICAgZWNyUmVwb3NpdG9yeUFybjogZGVmYXVsdHMuZmFrZUVjclJlcG9Bcm4sXG4gIH0pO1xuXG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKFwiQVdTOjpFQ1M6OlNlcnZpY2VcIiwge1xuICAgIExhdW5jaFR5cGU6ICdGQVJHQVRFJyxcbiAgICBEZXNpcmVkQ291bnQ6IDIsXG4gICAgRGVwbG95bWVudENvbmZpZ3VyYXRpb246IHtcbiAgICAgIE1heGltdW1QZXJjZW50OiAxNTAsXG4gICAgICBNaW5pbXVtSGVhbHRoeVBlcmNlbnQ6IDc1XG4gICAgfSxcbiAgICBQbGF0Zm9ybVZlcnNpb246IGVjcy5GYXJnYXRlUGxhdGZvcm1WZXJzaW9uLkxBVEVTVCxcbiAgfSk7XG5cbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZUxpa2UoXCJBV1M6OlNRUzo6UXVldWVcIiwge1xuICAgIFF1ZXVlTmFtZTogcXVldWVOYW1lLFxuICAgIEZpZm9RdWV1ZTogdHJ1ZVxuICB9KTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZShcIkFXUzo6RUMyOjpWUENcIiwge1xuICAgIENpZHJCbG9jazogJzE3Mi4xNjguMC4wLzE2J1xuICB9KTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZShcIkFXUzo6SUFNOjpQb2xpY3lcIiwge1xuICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICBTdGF0ZW1lbnQ6IFtcbiAgICAgICAge1xuICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgXCJzcXM6U2VuZE1lc3NhZ2VcIixcbiAgICAgICAgICAgIFwic3FzOkdldFF1ZXVlQXR0cmlidXRlc1wiLFxuICAgICAgICAgICAgXCJzcXM6R2V0UXVldWVVcmxcIlxuICAgICAgICAgIF0sXG4gICAgICAgICAgRWZmZWN0OiBcIkFsbG93XCIsXG4gICAgICAgICAgUmVzb3VyY2U6IHtcbiAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgIFwiTXlRdWV1ZUU2Q0E2MjM1XCIsXG4gICAgICAgICAgICAgIFwiQXJuXCJcbiAgICAgICAgICAgIF1cbiAgICAgICAgICB9XG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH1cbiAgfSk7XG5cbiAgLy8gQ29uZmlybSB3ZSBjcmVhdGVkIGFuIElzb2xhdGVkIFZQQ1xuICBleHBlY3Qoc3RhY2spLm5vdC50b0hhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RUMyOjpJbnRlcm5ldEdhdGV3YXknLCB7fSk7XG4gIGV4cGVjdChzdGFjaykudG9Db3VudFJlc291cmNlcygnQVdTOjpFQzI6OlZQQycsIDEpO1xuICBleHBlY3Qoc3RhY2spLnRvQ291bnRSZXNvdXJjZXMoJ0FXUzo6U1FTOjpRdWV1ZScsIDEpO1xuICBleHBlY3Qoc3RhY2spLnRvQ291bnRSZXNvdXJjZXMoJ0FXUzo6RUNTOjpTZXJ2aWNlJywgMSk7XG59KTtcblxudGVzdCgnRXhpc3Rpbmcgc2VydmljZS9uZXcgcXVldWUsIHB1YmxpYyBBUEksIGV4aXN0aW5nIFZQQycsICgpID0+IHtcbiAgLy8gQW4gZW52aXJvbm1lbnQgd2l0aCByZWdpb24gaXMgcmVxdWlyZWQgdG8gZW5hYmxlIGxvZ2dpbmcgb24gYW4gQUxCXG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICBjb25zdCBwdWJsaWNBcGkgPSB0cnVlO1xuICBjb25zdCBzZXJ2aWNlTmFtZSA9ICdjdXN0b20tbmFtZSc7XG4gIGNvbnN0IGN1c3RvbVZhcmlhYmxlTmFtZSA9ICdDVVNUT01fTkFNRSc7XG4gIGNvbnN0IGN1c3RvbUFybk5hbWUgPSAnQ1VTVE9NX0FSTl9OQU1FJztcbiAgY29uc3QgcXVldWVOYW1lID0gJ3Rlc3RRdWV1ZSc7XG4gIGNvbnN0IGRscU5hbWUgPSAnZGxxUXVldWUnO1xuXG4gIGNvbnN0IGV4aXN0aW5nVnBjID0gZGVmYXVsdHMuZ2V0VGVzdFZwYyhzdGFjayk7XG5cbiAgY29uc3QgW3Rlc3RTZXJ2aWNlLCB0ZXN0Q29udGFpbmVyXSA9IGRlZmF1bHRzLkNyZWF0ZUZhcmdhdGVTZXJ2aWNlKHN0YWNrLFxuICAgICd0ZXN0JyxcbiAgICBleGlzdGluZ1ZwYyxcbiAgICB1bmRlZmluZWQsXG4gICAgZGVmYXVsdHMuZmFrZUVjclJlcG9Bcm4sXG4gICAgdW5kZWZpbmVkLFxuICAgIHVuZGVmaW5lZCxcbiAgICB1bmRlZmluZWQsXG4gICAgeyBzZXJ2aWNlTmFtZSB9KTtcblxuICBuZXcgRmFyZ2F0ZVRvU3FzKHN0YWNrLCAndGVzdC1jb25zdHJ1Y3QnLCB7XG4gICAgcHVibGljQXBpLFxuICAgIGV4aXN0aW5nRmFyZ2F0ZVNlcnZpY2VPYmplY3Q6IHRlc3RTZXJ2aWNlLFxuICAgIGV4aXN0aW5nQ29udGFpbmVyRGVmaW5pdGlvbk9iamVjdDogdGVzdENvbnRhaW5lcixcbiAgICBleGlzdGluZ1ZwYyxcbiAgICBxdWV1ZVVybEVudmlyb25tZW50VmFyaWFibGVOYW1lOiBjdXN0b21WYXJpYWJsZU5hbWUsXG4gICAgcXVldWVBcm5FbnZpcm9ubWVudFZhcmlhYmxlTmFtZTogY3VzdG9tQXJuTmFtZSxcbiAgICBxdWV1ZVByb3BzOiB7XG4gICAgICBxdWV1ZU5hbWVcbiAgICB9LFxuICAgIGRlYWRMZXR0ZXJRdWV1ZVByb3BzOiB7XG4gICAgICBxdWV1ZU5hbWU6IGRscU5hbWVcbiAgICB9XG4gIH0pO1xuXG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKFwiQVdTOjpFQ1M6OlNlcnZpY2VcIiwge1xuICAgIFNlcnZpY2VOYW1lOiBzZXJ2aWNlTmFtZVxuICB9KTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZShcIkFXUzo6RUNTOjpUYXNrRGVmaW5pdGlvblwiLCB7XG4gICAgQ29udGFpbmVyRGVmaW5pdGlvbnM6IFtcbiAgICAgIHtcbiAgICAgICAgRW52aXJvbm1lbnQ6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBOYW1lOiBjdXN0b21Bcm5OYW1lLFxuICAgICAgICAgICAgVmFsdWU6IHtcbiAgICAgICAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICAgICAgICBcInRlc3Rjb25zdHJ1Y3R0ZXN0Y29uc3RydWN0cXVldWU2RDEyQzk5QlwiLFxuICAgICAgICAgICAgICAgIFwiQXJuXCJcbiAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgfVxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgTmFtZTogY3VzdG9tVmFyaWFibGVOYW1lLFxuICAgICAgICAgICAgVmFsdWU6IHtcbiAgICAgICAgICAgICAgUmVmOiBcInRlc3Rjb25zdHJ1Y3R0ZXN0Y29uc3RydWN0cXVldWU2RDEyQzk5QlwiXG4gICAgICAgICAgICB9XG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgICAgRXNzZW50aWFsOiB0cnVlLFxuICAgICAgICBJbWFnZToge1xuICAgICAgICAgIFwiRm46OkpvaW5cIjogW1xuICAgICAgICAgICAgXCJcIixcbiAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgXCIxMjM0NTY3ODkwMTIuZGtyLmVjci51cy1lYXN0LTEuXCIsXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBSZWY6IFwiQVdTOjpVUkxTdWZmaXhcIlxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICBcIi9mYWtlLXJlcG86bGF0ZXN0XCJcbiAgICAgICAgICAgIF1cbiAgICAgICAgICBdXG4gICAgICAgIH0sXG4gICAgICAgIE1lbW9yeVJlc2VydmF0aW9uOiA1MTIsXG4gICAgICAgIE5hbWU6IFwidGVzdC1jb250YWluZXJcIixcbiAgICAgICAgUG9ydE1hcHBpbmdzOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgQ29udGFpbmVyUG9ydDogODA4MCxcbiAgICAgICAgICAgIFByb3RvY29sOiBcInRjcFwiXG4gICAgICAgICAgfVxuICAgICAgICBdXG4gICAgICB9XG4gICAgXVxuICB9KTtcbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZUxpa2UoXCJBV1M6OlNRUzo6UXVldWVcIiwge1xuICAgIFF1ZXVlTmFtZTogcXVldWVOYW1lXG4gIH0pO1xuXG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKFwiQVdTOjpTUVM6OlF1ZXVlXCIsIHtcbiAgICBRdWV1ZU5hbWU6IGRscU5hbWVcbiAgfSk7XG5cbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZUxpa2UoXCJBV1M6OkVDMjo6VlBDXCIsIHtcbiAgICBDaWRyQmxvY2s6ICcxNzIuMTY4LjAuMC8xNidcbiAgfSk7XG5cbiAgLy8gQ29uZmlybSB3ZSBjcmVhdGVkIGEgUHVibGljL1ByaXZhdGUgVlBDXG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVDMjo6SW50ZXJuZXRHYXRld2F5Jywge30pO1xuICBleHBlY3Qoc3RhY2spLnRvQ291bnRSZXNvdXJjZXMoJ0FXUzo6RUMyOjpWUEMnLCAxKTtcbiAgZXhwZWN0KHN0YWNrKS50b0NvdW50UmVzb3VyY2VzKCdBV1M6OlNRUzo6UXVldWUnLCAyKTtcbiAgZXhwZWN0KHN0YWNrKS50b0NvdW50UmVzb3VyY2VzKCdBV1M6OkVDUzo6U2VydmljZScsIDEpO1xufSk7XG5cbi8vIFRlc3QgZXhpc3Rpbmcgc2VydmljZS9leGlzdGluZyBxdWV1ZSwgcHJpdmF0ZSBBUEksIG5ldyBWUENcbnRlc3QoJ0V4aXN0aW5nIHNlcnZpY2UvZXhpc3RpbmcgcXVldWUsIHByaXZhdGUgQVBJLCBleGlzdGluZyBWUEMnLCAoKSA9PiB7XG4gIC8vIEFuIGVudmlyb25tZW50IHdpdGggcmVnaW9uIGlzIHJlcXVpcmVkIHRvIGVuYWJsZSBsb2dnaW5nIG9uIGFuIEFMQlxuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgY29uc3QgcHVibGljQXBpID0gZmFsc2U7XG4gIGNvbnN0IHNlcnZpY2VOYW1lID0gJ2N1c3RvbS1uYW1lJztcbiAgY29uc3QgcXVldWVOYW1lID0gJ2N1c3RvbS1xdWV1ZS1uYW1lJztcblxuICBjb25zdCBleGlzdGluZ1ZwYyA9IGRlZmF1bHRzLmdldFRlc3RWcGMoc3RhY2ssIHB1YmxpY0FwaSk7XG5cbiAgY29uc3QgW3Rlc3RTZXJ2aWNlLCB0ZXN0Q29udGFpbmVyXSA9IGRlZmF1bHRzLkNyZWF0ZUZhcmdhdGVTZXJ2aWNlKHN0YWNrLFxuICAgICd0ZXN0JyxcbiAgICBleGlzdGluZ1ZwYyxcbiAgICB1bmRlZmluZWQsXG4gICAgZGVmYXVsdHMuZmFrZUVjclJlcG9Bcm4sXG4gICAgdW5kZWZpbmVkLFxuICAgIHVuZGVmaW5lZCxcbiAgICB1bmRlZmluZWQsXG4gICAgeyBzZXJ2aWNlTmFtZSB9KTtcblxuICBjb25zdCBleGlzdGluZ1F1ZXVlID0gbmV3IHNxcy5RdWV1ZShzdGFjaywgJ015UXVldWUnLCB7XG4gICAgcXVldWVOYW1lXG4gIH0pO1xuXG4gIG5ldyBGYXJnYXRlVG9TcXMoc3RhY2ssICd0ZXN0LWNvbnN0cnVjdCcsIHtcbiAgICBwdWJsaWNBcGksXG4gICAgZXhpc3RpbmdGYXJnYXRlU2VydmljZU9iamVjdDogdGVzdFNlcnZpY2UsXG4gICAgZXhpc3RpbmdDb250YWluZXJEZWZpbml0aW9uT2JqZWN0OiB0ZXN0Q29udGFpbmVyLFxuICAgIGV4aXN0aW5nVnBjLFxuICAgIGV4aXN0aW5nUXVldWVPYmo6IGV4aXN0aW5nUXVldWVcbiAgfSk7XG5cbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZUxpa2UoXCJBV1M6OkVDUzo6U2VydmljZVwiLCB7XG4gICAgU2VydmljZU5hbWU6IHNlcnZpY2VOYW1lLFxuICB9KTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZShcIkFXUzo6RUNTOjpUYXNrRGVmaW5pdGlvblwiLCB7XG4gICAgQ29udGFpbmVyRGVmaW5pdGlvbnM6IFtcbiAgICAgIHtcbiAgICAgICAgRW52aXJvbm1lbnQ6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBOYW1lOiBcIlNRU19RVUVVRV9BUk5cIixcbiAgICAgICAgICAgIFZhbHVlOiB7XG4gICAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgICAgXCJNeVF1ZXVlRTZDQTYyMzVcIixcbiAgICAgICAgICAgICAgICBcIkFyblwiXG4gICAgICAgICAgICAgIF1cbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIE5hbWU6IFwiU1FTX1FVRVVFX1VSTFwiLFxuICAgICAgICAgICAgVmFsdWU6IHtcbiAgICAgICAgICAgICAgUmVmOiBcIk15UXVldWVFNkNBNjIzNVwiXG4gICAgICAgICAgICB9XG4gICAgICAgICAgfVxuICAgICAgICBdLFxuICAgICAgICBFc3NlbnRpYWw6IHRydWUsXG4gICAgICAgIEltYWdlOiB7XG4gICAgICAgICAgXCJGbjo6Sm9pblwiOiBbXG4gICAgICAgICAgICBcIlwiLFxuICAgICAgICAgICAgW1xuICAgICAgICAgICAgICBcIjEyMzQ1Njc4OTAxMi5ka3IuZWNyLnVzLWVhc3QtMS5cIixcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIFJlZjogXCJBV1M6OlVSTFN1ZmZpeFwiXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIFwiL2Zha2UtcmVwbzpsYXRlc3RcIlxuICAgICAgICAgICAgXVxuICAgICAgICAgIF1cbiAgICAgICAgfSxcbiAgICAgICAgTWVtb3J5UmVzZXJ2YXRpb246IDUxMixcbiAgICAgICAgTmFtZTogXCJ0ZXN0LWNvbnRhaW5lclwiLFxuICAgICAgICBQb3J0TWFwcGluZ3M6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBDb250YWluZXJQb3J0OiA4MDgwLFxuICAgICAgICAgICAgUHJvdG9jb2w6IFwidGNwXCJcbiAgICAgICAgICB9XG4gICAgICAgIF1cbiAgICAgIH1cbiAgICBdXG4gIH0pO1xuXG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKFwiQVdTOjpTUVM6OlF1ZXVlXCIsIHtcbiAgICBRdWV1ZU5hbWU6IHF1ZXVlTmFtZVxuICB9KTtcbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZUxpa2UoXCJBV1M6OkVDMjo6VlBDXCIsIHtcbiAgICBDaWRyQmxvY2s6ICcxNzIuMTY4LjAuMC8xNidcbiAgfSk7XG5cbiAgLy8gQ29uZmlybSB3ZSBjcmVhdGVkIGFuIElzb2xhdGVkIFZQQ1xuICBleHBlY3Qoc3RhY2spLm5vdC50b0hhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RUMyOjpJbnRlcm5ldEdhdGV3YXknLCB7fSk7XG4gIGV4cGVjdChzdGFjaykudG9Db3VudFJlc291cmNlcygnQVdTOjpFQzI6OlZQQycsIDEpO1xuICBleHBlY3Qoc3RhY2spLnRvQ291bnRSZXNvdXJjZXMoJ0FXUzo6U1FTOjpRdWV1ZScsIDEpO1xuICBleHBlY3Qoc3RhY2spLnRvQ291bnRSZXNvdXJjZXMoJ0FXUzo6RUNTOjpTZXJ2aWNlJywgMSk7XG59KTtcblxudGVzdCgnVGVzdCBiYWQgcXVldWVQZXJtaXNzaW9ucycsICgpID0+IHtcblxuICAvLyBBbiBlbnZpcm9ubWVudCB3aXRoIHJlZ2lvbiBpcyByZXF1aXJlZCB0byBlbmFibGUgbG9nZ2luZyBvbiBhbiBBTEJcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gIGNvbnN0IHB1YmxpY0FwaSA9IGZhbHNlO1xuXG4gIGNvbnN0IHByb3BzID0ge1xuICAgIHB1YmxpY0FwaSxcbiAgICBlY3JSZXBvc2l0b3J5QXJuOiBkZWZhdWx0cy5mYWtlRWNyUmVwb0FybixcbiAgICB2cGNQcm9wczogeyBjaWRyOiAnMTcyLjAuMC4wLzE2JyB9LFxuICAgIGRlcGxveURlYWRMZXR0ZXJRdWV1ZTogZmFsc2UsXG4gICAgcXVldWVQZXJtaXNzaW9uczogWydSZWVkJ10sXG4gIH07XG5cbiAgY29uc3QgYXBwID0gKCkgPT4ge1xuICAgIG5ldyBGYXJnYXRlVG9TcXMoc3RhY2ssICd0ZXN0LWNvbnN0cnVjdCcsIHByb3BzKTtcbiAgfTtcblxuICBleHBlY3QoYXBwKS50b1Rocm93RXJyb3IoJ0ludmFsaWQgcXVldWUgcGVybWlzc2lvbiBzdWJtaXR0ZWQgLSBSZWVkJyk7XG59KTtcblxudGVzdCgnUXVldWUgaXMgZW5jcnlwdGVkIHdpdGggaW1wb3J0ZWQgQ01LIHdoZW4gc2V0IG9uIGVuY3J5cHRpb25LZXkgcHJvcCcsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKHVuZGVmaW5lZCwgdW5kZWZpbmVkLCB7XG4gICAgZW52OiB7IGFjY291bnQ6IFwiMTIzNDU2Nzg5MDEyXCIsIHJlZ2lvbjogJ3VzLWVhc3QtMScgfSxcbiAgfSk7XG5cbiAgY29uc3QgY21rID0gbmV3IGttcy5LZXkoc3RhY2ssICdjbWsnKTtcbiAgbmV3IEZhcmdhdGVUb1NxcyhzdGFjaywgJ3Rlc3QtY29uc3RydWN0Jywge1xuICAgIHB1YmxpY0FwaTogdHJ1ZSxcbiAgICBlY3JSZXBvc2l0b3J5QXJuOiBkZWZhdWx0cy5mYWtlRWNyUmVwb0FybixcbiAgICBlbmNyeXB0aW9uS2V5OiBjbWtcbiAgfSk7XG5cbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZShcIkFXUzo6U1FTOjpRdWV1ZVwiLCB7XG4gICAgS21zTWFzdGVyS2V5SWQ6IHtcbiAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgIFwiY21rMDFERTAzREFcIixcbiAgICAgICAgXCJBcm5cIlxuICAgICAgXVxuICAgIH1cbiAgfSk7XG59KTtcblxudGVzdCgnUXVldWUgaXMgZW5jcnlwdGVkIHdpdGggaW1wb3J0ZWQgQ01LIHdoZW4gc2V0IG9uIHF1ZXVlUHJvcHMuZW5jcnlwdGlvbk1hc3RlcktleSBwcm9wJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2sodW5kZWZpbmVkLCB1bmRlZmluZWQsIHtcbiAgICBlbnY6IHsgYWNjb3VudDogXCIxMjM0NTY3ODkwMTJcIiwgcmVnaW9uOiAndXMtZWFzdC0xJyB9LFxuICB9KTtcblxuICBjb25zdCBjbWsgPSBuZXcga21zLktleShzdGFjaywgJ2NtaycpO1xuICBuZXcgRmFyZ2F0ZVRvU3FzKHN0YWNrLCAndGVzdC1jb25zdHJ1Y3QnLCB7XG4gICAgcHVibGljQXBpOiB0cnVlLFxuICAgIGVjclJlcG9zaXRvcnlBcm46IGRlZmF1bHRzLmZha2VFY3JSZXBvQXJuLFxuICAgIHF1ZXVlUHJvcHM6IHtcbiAgICAgIGVuY3J5cHRpb25NYXN0ZXJLZXk6IGNta1xuICAgIH1cbiAgfSk7XG5cbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZShcIkFXUzo6U1FTOjpRdWV1ZVwiLCB7XG4gICAgS21zTWFzdGVyS2V5SWQ6IHtcbiAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgIFwiY21rMDFERTAzREFcIixcbiAgICAgICAgXCJBcm5cIlxuICAgICAgXVxuICAgIH1cbiAgfSk7XG59KTtcblxudGVzdCgnUXVldWUgaXMgZW5jcnlwdGVkIHdpdGggcHJvdmlkZWQgZW5jcnl0aW9uS2V5UHJvcHMnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjayh1bmRlZmluZWQsIHVuZGVmaW5lZCwge1xuICAgIGVudjogeyBhY2NvdW50OiBcIjEyMzQ1Njc4OTAxMlwiLCByZWdpb246ICd1cy1lYXN0LTEnIH0sXG4gIH0pO1xuXG4gIG5ldyBGYXJnYXRlVG9TcXMoc3RhY2ssICd0ZXN0LWNvbnN0cnVjdCcsIHtcbiAgICBwdWJsaWNBcGk6IHRydWUsXG4gICAgZWNyUmVwb3NpdG9yeUFybjogZGVmYXVsdHMuZmFrZUVjclJlcG9Bcm4sXG4gICAgZW5jcnlwdGlvbktleVByb3BzOiB7XG4gICAgICBhbGlhczogJ25ldy1rZXktYWxpYXMtZnJvbS1wcm9wcydcbiAgICB9XG4gIH0pO1xuXG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoXCJBV1M6OlNRUzo6UXVldWVcIiwge1xuICAgIEttc01hc3RlcktleUlkOiBcImFsaWFzL2F3cy9zcXNcIlxuICB9KTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OktNUzo6QWxpYXMnLCB7XG4gICAgQWxpYXNOYW1lOiAnYWxpYXMvbmV3LWtleS1hbGlhcy1mcm9tLXByb3BzJyxcbiAgICBUYXJnZXRLZXlJZDoge1xuICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICd0ZXN0Y29uc3RydWN0RW5jcnlwdGlvbktleTYxNTNCMDUzJyxcbiAgICAgICAgJ0FybidcbiAgICAgIF1cbiAgICB9XG4gIH0pO1xufSk7XG5cbnRlc3QoJ1F1ZXVlIGlzIGVuY3J5cHRlZCB3aXRoIFNRUy1tYW5hZ2VkIEtNUyBrZXkgd2hlbiBubyBvdGhlciBlbmNyeXB0aW9uIHByb3ByZXRpZXMgYXJlIHNldCcsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKHVuZGVmaW5lZCwgdW5kZWZpbmVkLCB7XG4gICAgZW52OiB7IGFjY291bnQ6IFwiMTIzNDU2Nzg5MDEyXCIsIHJlZ2lvbjogJ3VzLWVhc3QtMScgfSxcbiAgfSk7XG5cbiAgbmV3IEZhcmdhdGVUb1NxcyhzdGFjaywgJ3Rlc3QtY29uc3RydWN0Jywge1xuICAgIHB1YmxpY0FwaTogdHJ1ZSxcbiAgICBlY3JSZXBvc2l0b3J5QXJuOiBkZWZhdWx0cy5mYWtlRWNyUmVwb0FybixcbiAgfSk7XG5cbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZShcIkFXUzo6U1FTOjpRdWV1ZVwiLCB7XG4gICAgS21zTWFzdGVyS2V5SWQ6IFwiYWxpYXMvYXdzL3Nxc1wiXG4gIH0pO1xufSk7XG5cbnRlc3QoJ1F1ZXVlIGlzIGVuY3J5cHRlZCB3aXRoIGN1c3RvbWVyIG1hbmFnZWQgS01TIEtleSB3aGVuIGVuYWJsZSBlbmNyeXB0aW9uIGZsYWcgaXMgdHJ1ZScsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKHVuZGVmaW5lZCwgdW5kZWZpbmVkLCB7XG4gICAgZW52OiB7IGFjY291bnQ6IFwiMTIzNDU2Nzg5MDEyXCIsIHJlZ2lvbjogJ3VzLWVhc3QtMScgfSxcbiAgfSk7XG5cbiAgbmV3IEZhcmdhdGVUb1NxcyhzdGFjaywgJ3Rlc3QtY29uc3RydWN0Jywge1xuICAgIHB1YmxpY0FwaTogdHJ1ZSxcbiAgICBlY3JSZXBvc2l0b3J5QXJuOiBkZWZhdWx0cy5mYWtlRWNyUmVwb0FybixcbiAgICBlbmFibGVFbmNyeXB0aW9uV2l0aEN1c3RvbWVyTWFuYWdlZEtleTogdHJ1ZVxuICB9KTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKFwiQVdTOjpTUVM6OlF1ZXVlXCIsIHtcbiAgICBLbXNNYXN0ZXJLZXlJZDoge1xuICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgXCJ0ZXN0Y29uc3RydWN0RW5jcnlwdGlvbktleTYxNTNCMDUzXCIsXG4gICAgICAgIFwiQXJuXCJcbiAgICAgIF1cbiAgICB9XG4gIH0pO1xufSk7Il19