# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['nvelope']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'nvelope',
    'version': '0.3.1',
    'description': 'Painless JSON marshalling and unmarshalling',
    'long_description': '[![codecov](https://codecov.io/gh/monomonedula/nvelope/branch/master/graph/badge.svg?token=yunFiDdUEK)](https://codecov.io/gh/monomonedula/nvelope)\n[![Build Status](https://app.travis-ci.com/monomonedula/nvelope.svg?branch=master)](https://app.travis-ci.com/monomonedula/nvelope)\n# nvelope\n\nDefine your JSON schema as Python dataclasses\n\n## Installation\n`pip install nvelope`\n\n\n## The problem it solves\n\nThis is basically sommething like JSON-schema, but it works \nwith static type checking, since the classes you define are just regular\npython dataclasses which can (and should) be type checked with `mypy` library.\n\n\nIt also lets not to just define the structure of your JSON data in a\nsingle place in your\npython code, but also to define\ncustom checks and conversions from/to JSON for any type you want.\n\n### Original use case\nSay you have two\nmicroservices communicating via JSON messages, both written in python.\n\nYou may define a shared package with the messages definition \nand use the model\'s `.as_json()` method on one end to serialize the message\nand `.from_json()` on the other to convert it into a DTO, \nchecking and modifying the fields and their values along\nthe way exactly how you defined it in a single place.\n\nCombining this with static type checking (and maybe some unit tests)\nyou can ensure that any message one microservice can send,\nthe other can read as long as they use the same model to pack/unpack their JSONs.\n\n## Usage\n\nSay you have a JSON representing a user in your app looking something like this\n```json\n{\n    "id": 530716139,\n    "username": "johndoe",\n    "language_code": "en"\n}\n```\n\nYou define an envelope for it\n\n```python\nfrom dataclasses import dataclass\nfrom typing import Optional\n\nfrom nvelope import (Obj, int_conv, string_conv)\n\n@dataclass      # note the @dataclass decorator, it is important\nclass User(Obj):\n    _conversion = {\n        "id": int_conv,\n        "language_code": string_conv,\n        "username": string_conv,\n    }\n\n    id: int\n    language_code: str\n    username: str\n\n```\n\n\nNow you have a model that knows how to read data from the JSON \n(not the raw string, actually, but to the types that are allowed by the\nstandard `json.dumps` function e.g. `dict`, `list`, `str`, `int`, `float`, `bool`, `None` ) ...\n\n```python\n\nuser = User.from_json(\n    {\n        "id": 530716139,\n        "username": "johndoe",\n        "language_code": "en"\n    }\n)\n```\n... and knows how to convert itself into JSON \n\n```python\nUser(\n    id=530716139,\n    username="johndoe",\n    language_code="en",\n).as_json() \n\n# returns a dictionary {\n#     "id": 530716139,\n#     "username": "johndoe",\n#     "language_code": "en"\n# }\n```\n\n\n### Compound envelopes\nYou can also define compound envelopes.\n\nSay we want to define a message and include info about the sender.\nHaving defined the `User` envelope, we can do it like this:\n\n```python\n\nfrom nvelope import CompoundConv\n\n@dataclass\nclass Message(Obj):\n    _conversion = {\n        "message_id": int_conv,\n        "from_": CompoundConv(User),\n        "text": string_conv,\n    }\n\n    from_: User\n    text: str\n    message_id: int\n```\nand use it the same way:\n\n```python\n# reading an obj from json like this\n\nMessage.from_json(\n    {\n        "message_id": 44,\n        "text": "hello there",\n        "from_": {\n            "id": 530716139,\n            "username": "johndoe",\n            "language_code": "en"\n        }\n    }\n)\n\n# and dumping an object to json like this\nMessage(\n    message_id=44,\n    text="whatever",\n    from_=User(\n        id=530716139,\n        username="johndoe",\n        language_code="en",\n    )\n).as_json()\n```\n\n\n### Arrays\n\nThis is how you define arrays:\n\n```python\nfrom nvelope import Arr\n\n\nclass Users(Arr):\n    conversion = CompoundConv(User)\n\n\n# Same API inherited from nvelope.Compound interface\n\nUsers.from_json(\n    [\n        {\n            "id": 530716139,\n            "username": "johndoe",\n            "language_code": "en",\n        },\n        {\n            "id": 452341341,\n            "username": "ivandrago",\n            "language_code": "ru",\n        }\n    ]\n)\n\nUsers(\n    [\n        User(\n            id=530716139,\n            username="johndoe",\n            language_code="en",\n        ),\n        User(\n            id=452341341,\n            username="ivandrago",\n            language_code="ru",\n        ),\n    ]\n).as_json()\n```\n\n### Field aliases\n\nAt some point you may need to define an envelope for an API containing certain field names which cannot be\nused in python since they are reserved keywords.\n\nThere\'s a solution for this:\n\n```python\nfrom nvelope import ObjWithAliases\n\n@dataclass\nclass Comment(ObjWithAliases):\n    _conversion = {\n        "text": string_conv,\n        "from_": int_conv,\n    }\n    \n    \n    _alias_to_actual = {\n        "from_": "from",\n    }\n    \n    text: str\n    from_: User\n\n```\n\nIn this case `from` key gets replaced by `from_` in the python model. \n\n### Missing and optional fields\n\nThere\'s a difference between fields that can be set to `None` and fields which may be missing in the JSON at all.\n\nThis is how you specify that a some field may be missing from the JSON and that\'s OK:\n```python\nfrom typing import Optional\n\nfrom nvelope import MaybeMissing\nfrom nvelope import OptionalConv\n\n@dataclass\nclass Comment(ObjWithAliases):\n    _alias_to_actual = {\n        "from_": "from",\n    }\n    \n    text: str\n    img: Optional[str]          # this field can be set to None (null), but is must always be present in the JSON\n    from_: MaybeMissing[User]   # this field can be missing from JSON body\n\n    _conversion = {\n        "text": string_conv,\n        "img": OptionalConv(string_conv),   # note the wrapping with OptionalConv\n        "from_": int_conv,\n    }\n\n```\n\nThis is how you check if the `MaybeMissing` field is actually missing\n```python\ncomment.from_.has()     # returns False if the field is missing\n```\n\nand this is how you get the value:\n```python\ncomment.value()     # raises an error if there\'s no value, \n                    # so it is recommended to check the output of .has()\n                    #  before calling .value() \n```\n\n\n### Custom conversions\n\n\nYou may define a custom conversions inheriting from `nvelope.nvelope.Conversion` abstract base class \nor using `nvelope.nvelope.ConversionOf` class. \n\nFor example, this is how `datetime_iso_format_conv` is defined:\n\n```python\nfrom nvelope import WithTypeCheck, ConversionOf\n\ndatetime_iso_format_conv = WithTypeCheck(\n    datetime.datetime,\n    ConversionOf(\n        to_json=lambda v: v.isoformat(),\n        from_json=lambda s: datetime.datetime.fromisoformat(s),\n    ),\n)\n\n```\n\nSay we want to jsonify a `datetime` field as POSIX timestamp, instead of storing it in ISO string format.\n\n```python\ndatetime_timestamp_conv = ConversionOf(\n    to_json=lambda v: v.timestamp(),\n    from_json=lambda s: datetime.datetime.fromtimestamp(s),\n)\n```\n\nWe could also add `WithTypeCheck` wrapper in order to add explicit check that \nthe value passed into `.from_json()`\nis indeed `float`.\n\n```python\ndatetime_timestamp_conv = WithTypeCheck(\n    float,\n    ConversionOf(\n        to_json=lambda v: v.timestamp(),\n        from_json=lambda s: datetime.datetime.fromtimestamp(s),\n    )\n)\n```\n\n',
    'author': 'monomonedula',
    'author_email': 'valh@tuta.io',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/monomonedula/nvelope',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
