# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['camunda',
 'camunda.client',
 'camunda.external_task',
 'camunda.utils',
 'camunda.variables']

package_data = \
{'': ['*']}

install_requires = \
['aiohttp>=3.7.4,<4.0.0']

setup_kwargs = {
    'name': 'avikom-camunda-client',
    'version': '0.8.0',
    'description': 'A python client for Camunda to process external service tasks.',
    'long_description': '# avikom-camunda-client\n\n[![Version](https://img.shields.io/badge/version-0.8.0-orange.svg)](https://github.com/OpenAvikom/camunda-client)\n[![Build Status](https://github.com/OpenAvikom/camunda-client/actions/workflows/python-package.yml/badge.svg)](https://github.com/OpenAvikom/camunda-client/actions/workflows/python-package.yml)\n[![PyPI](https://img.shields.io/pypi/v/avikom-camunda-client.svg)](https://pypi.org/project/avikom-camunda-client)\n[![GitHub commits](https://img.shields.io/github/commits-since/OpenAvikom/camunda-client/0.8.0.svg)](https://github.com/OpenAvikom/camunda-client/compare/0.8.0...master)\n[![License](https://img.shields.io/github/license/OpenAvikom/camunda-client.svg)](LICENSE)\n\n\nThis fork of [yogeshrnaik/camunda-external-task-client-python3](https://github.com/yogeshrnaik/camunda-external-task-client-python3) makes use of `async/await` and operates workers asynchronously rather than threaded. Additionally, it contains some convenience functions that wrap the Camunda REST API and speed up development.\n\n## Installation\n\nThe easiest way to install this client is via `pip`:\n\n```bash\npip install git+https://github.com/OpenAvikom/camunda-external-task-client-python3\n```\n\nIf you want to run the examples, it might be easier to first clone the project and run everything locally:\n\n```bash\ngit clone https://github.com/OpenAvikom/camunda-external-task-client-python3.git\ncd camunda-external-task-client-python3\npip install -e .   # install module editable \n```\n\n## Hello World Example\n\nThis example will show you how to let a Python client communicate with Camunda.\n\nFirs, we assume that Camunda is up and running. If that\'s not the case, you might want to have a look at the [Camunda Platform Docker Images](https://github.com/camunda/docker-camunda-bpm-platform).\nSecond, we assume that Camunda is running at [localhost:8080](localhost:8080). If not, you need to adapt the example below accordingly.\nThird, you need a way to deploy and run a BPMN process. The [Camunda Modeler](https://camunda.com/de/download/modeler/) is probably a good point of departure.\n\n### The Model\n\nThe file [`bpmn_process/hello_world.bpmn`](bpmn_process/hello_world.bpmn) contains a very simple model:\n\n\n![](img/hello_world.png)\n\nWe have a start event (left circle), a ServiceTask to conduct and an end event (right circle). When you hit the \'Start Current Diagram\' (the \'play\' button in the modeler) nothing much will happen.\nThe process will be running though. You can check [localhost:8080](localhost:8080) to make sure.\n\nNot let\'s have a closer look at the \'Hello World\' external task (the square symbol with the cogs at the top right).\nThe interesting fields in the property panel are `Implementation` and `Topic`:\n\n![](img/modeler_properties.png)\n\n`Implementation` tells Camunda, that the task should be resolved by an `External` worker and `Topic` is a string that more or less describes the task to be conducted.\nWe see that Camunda is now waiting for an external worker that is capable of conducting a task with the topic `HelloWorldTask`.\nThe Camunda Cockpit will show you a running process waiting for `HelloWorld` to be completed by an external process.\nNow let\'s create a Python client which will subscribe to that topic and do tasks by just returning a success event.\nThe file can be found in [`examples/hello_world.py`](examples/hello_world.py):\n\n```python\nimport aiohttp\nimport asyncio\n\nfrom camunda.external_task.external_task import ExternalTask\nfrom camunda.external_task.external_task_worker import ExternalTaskWorker\nfrom camunda.external_task.external_task_result import ExternalTaskResult\n\n\nasync def main():\n    # let\'s create an async http context with aiohttp\n    # aiohttp will close the connection when the worker returns (it won\'t though)\n    async with aiohttp.ClientSession() as session:\n        # We create a worker with a task id and pass the http session as well as the REST endpoint of Camunda.\n        # You need to change \'base_url\' in case your Camunda engine is configured differently.\n        worker = ExternalTaskWorker(\n            worker_id=1, base_url="http://localhost:8080/engine-rest", session=session\n        )\n        print("waiting for a task ...")\n        # Subscribe is an async function which will block until the worker is cancelled with `worker.cancel()`,\n        # In this example, no one will do this. We will stop the program with Ctrl+C instead\n        # When the worker detects a new task for the topic assigned to `topic_name` it will trigger the\n        # function/method passed to `action`.\n        await worker.subscribe(topic_names="HelloWorldTask", action=process)\n\n\n# this will be called when a task for the subscribed topic is available\nasync def process(task: ExternalTask) -> ExternalTaskResult:\n    print("I got a task!")\n    # To communicate the successfull processing of a task, we return an ExternalTaskResult created by `task.complete` .\n    # If we call `task.failure` instead, Camunda will publish the task again until\n    # some client finally completes it or the maximum amount of retries is reached.\n    return task.complete()\n\n\n# run the main task\nasyncio.run(main())\n```\n\nYou can run that example from the project folder with:\n\n```bash\npython examples/hello_world.py\n```\n\nYou should see something like this in your terminal:\n\n```\npython ./examples/hello_world.py\nwaiting for a task ...\nI got a task!\n```\n\nIf you don\'t see the second line you probably need to start the Camunda process (in the modeler).\n\n\n## Working with data\n\nThe example above is quite trivial since our client just returns a success event/result but nothing actually happens.\nIn the next example, we will let the python worker decide whether a passed number is odd or even.\nFirst, let\'s have a look at the BPMN [bpmn_process/odd_number.bpmn](bpmn_process/odd_number.bpmn):\n\n![](img/odd_number.png)\n\nWe have to external `ServiceTasks` labeled `Number is even!` and `Number is odd!`.\nBoth use the same topic `EchoTask` which we will use to print text to the terminal.\nHave a look in the property panel and click the `Input/Output` tab of one of the tasks.\nYou should see something like this:\n\n![](img/echo_params.png)\n\nThe task defines one input parameter named `text` of type `Text` with a value of `Number is even!` (or odd).\nNext, let\'s have a look at `Number Check`:\n\n```XML\n<bpmn:serviceTask id="Activity_NumberCheck" name="Number Check" camunda:type="external" camunda:topic="NumberCheckTask">\n    <bpmn:extensionElements>\n    <camunda:inputOutput>\n        <camunda:inputParameter name="number">42</camunda:inputParameter>\n        <camunda:outputParameter name="isOdd">${result}</camunda:outputParameter>\n    </camunda:inputOutput>\n    </bpmn:extensionElements>\n    <bpmn:incoming>Flow_Start</bpmn:incoming>\n    <bpmn:outgoing>Flow_OutCome</bpmn:outgoing>\n</bpmn:serviceTask>\n```\n\nNow you might say \'Wait! Where does this come from?\'.\nYou can use the Modeler GUI to edit values but you can also edit the BPMN-XML directly.\nIn the lower left you will see two tabs labelled `Diagram` and `XML`. This is where you change the view.\n\nWe see that the topic `camunda:topic` has been set to `NumberCheckTask` and that the service task contains two parameters in the `camunda:inputOutput` block.\nWe assign `42` to `number` and assign the received out parameter `result` to the environment variable `isOdd`.\nOf course, you can change the value of `number` as you like. \nYou also don\'t need to do it in the XML view.\nJust change back to `Diagram` and look for the `Input/Output` tab in the property panel of `Number Check`.\n\nLast, we have a look at the Gateway (the diamond-shaped box with the big X).\nThe gateway itself has just a label. \nThis does nothing but can be used to clarify what the gateway will evaluate.\nWe will evaluate our previously assigned `isOdd` variable.\nTo do so, we need to set conditions on the flows leaving the gateway.\nIf you click on the flow to the bottom (labeled with `"true"`), the property panel will look like this:\n\n![](img/flow_params.png)\n\nThe `name` parameter is again just something that makes your model easier to comprehend.\nThe `Condition Type` and `Expression` will determine if a Flow is considered valid.\nWe chose `Expression` since we want to check the value of `isOdd`.\nIn Camunda, this is done as seen in the picture above.\nWhen `isOdd == "true"` the service task `Number is odd!` will be executed and the process will end after this.\nThe flow above the gateway has a line crossing the flow which means that this is the default flow when no other condition is met.\nSo, what we do is we check whether `isOdd` is equal to `"true"`, execute `Number is odd!` if that is the case or `Number is even!` otherwise.\n\nNow let\'s see how [examples/odd_number.py](examples/odd_number.py) looks like:\n\n```python\nimport aiohttp\nimport asyncio\n\nfrom camunda.external_task.external_task import ExternalTask, Variables\nfrom camunda.external_task.external_task_worker import ExternalTaskWorker\nfrom camunda.external_task.external_task_result import ExternalTaskResult\n\n\nasync def main():\n    async with aiohttp.ClientSession() as session:\n        # a new parameter! We can pass \'asyncResponseTimeout\' to Camunda to configure how long a \'fetch and lock\' connection\n        # is kept alive. If this value is rather large, clients do not need to reconnect that often but it will take longer for a\n        # worker to properly shut down. The default value is 30 seconds (300000)\n        worker = ExternalTaskWorker(\n            worker_id=4,\n            base_url="http://localhost:8080/engine-rest",\n            session=session,\n            config={"asyncResponseTimeout": 5000},  # wait 5 seconds before timeout\n        )\n        # Our worker will now subscribe to two topics now\n        # We will create a new task with `asyncio.create_task` and await only the second subscribe\n        asyncio.create_task(worker.subscribe(topic_names="NumberCheckTask", action=number_check))\n        await worker.subscribe(topic_names="EchoTask", action=echo)\n\n\nasync def number_check(task: ExternalTask) -> ExternalTaskResult:\n    try:\n        number = task.context_variables["number"]\n        print(f"We received {number} for checking...")\n        # We set a locally scoped variable \'result\' to \'true\' or \'false\'\n        task.local_variables.set_variable(\n            "result", "true" if int(number) % 2 != 0 else "false", Variables.ValueType.STRING\n        )\n        # We pass the variables object as LOCAL variables which will only be available in the context of the task\n        # that called the external task worker. The result must be assigned in case it should be used somewhere else.\n        # Just have a look at the odd_number.bpmn to see how.\n        return task.complete()\n    # If your input could not be parsed with `int()` the task will fail\n    # and another external service could try to do better.\n    except Exception as err:\n        print(f"Oh no! Something went wrong: {err}")\n        return task.failure(\n            error_message=err.__class__.__name__,\n            error_details=str(err),\n            max_retries=3,\n            retry_timeout=5000,\n        )\n\n\nasync def echo(task: ExternalTask) -> ExternalTaskResult:\n    print(f"Camunda wants to say: {task.context_variables[\'text\']}")\n    return task.complete()\n\n\n# run the main task\nasyncio.run(main())\n```\n\nWe now have two callbacks and our worker will subscribe them the appropriate topic as seen in the BPMN model.\nWe also reduce the `asyncResponseTimeout` of the worker to prevent already closed python clients to accidently fetch tasks.\nThis is just a workaround for the sake of simplicity.\nIt is recommended to actually wait for worker subscriptions to return (by awaiting `worker.cancel()`) before shutting down the whole process.\n\nIn `number_check` and in `echo`, we see how to retrieve variables from the `ExternalTask` object `task`.\nNote that retrieving variables with `task.context_variables["key"]` will raise a `KeyError` if `key` does not exists.\nTo deal with optional variables you can use `task.context_variables.get_variable("key")` which will return `None` if `key` cannot be found.\nIn `number_check`, there is also shown how a `Variables` object is created, a value is assigned and how this object is passed as a **local** variables object.\nLocal variables can only be used in the scope of the service task.\nThis is why we have to assign `result` to an output parameter in Camunda.\nWe could also pass `variables` to `global_variables` instead.\nThis way, `result` could be used in the whole process.\nHowever, in more complex scenarios this might clutter the environment and may even lead to colliding variable definitions.\nKeeping things local increases the control by keeping variable scopes narrow and also *visible* in the BPMN model.\n\nWhen you run the example, you should see an output like this after you have started the Camunda process:\n\n```\npython ./examples/odd_number.py\nWe received 42 for checking...\nCamunda wants to say: Number is even!\n```\n\n## Managing multiple workers/subscriptions\n\nThe above mentions some issues when clients are stopped and restarted.\nCamunda connections might not have been properly released the next time a task is scheduled by Camunda.\nThis may cause the already stopped/inactive instance to lock the task.\nOne way to deal with this is to catch a KeyboardInterrupt or any other kind of event that signalizes a shutdown and await `worker.cancel()` as shown in [examples/manage_tasks.py](examples/manage_tasks.py):\n\n```python\nimport aiohttp\nimport asyncio\n\nfrom camunda.external_task.external_task import ExternalTask, Variables\nfrom camunda.external_task.external_task_worker import ExternalTaskWorker\nfrom camunda.external_task.external_task_result import ExternalTaskResult\n\n\nclass Worker:\n    def __init__(self):\n        self.worker = None\n        self.loop = None\n\n    def start(self):\n        """Run the worker and block forever"""\n        self.loop = asyncio.get_event_loop()\n        self.loop.run_until_complete(self._run())\n\n    async def _run(self):\n        async with aiohttp.ClientSession() as session:\n            self.worker = ExternalTaskWorker(\n                worker_id=4, base_url="http://localhost:8080/engine-rest", session=session\n            )\n            # dispatch the first subscription\n            self.loop.create_task(\n                self.worker.subscribe(topic_names="NumberCheckTask", action=number_check)\n            )\n            # and block the current task with the second subscription again\n            await self.worker.subscribe(topic_names="EchoTask", action=echo)\n\n    def stop(self):\n        self.loop.run_until_complete(self.worker.cancel())\n\n\nasync def number_check(task: ExternalTask) -> ExternalTaskResult:\n    try:\n        number = task.context_variables["number"]\n        print(f"We received {number} for checking...")\n        task.local_variables.set_variable(\n            "result", "true" if int(number) % 2 != 0 else "false", Variables.ValueType.STRING\n        )\n        return task.complete()\n    except Exception as err:\n        print(f"Oh no! Something went wrong: {err}")\n        return task.failure()\n\n\nasync def echo(task: ExternalTask) -> ExternalTaskResult:\n    print(f"Camunda wants to say: {task.context_variables[\'text\']}")\n    return task.complete()\n\n\n# run the main task\ntry:\n    worker = Worker()\n    worker.start()\nexcept KeyboardInterrupt:\n    # Stopping workers might take a while.\n    # How long it will take depends on the chosen asyncResponseTimeout (default is 30000 ms)\n    print(f"Stopping workers...")\n    worker.stop()\nprint(f"All done!")\n```\n\nThe code above basically does the same as the `odd_number` example before but we wrapped the asynchronous bits into a `Worker` class and added methods to start and stop workers and their subscriptions. Depending on how long you are willing to wait for a shutdown you might want to adjust `asyncResponseTimeout`.\n',
    'author': 'Alexander Neumann',
    'author_email': 'aleneum@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://www.avikom.app',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<3.10',
}


setup(**setup_kwargs)
