"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomResourceHandler = void 0;
/* eslint-disable no-console */
const https = require("https");
const url = require("url");
// eslint-disable-next-line import/no-extraneous-dependencies
const AWS = require("aws-sdk");
// eslint-disable-next-line @typescript-eslint/ban-types
class CustomResourceHandler {
    constructor(event, context) {
        this.event = event;
        this.context = context;
        this.timedOut = false;
        this.timeout = setTimeout(async () => {
            await this.respond({
                status: 'FAILED',
                reason: 'Lambda Function Timeout',
                data: this.context.logStreamName,
            });
            this.timedOut = true;
        }, context.getRemainingTimeInMillis() - 1200);
        this.event = event;
        this.physicalResourceId = extractPhysicalResourceId(event);
    }
    /**
     * Handles executing the custom resource event. If `stateMachineArn` is present
     * in the props then trigger the waiter statemachine
     */
    async handle() {
        try {
            if ('stateMachineArn' in this.event.ResourceProperties) {
                const req = {
                    stateMachineArn: this.event.ResourceProperties.stateMachineArn,
                    name: this.event.RequestId,
                    input: JSON.stringify(this.event),
                };
                await this.startExecution(req);
                return;
            }
            else {
                const response = await this.processEvent(this.event.ResourceProperties);
                return response;
            }
        }
        catch (e) {
            console.log(e);
            throw e;
        }
        finally {
            clearTimeout(this.timeout);
        }
    }
    /**
     * Handle async requests from the waiter state machine
     */
    async handleIsComplete() {
        try {
            const result = await this.processEvent(this.event.ResourceProperties);
            return result;
        }
        catch (e) {
            console.log(e);
            return;
        }
        finally {
            clearTimeout(this.timeout);
        }
    }
    /**
     * Start a step function state machine which will wait for the request
     * to be successful.
     */
    async startExecution(req) {
        try {
            const sfn = new AWS.StepFunctions();
            await sfn.startExecution(req).promise();
        }
        finally {
            clearTimeout(this.timeout);
        }
    }
    respond(response) {
        if (this.timedOut) {
            return;
        }
        const cfResponse = {
            Status: response.status,
            Reason: response.reason,
            PhysicalResourceId: this.physicalResourceId,
            StackId: this.event.StackId,
            RequestId: this.event.RequestId,
            LogicalResourceId: this.event.LogicalResourceId,
            NoEcho: false,
            Data: response.data,
        };
        const responseBody = JSON.stringify(cfResponse);
        console.log('Responding to CloudFormation', responseBody);
        const parsedUrl = url.parse(this.event.ResponseURL);
        const requestOptions = {
            hostname: parsedUrl.hostname,
            path: parsedUrl.path,
            method: 'PUT',
            headers: {
                'content-type': '',
                'content-length': Buffer.byteLength(responseBody, 'utf8'),
            },
        };
        return new Promise((resolve, reject) => {
            try {
                const request = https.request(requestOptions, resolve);
                request.on('error', reject);
                request.write(responseBody);
                request.end();
            }
            catch (e) {
                reject(e);
            }
            finally {
                clearTimeout(this.timeout);
            }
        });
    }
}
exports.CustomResourceHandler = CustomResourceHandler;
function extractPhysicalResourceId(event) {
    switch (event.RequestType) {
        case 'Create':
            return event.LogicalResourceId;
        case 'Update':
        case 'Delete':
            return event.PhysicalResourceId;
    }
}
//# sourceMappingURL=data:application/json;base64,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