import { IVpc } from '@aws-cdk/aws-ec2';
import { AwsLogDriver, BaseService, CloudMapOptions, Cluster, ContainerDefinition, ContainerImage, ICluster, LogDriver, PropagatedTagSource, Secret } from '@aws-cdk/aws-ecs';
import { NetworkListener, NetworkLoadBalancer, NetworkTargetGroup } from '@aws-cdk/aws-elasticloadbalancingv2';
import { IRole } from '@aws-cdk/aws-iam';
import { IHostedZone } from '@aws-cdk/aws-route53';
import { Duration } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { Construct as CoreConstruct } from '@aws-cdk/core';
/**
 * The properties for the base NetworkMultipleTargetGroupsEc2Service or NetworkMultipleTargetGroupsFargateService service.
 *
 * @stability stable
 */
export interface NetworkMultipleTargetGroupsServiceBaseProps {
    /**
     * The name of the cluster that hosts the service.
     *
     * If a cluster is specified, the vpc construct should be omitted. Alternatively, you can omit both cluster and vpc.
     *
     * @default - create a new cluster; if both cluster and vpc are omitted, a new VPC will be created for you.
     * @stability stable
     */
    readonly cluster?: ICluster;
    /**
     * The VPC where the container instances will be launched or the elastic network interfaces (ENIs) will be deployed.
     *
     * If a vpc is specified, the cluster construct should be omitted. Alternatively, you can omit both vpc and cluster.
     *
     * @default - uses the VPC defined in the cluster or creates a new VPC.
     * @stability stable
     */
    readonly vpc?: IVpc;
    /**
     * The properties required to create a new task definition.
     *
     * Only one of TaskDefinition or TaskImageOptions must be specified.
     *
     * @default - none
     * @stability stable
     */
    readonly taskImageOptions?: NetworkLoadBalancedTaskImageProps;
    /**
     * The desired number of instantiations of the task definition to keep running on the service.
     *
     * The minimum value is 1
     *
     * @default - If the feature flag, ECS_REMOVE_DEFAULT_DESIRED_COUNT is false, the default is 1;
     * if true, the default is 1 for all new services and uses the existing services desired count
     * when updating an existing service.
     * @stability stable
     */
    readonly desiredCount?: number;
    /**
     * Name of the service.
     *
     * @default - CloudFormation-generated name.
     * @stability stable
     */
    readonly serviceName?: string;
    /**
     * The period of time, in seconds, that the Amazon ECS service scheduler ignores unhealthy Elastic Load Balancing target health checks after a task has first started.
     *
     * @default - defaults to 60 seconds if at least one load balancer is in-use and it is not already set
     * @stability stable
     */
    readonly healthCheckGracePeriod?: Duration;
    /**
     * The network load balancer that will serve traffic to the service.
     *
     * @default - a new load balancer with a listener will be created.
     * @stability stable
     */
    readonly loadBalancers?: NetworkLoadBalancerProps[];
    /**
     * Specifies whether to propagate the tags from the task definition or the service to the tasks in the service.
     *
     * Tags can only be propagated to the tasks within the service during service creation.
     *
     * @default - none
     * @stability stable
     */
    readonly propagateTags?: PropagatedTagSource;
    /**
     * Specifies whether to enable Amazon ECS managed tags for the tasks within the service.
     *
     * For more information, see
     * [Tagging Your Amazon ECS Resources](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html)
     *
     * @default false
     * @stability stable
     */
    readonly enableECSManagedTags?: boolean;
    /**
     * The options for configuring an Amazon ECS service to use service discovery.
     *
     * @default - AWS Cloud Map service discovery is not enabled.
     * @stability stable
     */
    readonly cloudMapOptions?: CloudMapOptions;
    /**
     * Properties to specify NLB target groups.
     *
     * @default - default portMapping registered as target group and attached to the first defined listener
     * @stability stable
     */
    readonly targetGroups?: NetworkTargetProps[];
}
/**
 * Options for configuring a new container.
 *
 * @stability stable
 */
export interface NetworkLoadBalancedTaskImageProps {
    /**
     * The image used to start a container.
     *
     * Image or taskDefinition must be specified, but not both.
     *
     * @default - none
     * @stability stable
     */
    readonly image: ContainerImage;
    /**
     * The environment variables to pass to the container.
     *
     * @default - No environment variables.
     * @stability stable
     */
    readonly environment?: {
        [key: string]: string;
    };
    /**
     * The secrets to expose to the container as an environment variable.
     *
     * @default - No secret environment variables.
     * @stability stable
     */
    readonly secrets?: {
        [key: string]: Secret;
    };
    /**
     * Flag to indicate whether to enable logging.
     *
     * @default true
     * @stability stable
     */
    readonly enableLogging?: boolean;
    /**
     * The log driver to use.
     *
     * @default - AwsLogDriver if enableLogging is true
     * @stability stable
     */
    readonly logDriver?: LogDriver;
    /**
     * The name of the task execution IAM role that grants the Amazon ECS container agent permission to call AWS APIs on your behalf.
     *
     * @default - No value
     * @stability stable
     */
    readonly executionRole?: IRole;
    /**
     * The name of the task IAM role that grants containers in the task permission to call AWS APIs on your behalf.
     *
     * @default - A task role is automatically created for you.
     * @stability stable
     */
    readonly taskRole?: IRole;
    /**
     * The container name value to be specified in the task definition.
     *
     * @default - none
     * @stability stable
     */
    readonly containerName?: string;
    /**
     * A list of port numbers on the container that is bound to the user-specified or automatically assigned host port.
     *
     * If you are using containers in a task with the awsvpc or host network mode, exposed ports should be specified using containerPort.
     * If you are using containers in a task with the bridge network mode and you specify a container port and not a host port,
     * your container automatically receives a host port in the ephemeral port range.
     *
     * Port mappings that are automatically assigned in this way do not count toward the 100 reserved ports limit of a container instance.
     *
     * For more information, see
     * [hostPort](https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PortMapping.html#ECS-Type-PortMapping-hostPort).
     *
     * @default - [80]
     * @stability stable
     */
    readonly containerPorts?: number[];
    /**
     * The name of a family that this task definition is registered to.
     *
     * A family groups multiple versions of a task definition.
     *
     * @default - Automatically generated name.
     * @stability stable
     */
    readonly family?: string;
}
/**
 * Properties to define an network load balancer.
 *
 * @stability stable
 */
export interface NetworkLoadBalancerProps {
    /**
     * Name of the load balancer.
     *
     * @stability stable
     */
    readonly name: string;
    /**
     * Listeners (at least one listener) attached to this load balancer.
     *
     * @default - none
     * @stability stable
     */
    readonly listeners: NetworkListenerProps[];
    /**
     * Determines whether the Load Balancer will be internet-facing.
     *
     * @default true
     * @stability stable
     */
    readonly publicLoadBalancer?: boolean;
    /**
     * The domain name for the service, e.g. "api.example.com.".
     *
     * @default - No domain name.
     * @stability stable
     */
    readonly domainName?: string;
    /**
     * The Route53 hosted zone for the domain, e.g. "example.com.".
     *
     * @default - No Route53 hosted domain zone.
     * @stability stable
     */
    readonly domainZone?: IHostedZone;
}
/**
 * Properties to define an network listener.
 *
 * @stability stable
 */
export interface NetworkListenerProps {
    /**
     * Name of the listener.
     *
     * @stability stable
     */
    readonly name: string;
    /**
     * The port on which the listener listens for requests.
     *
     * @default 80
     * @stability stable
     */
    readonly port?: number;
}
/**
 * Properties to define a network load balancer target group.
 *
 * @stability stable
 */
export interface NetworkTargetProps {
    /**
     * The port number of the container.
     *
     * Only applicable when using application/network load balancers.
     *
     * @stability stable
     */
    readonly containerPort: number;
    /**
     * Name of the listener the target group attached to.
     *
     * @default - default listener (first added listener)
     * @stability stable
     */
    readonly listener?: string;
}
/**
 * The base class for NetworkMultipleTargetGroupsEc2Service and NetworkMultipleTargetGroupsFargateService classes.
 *
 * @stability stable
 */
export declare abstract class NetworkMultipleTargetGroupsServiceBase extends CoreConstruct {
    /**
     * (deprecated) The desired number of instantiations of the task definition to keep running on the service.
     *
     * @deprecated - Use `internalDesiredCount` instead.
     */
    readonly desiredCount: number;
    /**
     * The desired number of instantiations of the task definition to keep running on the service.
     *
     * The default is 1 for all new services and uses the existing services desired count
     * when updating an existing service, if one is not provided.
     *
     * @stability stable
     */
    readonly internalDesiredCount?: number;
    /**
     * The Network Load Balancer for the service.
     *
     * @stability stable
     */
    readonly loadBalancer: NetworkLoadBalancer;
    /**
     * The listener for the service.
     *
     * @stability stable
     */
    readonly listener: NetworkListener;
    /**
     * The cluster that hosts the service.
     *
     * @stability stable
     */
    readonly cluster: ICluster;
    /**
     * @stability stable
     */
    protected logDriver?: LogDriver;
    /**
     * @stability stable
     */
    protected listeners: NetworkListener[];
    /**
     * @stability stable
     */
    protected targetGroups: NetworkTargetGroup[];
    private loadBalancers;
    /**
     * Constructs a new instance of the NetworkMultipleTargetGroupsServiceBase class.
     *
     * @stability stable
     */
    constructor(scope: Construct, id: string, props?: NetworkMultipleTargetGroupsServiceBaseProps);
    /**
     * Returns the default cluster.
     *
     * @stability stable
     */
    protected getDefaultCluster(scope: Construct, vpc?: IVpc): Cluster;
    /**
     * @stability stable
     */
    protected createAWSLogDriver(prefix: string): AwsLogDriver;
    /**
     * @stability stable
     */
    protected findListener(name?: string): NetworkListener;
    /**
     * @stability stable
     */
    protected registerECSTargets(service: BaseService, container: ContainerDefinition, targets: NetworkTargetProps[]): NetworkTargetGroup;
    /**
     * @stability stable
     */
    protected addPortMappingForTargets(container: ContainerDefinition, targets: NetworkTargetProps[]): void;
    /**
     * Create log driver if logging is enabled.
     */
    private createLogDriver;
    private validateInput;
    private createLoadBalancer;
    private createListener;
    private createDomainName;
}
