import tarfile
import re
from urllib.request import urlretrieve

import numpy as np
import pandas as pd
import rocks

from classy import config
from classy import core
from classy.log import logger

# PREPROCESSING = {
#     "tholen": {},
#     "demeo": {},
#     "mahlke": {
#         "resample": {
#             "grid": taxonomies.mahlke.WAVE,
#             "bounds_error": False,
#             "fill_value": (np.nan, np.nan),
#         },
#     },
# }


def load_index():
    """Load the MITHNEOS reflectance spectra index."""

    PATH_INDEX = config.PATH_CACHE / "mithneos/index.csv"

    if not PATH_INDEX.is_file():
        retrieve_spectra()

    return pd.read_csv(PATH_INDEX, dtype={"number": "Int64"})


def load_spectrum(spec):
    """Load a cached MITHNEOS spectrum."""
    PATH_SPEC = config.PATH_CACHE / f"mithneos/{spec.filename}"

    data = pd.read_csv(
        PATH_SPEC, names=["wave", "refl", "err", "flag"], delimiter="\s+"
    )
    # 2 - reject. This is flag 0 in MITHNEOS
    flags = [0 if f != 0 else 2 for f in data["flag"].values]

    spec = core.Spectrum(
        wave=data["wave"],
        refl=data["refl"],
        refl_err=data["err"],
        flag=flags,
        source="MITHNEOS",
        run=spec.run,
        name=spec["name"],
        number=spec.number,
        bibcode=spec.bibcode,
        shortbib=spec.shortbib,
    )
    spec._source = "MITHNEOS"
    return spec


def retrieve_spectra():
    """Retrieve all MITHNEOS spectra to mithneos/ the cache directory."""

    URL = "http://smass.mit.edu/data/spex"

    # Create directory structure and check if the spectrum is already cached
    PATH_MITHNEOS = config.PATH_CACHE / "mithneos/"
    PATH_MITHNEOS.mkdir(parents=True, exist_ok=True)

    logger.info("Retrieving all MITHNEOS reflectance spectra [34MB] to cache...")

    # Get data from smass.mit.edu
    for run, spectra in RUNS.items():
        PATH_OUT = PATH_MITHNEOS / run
        PATH_OUT.mkdir(parents=True, exist_ok=True)

        for spec in spectra:
            url_archive = f"{URL}/{run}/{spec}"
            urlretrieve(url_archive, PATH_OUT / spec)

    # Create index
    index = pd.DataFrame()

    for run in RUNS:
        PATH_DIR = PATH_MITHNEOS / run

        for file_ in PATH_DIR.glob("a*txt"):
            # ------
            # Extract target from filename
            id_mithneos = file_.name.split(".")[0]

            # Asteroid Unnumbered: extract designation
            if id_mithneos.startswith("au"):
                id_mithneos = id_mithneos.lstrip("au")
                designation = re.match(
                    r"([1A][8-9][0-9]{2}[ _]?[A-Za-z]{2}[0-9]{0,3})|"
                    r"(20[0-9]{2}[_ ]?[A-Za-z]{2}[0-9]{0,3})",
                    id_mithneos,
                )
                match = designation.group(0)

            # Asteroid: extract number
            elif id_mithneos.startswith("a"):
                id_mithneos = id_mithneos.lstrip("a")
                number = re.match(r"(\d\d\d\d\d\d)", id_mithneos)
                try:
                    match = number.group(0)
                except AttributeError:
                    continue

            name, number = rocks.id(match)

            # ------
            # Append to index
            entry = pd.DataFrame(
                {
                    "name": name,
                    "number": number,
                    "filename": f"{run}/{file_.name}",
                    "run": run,
                    "shortbib": "Binzel+ 2019",
                    "bibcode": "2019Icar..324...41B",
                },
                index=[0],
            )
            index = pd.concat([index, entry], ignore_index=True)

    index["number"] = index["number"].astype("Int64")
    index.to_csv(PATH_MITHNEOS / "index.csv", index=False)


RUNS = {
    "sp01": [
        "au2000QW7.sp01.txt",
        "au1993QP.sp01.txt",
        "au2000PG3.sp01.txt",
    ],
    "sp02": [
        "a018106.sp02.txt",
        "a018172.sp02.txt",
        "au2000GK137.sp02.txt",
        "au2000SA10.sp02.txt",
        "a004486.sp02.txt",
        "a003102.sp02.txt",
        "au2000SV20.sp02.txt",
        "au2000SB8.sp02.txt",
        "au2000SP43.sp02.txt",
        "au2000OG8.sp02.txt",
        "au2000DP107.sp02.txt",
        "au1998UP1.sp02.txt",
        "a018916.sp02.txt",
        "a017274.sp02.txt",
    ],
    "sp03": [
        "a020790.sp03.txt",
        "au2000CK33.sp03.txt",
        "au2001BE10.sp03.txt",
        "a018736.sp03.txt",
        "a019356.sp03.txt",
        "au2000BD19.sp03.txt",
        "au2000SS164.sp03.txt",
        "au2000XL44.sp03.txt",
        "a001620.sp03.txt",
        "au1950DA.sp03.txt",
        "a020429.sp03.txt",
        "a004179.sp03.txt",
        "au1992BL2.sp03.txt",
        "a004688.sp03.txt",
        "a003199.sp03.txt",
    ],
    "sp04": [
        "au2001CP36.sp04.txt",
    ],
    "sp05": [
        "a001036.sp05.txt",
        "au2001FA1.sp05.txt",
        "au2000EE104.sp05.txt",
        "a001864.sp05.txt",
        "au2001EB.sp05.txt",
        "au2001FO32.sp05.txt",
        "a005587.sp05.txt",
        "a005604.sp05.txt",
        "a023187.sp05.txt",
        "a024475.sp05.txt",
    ],
    "sp06": [
        "au2001ME1.sp06.txt",
        "au2001KA67.sp06.txt",
        "au2001KP41.sp06.txt",
        "a003103.sp06.txt",
        "au1999KW4.sp06.txt",
    ],
    "sp07": [
        "a001916.sp07.txt",
        "au2001MQ3.sp07.txt",
    ],
    "sp08": [
        "a005786.sp08.txt",
        "a000719.sp08.txt",
        "au1996PC1.sp08.txt",
    ],
    "sp09": [
        "au2002AV.sp09.txt",
        "a0011054.sp09.txt",
        "a001627.sp09.txt",
        "au2002AA.sp09.txt",
        "a001943.sp09.txt",
    ],
    "sp11": [
        "a007341.sp11.txt",
        "a006455.sp11.txt",
        "au2001VS78.sp11.txt",
        "au2000GD2.sp11.txt",
        "a036284.sp11.txt",
    ],
    "sp12": [
        "a005645.sp12.txt",
        "au2001XN254.sp12.txt",
        "au2002GO5.sp12.txt",
        "au2000SL.sp12.txt",
        "a031669.sp12.txt",
        "a038071.sp12.txt",
        "au2001VS78.sp12.txt",
    ],
    "sp13": [
        "a001685.sp13.txt",
        "au1999HF1.sp13.txt",
    ],
    "sp14": [
        "a001580.sp14.txt",
    ],
    "sp15": [
        "a000433.sp15.txt",
    ],
    "sp16": [
        "au2002NY40.sp16.txt",
        "a000433.sp16.txt",
    ],
    "sp17": [
        "a003753.sp17.txt",
        "au2002HK12.sp17.txt",
        "a001685.sp17.txt",
        "a000433.sp17.txt",
        "au2002QH10.sp17.txt",
        "au2002QE15.sp17.txt",
        "au1999HF1.sp17.txt",
        "au1998RO1.sp17.txt",
        "au2000ED104.sp17.txt",
    ],
    "sp19": [
        "au1998UO1.sp19.txt",
        "a006455.sp19.txt",
        "au2002NX18.sp19.txt",
        "au2002RS28.sp19.txt",
        "a025330.sp19.txt",
        "au1998WM.sp19.txt",
        "au1989VA.sp19.txt",
        "au2002SY50.sp19.txt",
        "a003200.sp19.txt",
        "a022753.sp19.txt",
    ],
    "sp20": [
        "au1992SY.sp20.txt",
        "a035107.sp20.txt",
        "au1993OM7.sp20.txt",
        "au1999OP3.sp20.txt",
        "au2002XO14.sp20.txt",
    ],
    "sp21": [
        "au2003AJ73.sp21.txt",
        "a054789.sp21.txt",
        "au2003ED50.sp21.txt",
        "au1978CA.sp21.txt",
    ],
    "sp23": [
        "au2003FT3.sp23.txt",
        "au2003LC5.sp23.txt",
        "a065996.sp23.txt",
        "au2001LF.sp23.txt",
    ],
    "sp25": [
        "a063164.sp25.txt",
        "a000105.sp25.txt",
        "a008444.sp25.txt",
        "a020786.sp25.txt",
        "a002074.sp25.txt",
        "a000014.sp25.txt",
        "a001468.sp25.txt",
        "a019127.sp25.txt",
        "a000181.sp25.txt",
        "au2000GF2.sp25.txt",
        "a004197.sp25.txt",
        "a003402.sp25.txt",
        "a024445.sp25.txt",
        "a005641.sp25.txt",
        "a066146.sp25.txt",
        "a000063.sp25.txt",
        "a000065.sp25.txt",
    ],
    "sp26": [
        "au1998FG2.sp26.txt",
        "a069230.sp26.txt",
        "a006063.sp26.txt",
        "a068278.sp26.txt",
        "a000073.sp26.txt",
        "au1997AE12.sp26.txt",
        "au2002GJ103.sp26.txt",
        "au2003RL10.sp26.txt",
        "a022771.sp26.txt",
        "a000250.sp26.txt",
    ],
    "sp27": [
        "au2002AL14.sp27.txt",
        "au2003YK118.sp27.txt",
        "a006239.sp27.txt",
        "a003122.sp27.txt",
        "a001036.sp27.txt",
        "au2000XH44.sp27.txt",
    ],
    "sp28": [
        "a000201.sp28.txt",
        "a000015.sp28.txt",
        "a001374.sp28.txt",
        "au2000XH44.sp28.txt",
        "a006411.sp28.txt",
        "a000010.sp28.txt",
        "a000034.sp28.txt",
        "a003858.sp28.txt",
        "a000344.sp28.txt",
        "a001131.sp28.txt",
        "a000005.sp28.txt",
        "a000570.sp28.txt",
        "a001508.sp28.txt",
        "a000007.sp28.txt",
    ],
    "sp29": [
        "a001139.sp29.txt",
        "au2000JS66.sp29.txt",
        "a000051.sp29.txt",
        "a004558.sp29.txt",
        "a000067.sp29.txt",
        "a007092.sp29.txt",
    ],
    "sp30": [
        "a004179.sp30.txt",
        "a003908.sp30.txt",
        "au2000SY2.sp30.txt",
        "a007350.sp30.txt",
        "a052750.sp30.txt",
        "a005642.sp30.txt",
    ],
    "sp31": [
        "a000027.sp31.txt",
        "a000021.sp31.txt",
        "au1998UO1.sp31.txt",
        "au2004FH11.sp31.txt",
        "a000056.sp31.txt",
        "a000944.sp31.txt",
        "au2004EW9.sp31.txt",
        "a000111.sp31.txt",
        "au2004RN9.sp31.txt",
        "a005817.sp31.txt",
        "a000773.sp31.txt",
    ],
    "sp32": [
        "a000040.sp32.txt",
        "au1998UO1.sp32.txt",
        "a000512.sp32.txt",
        "au2000RL77.sp32.txt",
        "au1999LF6.sp32.txt",
        "a003873.sp32.txt",
    ],
    "sp33": [
        "au2004JA27.sp33.txt",
        "au2001SE170.sp33.txt",
        "au2004TP1.sp33.txt",
        "au2001CC21.sp33.txt",
    ],
    "sp34": [
        "a003200.sp34.txt",
        "au2004TU12.sp34.txt",
    ],
    "sp35": [
        "a000279.sp35.txt",
        "a000699.sp35.txt",
        "a005653.sp35.txt",
        "a099942subm.sp35.txt",
        "au2004MN4.sp35.txt",
        "au2004OB.sp35.txt",
        "a002195.sp35.txt",
    ],
    "sp36": [
        "a022753.sp36.txt",
        "a006386.sp36.txt",
        "a030825.sp36.txt",
        "a003199.sp36.txt",
        "au2004QD3.sp36.txt",
        "a001565.sp36.txt",
        "au2004EW.sp36.txt",
    ],
    "sp37": [
        "a001204.sp37.txt",
        "a001660.sp37.txt",
        "a001065.sp37.txt",
        "au2005CK38.sp37.txt",
        "a001036.sp37.txt",
        "a001685.sp37.txt",
        "a001640.sp37.txt",
        "a000246.sp37.txt",
    ],
    "sp38": [
        "a000191.sp38.txt",
        "a005641.sp38.txt",
        "au2005GO22.sp38.txt",
        "au1999HF1.sp38.txt",
        "au2005AB.sp38.txt",
        "a000776.sp38.txt",
        "a004055.sp38.txt",
        "au2005GR33.sp38.txt",
        "au2002CD.sp38.txt",
    ],
    "sp39": [
        "a000308.sp39.txt",
        "a002246.sp39.txt",
        "a001542.sp39.txt",
        "au2005GU.sp39.txt",
        "a007350.sp39.txt",
        "a003248.sp39.txt",
        "au2004QT24.sp39.txt",
    ],
    "sp40": [
        "a006611.sp40.txt",
        "au2005AT42.sp40.txt",
        "a000617.sp40.txt",
        "a002378.sp40.txt",
        "au1998VF31.sp40.txt",
        "a000069.sp40.txt",
        "au2005JB.sp40.txt",
        "a000153.sp40.txt",
        "a000269.sp40.txt",
        "au2005ED318.sp40.txt",
        "au2005JR5.sp40.txt",
        "a001583.sp40.txt",
        "a002311.sp40.txt",
    ],
    "sp41": [
        "a000264.sp41.txt",
        "a005261.sp41.txt",
        "a002063.sp41.txt",
        "a000001.sp41.txt",
        "a000013.sp41.txt",
        "a000785.sp41.txt",
    ],
    "sp42": [
        "a001332.sp42.txt",
        "a001566.sp42.txt",
        "au2004YZ23.sp42.txt",
        "a001143.sp42.txt",
        "a005641.sp42.txt",
        "a006178.sp42.txt",
    ],
    "sp43": [
        "au2005ND7.sp43.txt",
        "au2002EX12.sp43.txt",
        "a006456.sp43.txt",
        "a005660.sp43.txt",
    ],
    "sp44": [
        "a000076.sp44.txt",
        "a013553.sp44.txt",
        "au2005QP87.sp44.txt",
        "a000090.sp44.txt",
        "a001167.sp44.txt",
        "a000132.sp44.txt",
        "a000266.sp44.txt",
        "a066251.sp44.txt",
        "au1977VA.sp44.txt",
        "au1999RQ36.sp44.txt",
        "a008567.sp44.txt",
        "a047581.sp44.txt",
        "a000087.sp44.txt",
    ],
    "sp45": [
        "a000024.sp45.txt",
        "a002099.sp45.txt",
        "au1998ST49.sp45.txt",
        "a006047.sp45.txt",
        "au1983TA.sp45.txt",
        "au2004FN18.sp45.txt",
        "a000048.sp45.txt",
        "a066251.sp45.txt",
        "a002335.sp45.txt",
        "a000097.sp45.txt",
        "a003753.sp45.txt",
        "a003361.sp45.txt",
        "a065784.sp45.txt",
        "a000345.sp45.txt",
    ],
    "sp46": [
        "au2004VG64.sp46.txt",
        "a066251.sp46.txt",
        "a000066.sp46.txt",
        "a001180.sp46.txt",
        "a000078.sp46.txt",
        "au2002RX211.sp46.txt",
        "a002335.sp46.txt",
    ],
    "sp47": [
        "au2005JE46n2.sp47.txt",
        "au2005UL5.sp47.txt",
        "a000011.sp47.txt",
        "a001862n1.sp47.txt",
        "au2005JE46n1.sp47.txt",
        "a000151.sp47.txt",
        "a000099.sp47.txt",
        "a000210.sp47.txt",
        "au2005QE166.sp47.txt",
        "a003635.sp47.txt",
        "a005332.sp47.txt",
        "a000175.sp47.txt",
        "a001862n2.sp47.txt",
    ],
    "sp48": [
        "a000706.sp48.txt",
        "a000066.sp48.txt",
        "a002850.sp48.txt",
        "a000019.sp48.txt",
        "a008567.sp48.txt",
        "au2005JE46.sp48.txt",
        "a001862.sp48.txt",
        "au2005WX.sp48.txt",
        "a000055.sp48.txt",
        "a000043.sp48.txt",
        "a002823.sp48.txt",
    ],
    "sp49": [
        "a002107.sp49.txt",
        "au2006BN55.sp49.txt",
        "a023183.sp49.txt",
        "a000706.sp49.txt",
        "a000019.sp49.txt",
        "a000096.sp49.txt",
        "a000055.sp49.txt",
    ],
    "sp50": [
        "a003317.sp50.txt",
        "a085709.sp50.txt",
        "a003628.sp50.txt",
        "a000192.sp50.txt",
        "a001310.sp50.txt",
        "au2006HY51.sp50.txt",
        "a000236.sp50.txt",
        "a009992.sp50.txt",
    ],
    "sp51": [
        "a003703.sp51.txt",
        "au2003YQ117.sp51.txt",
        "au2006HQ30.sp51.txt",
        "a005379.sp51.txt",
        "a001036.sp51.txt",
    ],
    "sp52": [
        "a002064.sp52.txt",
        "a000114.sp52.txt",
        "a000052.sp52.txt",
        "a068950.sp52.txt",
        "au2006HR30.sp52.txt",
    ],
    "sp53": [
        "a021238.sp53.txt",
        "a000131.sp53.txt",
        "a000908.sp53.txt",
        "au2006NM.sp53.txt",
    ],
    "sp54": [
        "au2006RZ.sp54.txt",
    ],
    "sp55": [
        "a001300.sp55.txt",
        "a001980.sp55.txt",
        "a001494.sp55.txt",
        "a086212.sp55.txt",
        "a053435.sp55.txt",
        "au2006RG2.sp55.txt",
        "a010115.sp55.txt",
        "a006585.sp55.txt",
        "a025916.sp55.txt",
        "a000077.sp55.txt",
        "a005143.sp55.txt",
        "a000022.sp55.txt",
        "a005292.sp55.txt",
        "au2006UC185.sp55.txt",
        "a085713.sp55.txt",
    ],
    "sp56": [
        "au2006UM.sp56.txt",
        "a005011.sp56.txt",
        "a010195.sp56.txt",
        "au2000SB1.sp56.txt",
        "au2006VQ13.sp56.txt",
        "a001866.sp56.txt",
        "a001300.sp56.txt",
        "a002919.sp56.txt",
        "au2006UL217.sp56.txt",
    ],
    "sp57": [
        "a000234.sp57.txt",
        "a002867.sp57.txt",
        "au2003SD220.sp57.txt",
        "au2006WS3.sp57.txt",
        "a000101.sp57.txt",
        "au2006UN216.sp57.txt",
        "au2004XL14.sp57.txt",
        "au2001XP1.sp57.txt",
        "a002212.sp57.txt",
    ],
    "sp58": [
        "a002867.sp58.txt",
        "a139622.sp58.txt",
        "a053233.sp58.txt",
        "a005693.sp58.txt",
        "au2000YM29.sp58.txt",
        "a013791.sp58.txt",
        "a004544.sp58.txt",
        "a000016.sp58.txt",
    ],
    "sp59": [
        "au2000CQ101.sp59.txt",
        "au2006VV2.sp59.txt",
        "au2003CJ11.sp59.txt",
        "a007304.sp59.txt",
        "a086039.sp59.txt",
        "au2002EQ9.sp59.txt",
    ],
    "sp60": [
        "a026858.sp60.txt",
    ],
    "sp61": [
        "a000364.sp61.txt",
        "a004954.sp61.txt",
        "a085275.sp61.txt",
        "a145656.sp61.txt",
        "a000021.sp61.txt",
        "au2007FV42.sp61.txt",
        "au2003MT9.sp61.txt",
    ],
    "sp62": [
        "au2007DT103n1.sp62.txt",
        "au2007DT103n2.sp62.txt",
        "au2007PF28.sp62.txt",
    ],
    "sp63": [
        "au2007pu11.sp63.txt",
        "au2007lr32.sp63.txt",
    ],
    "sp64": [
        "a000261.sp64.txt",
        "au2002RC118.sp64.txt",
        "a000364.sp64.txt",
        "a162117.sp64.txt",
        "a004744.sp64.txt",
        "au2001RY47.sp64.txt",
        "a004954.sp64.txt",
        "a000110.sp64.txt",
        "au2007RF5.sp64.txt",
        "au2002SV.sp64.txt",
        "au2007SG11.sp64.txt",
        "a162117-note.txt",
        "a001406.sp64.txt",
    ],
    "sp65": [
        "a002956.sp65.txt",
        "a159493.sp65.txt",
        "a000584.sp65.txt",
        "a099907.sp65.txt",
        "a011500.sp65.txt",
    ],
    "sp66": [
        "a153591.sp66.txt",
        "a170891.sp66.txt",
    ],
    "sp67": [
        "a090403.sp67.txt",
        "a005720.sp67.txt",
        "a068359.sp67.txt",
        "a000006.sp67.txt",
        "a099935.sp67.txt",
        "au2008AF4.sp67.txt",
        "a005131.sp67.txt",
        "a086324.sp67.txt",
        "au2000CH59.sp67.txt",
        "a003288.sp67.txt",
    ],
    "sp68": [
        "a011398.sp68.txt",
        "a001620.sp68.txt",
        "au2008CL1.sp68.txt",
        "au2008ED8.sp68.txt",
        "a000006.sp68.txt",
        "a003691.sp68.txt",
        "a153591.sp68.txt",
    ],
    "sp69": [
        "au2005NB7.sp69.txt",
        "au2004VW14.sp69.txt",
        "a037384.sp69.txt",
        "a000244.sp69.txt",
    ],
    "sp70": [
        "au2008DE.sp70.txt",
        "a000244.sp70.txt",
        "a053319.sp70.txt",
        "au2008HW1.sp70.txt",
        "a000041.sp70.txt",
        "au2007VR183.sp70.txt",
        "au2008HS3.sp70.txt",
    ],
    "sp71": [
        "a173689.sp71.txt",
        "a035107.sp71.txt",
        "au2000AD205.sp71.txt",
        "au2007LL.sp71.txt",
        "a003288.sp71.txt",
        "a000044.sp71.txt",
        "a007889.sp71.txt",
        "a007092.sp71.txt",
    ],
    "sp72": [
        "a007889.sp72.txt",
        "a001917.sp72.txt",
        "a001951.sp72.txt",
        "au2008JT35.sp72.txt",
        "au2005TU45.sp72.txt",
        "a007482.sp72.txt",
    ],
    "sp73": [
        "au1998BE7.sp73.txt",
        "a004179.sp73.txt",
        "a163000.sp73.txt",
        "a016960.sp73.txt",
        "a164400.sp73.txt",
        "a001627.sp73.txt",
        "a039572.sp73.txt",
        "a008567.sp73.txt",
    ],
    "sp74": [
        "a004179.sp74.txt",
        "a137032.sp74.txt",
        "a185851.sp74.txt",
        "a001627.sp74.txt",
        "au1998SJ70.sp74.txt",
        "a016960.sp74.txt",
        "a066146.sp74.txt",
        "a008567.sp74.txt",
        "a102528.sp74.txt",
        "au2008QS11.sp74.txt",
    ],
    "sp75": [
        "a000079.sp75.txt",
        "au2008US4.sp75.txt",
        "a001865.sp75.txt",
        "a000009.sp75.txt",
        "au2008SQ1.sp75.txt",
        "a000245.sp75.txt",
    ],
    "sp76": [
        "au2006VB14.sp76.txt",
        "a053430.sp76.txt",
        "a085236.sp76.txt",
        "a000021.sp76.txt",
        "a000032.sp76.txt",
        "a000532.sp76.txt",
        "au2008UE7.sp76.txt",
        "a162900.sp76.txt",
        "au2008WD14.sp76.txt",
        "a001627.sp76.txt",
        "au2006SZ217.sp76.txt",
        "a000863.sp76.txt",
    ],
    "sp77": [
        "a001627.sp77.txt",
        "a141052.sp77.txt",
        "a016834.sp77.txt",
    ],
    "sp78": [
        "a206910.sp78.txt",
        "au2009DD45.sp78.txt",
        "a003554.sp78.txt",
        "a014402.sp78.txt",
    ],
    "sp79": [
        "a052768.sp79.txt",
        "a175706.sp79.txt",
        "au2004VC.sp79.txt",
        "au2007BT2.sp79.txt",
        "a000045.sp79.txt",
        "au2008SV11.sp79.txt",
        "a143651.sp79.txt",
    ],
    "sp80": [
        "a175706.sp80.txt",
        "a207945.sp80.txt",
        "a143651.sp80.txt",
        "a001036.sp80.txt",
        "a020786.sp80.txt",
        "a005604.sp80.txt",
    ],
    "sp81": [
        "au2002lv.sp81.txt",
        "a020786.sp81.txt",
        "a154244.sp81.txt",
        "a005620.sp81.txt",
        "a004015.sp81.txt",
    ],
    "sp82": [
        "a162483.sp82.txt",
        "au2002lv.sp82.txt",
        "a015745.sp82.txt",
    ],
    "sp83": [
        "a017274.sp83.txt",
        "au2009qj9.sp83.txt",
        "a163697.sp83.txt",
    ],
    "sp84": [
        "a000062.sp84.txt",
        "a005626.sp84.txt",
        "a143487.sp84.txt",
        "a000041.sp84.txt",
        "a159402.sp84.txt",
        "a152664.sp84.txt",
        "au2009kc3.sp84.txt",
        "au2000co101.sp84.txt",
        "a003819.sp84.txt",
        "a138883.sp84.txt",
        "a000268.sp84.txt",
        "a016834.sp84.txt",
    ],
    "sp85": [
        "a000762.sp85.txt",
        "au2005rq6.sp85.txt",
        "a000213.sp85.txt",
        "a217807.sp85.txt",
        "a004455.sp85.txt",
        "a003819.sp85.txt",
        "a217796.sp85.txt",
        "a001910.sp85.txt",
        "a003552.sp85.txt",
    ],
    "sp86": [
        "a000004.sp86.txt",
        "au1998vf32.sp86.txt",
        "a000142.sp86.txt",
    ],
    "sp87": [
        "au2009UV18.sp87.txt",
        "a004455.sp87.txt",
        "a005131.sp87.txt",
        "a103067.sp87.txt",
        "a024761.sp87.txt",
        "a000113.sp87.txt",
        "a054789.sp87.txt",
    ],
    "sp88": [
        "au2001PT9.sp88.txt",
        "a003554.sp88.txt",
        "a001627.sp88.txt",
        "a011066.sp88.txt",
        "a229007.sp88.txt",
        "au2009UN3.sp88.txt",
        "au2006EE1.sp88.txt",
    ],
    "sp89": [
        "a054789.sp89.txt",
        "au2010CC180.sp89.txt",
        "a085839.sp89.txt",
        "a068216.sp89.txt",
        "a004580.sp89.txt",
        "a000339.sp89.txt",
        "a000845.sp89.txt",
        "a001021.sp89.txt",
        "a088254.sp89.txt",
        "a000002.sp89.txt",
        "a218863.sp89.txt",
        "a003554.sp89.txt",
        "a000145.sp89.txt",
        "a000301.sp89.txt",
        "a000679.sp89.txt",
        "a002463.sp89.txt",
        "a152931.sp89.txt",
    ],
    "sp90": [
        "au2010CM44.sp90.txt",
        "au2005YU55.sp90.txt",
    ],
    "sp91": [
        "a001076.sp91.txt",
        "a237442.sp91.txt",
        "a000141.sp91.txt",
        "a003671.sp91.txt",
        "a000377.sp91.txt",
        "a085818.sp91.txt",
        "a003613.sp91.txt",
        "a234061.sp91.txt",
    ],
    "sp92": [
        "a000984.sp92.txt",
        "a000180.sp92.txt",
        "a000103.sp92.txt",
        "a000029.sp92.txt",
        "a002078.sp92.txt",
        "a000039.sp92.txt",
        "a003833.sp92.txt",
        "a012923.sp92.txt",
        "a000288.sp92.txt",
        "a000108.sp92.txt",
        "a000793.sp92.txt",
        "a003674.sp92.txt",
        "a241662.sp92.txt",
        "a000082.sp92.txt",
        "a000057.sp92.txt",
        "au2003MJ4.sp92.txt",
        "a000026.sp92.txt",
        "a000371.sp92.txt",
        "a003613.sp92.txt",
        "au2005ED318.sp92.txt",
        "a161998.sp92.txt",
        "a000808.sp92.txt",
        "a000119.sp92.txt",
        "a085989.sp92.txt",
        "a000030.sp92.txt",
    ],
    "sp93": [
        "a006239.sp93.txt",
        "a002449.sp93.txt",
        "a017107.sp93.txt",
        "a152558.sp93.txt",
        "a138404.sp93.txt",
        "a019764.sp93.txt",
        "a007336.sp93.txt",
        "a025916.sp93.txt",
        "a154029.sp93.txt",
        "a014222.sp93.txt",
        "au2010RF12.sp93.txt",
        "a006249.sp93.txt",
        "a000416.sp93.txt",
        "a000316.sp93.txt",
        "a007472.sp93.txt",
        "au2010RX30.sp93.txt",
        "au2010LY63.sp93.txt",
        "a066146.sp93.txt",
        "a000403.sp93.txt",
    ],
    "sp94": [
        "a001912.sp94.txt",
        "a153814.sp94.txt",
        "au2000TU28.sp94.txt",
        "a000025.sp94.txt",
        "a000158.sp94.txt",
        "au2010TD54.sp94.txt",
        "au2003UV11.sp94.txt",
        "a065679.sp94.txt",
        "a248083.sp94.txt",
        "au2005XU77.sp94.txt",
        "a000061.sp94.txt",
        "au2007RU17.sp94.txt",
        "a000037.sp94.txt",
        "a100926.sp94.txt",
        "a066146.sp94.txt",
        "a003485.sp94.txt",
        "a000389.sp94.txt",
    ],
    "sp95": [
        "a005392.sp95.txt",
        "au2003UV11.sp95.txt",
        "a000237.sp95.txt",
    ],
    "sp96": [
        "a000003.sp96.txt",
        "a000020.sp96.txt",
        "a000017.sp96.txt",
        "a002990.sp96.txt",
        "a000023.sp96.txt",
        "a154715.sp96.txt",
        "au2000AZ93.sp96.txt",
        "a002036.sp96.txt",
        "a005496.sp96.txt",
        "a096590.sp96.txt",
    ],
    "sp97": [
        "a003029.sp97.txt",
        "a000534.sp97.txt",
        "a022959.sp97.txt",
        "a000298.sp97.txt",
        "a005209.sp97.txt",
        "a001709.sp97.txt",
        "a253841.sp97.txt",
        "a000720.sp97.txt",
        "a000036.sp97.txt",
        "a163250.sp97.txt",
        "a000596.sp97.txt",
        "a036284.sp97.txt",
    ],
    "sp98": [
        "a011885.sp98.txt",
        "a003554.sp98.txt",
        "a000047.sp98.txt",
        "a089958.sp98.txt",
        "au2011BE38.sp98.txt",
        "au2008FU6.sp98.txt",
        "a267729.sp98.txt",
        "a023187.sp98.txt",
        "a137170.sp98.txt",
        "a000081.sp98.txt",
    ],
    "sp99": [
        "a141432.sp99.txt",
        "au2001QP181.sp99.txt",
        "au2004EM66.sp99.txt",
        "a000121.sp99.txt",
        "au2006JF42.sp99.txt",
        "a003288.sp99.txt",
        "a033745.sp99.txt",
        "a000160.sp99.txt",
        "a274138.sp99.txt",
        "a260141.sp99.txt",
        "a000854.sp99.txt",
    ],
    "sp100": [
        "a267494.sp100.txt",
        "au1998KM3.sp100.txt",
        "a037336.sp100.txt",
        "au2011EZ78.sp100.txt",
        "a169675.sp100.txt",
        "a003988.sp100.txt",
        "au2001QP181.sp100.txt",
        "a000038.sp100.txt",
        "a020086.sp100.txt",
        "a007358.sp100.txt",
    ],
    "dm02": [
        "a001412.dm02.txt",
        "a001701.dm02.txt",
        "a033952.dm02.txt",
        "a010327.dm02.txt",
        "a027151.dm02.txt",
        "a000354.dm02.txt",
        "a056690.dm02.txt",
        "a009763.dm02.txt",
        "a035925.dm02.txt",
    ],
    "sp101": [
        "a000433.sp101.txt",
        "a005577.sp101.txt",
        "a000600.sp101.txt",
        "a000939.sp101.txt",
        "au2009av.sp101.txt",
        "au2011ov4.sp101.txt",
        "a000631.sp101.txt",
        "au2011ga55.sp101.txt",
        "a138524.sp101.txt",
        "a000208.sp101.txt",
    ],
    "dm03": [
        "a031735.dm03.txt",
        "a041357.dm03.txt",
        "a138524.dm03.txt",
        "au2011ps.dm03.txt",
    ],
    "sp102": [
        "au2011lj19.sp102.txt",
        "a000118.sp102.txt",
        "a163081.sp102.txt",
        "au2000sp43.sp102.txt",
        "a136923.sp102.txt",
        "au2011sr5.sp102.txt",
        "au2011pt1.sp102.txt",
        "a138524.sp102.txt",
        "au2000ej37.sp102.txt",
        "au1991tf3.sp102.txt",
        "a000925.sp102.txt",
        "a000433.sp102.txt",
        "a283460.sp102.txt",
    ],
    "sp103": [
        "a137799.sp103.txt",
        "a297418.sp103.txt",
        "au2004qj7.sp103.txt",
        "a163081.sp103.txt",
        "a003352.sp103.txt",
        "au2011tn9.sp103.txt",
        "au1990ua.sp103.txt",
        "a000433.sp103.txt",
        "a007341.sp103.txt",
        "a001036.sp103.txt",
        "a004183.sp103.txt",
        "a001943.sp103.txt",
    ],
    "sp104": [
        "a001768.sp104.txt",
        "a154302.sp104.txt",
        "a219071.sp104.txt",
        "au2003fh.sp104.txt",
        "a008567.sp104.txt",
        "a001036.sp104.txt",
        "au2011sr69.sp104.txt",
    ],
    "dm04": [
        "a170502.dm04.txt",
        "a027791.dm04.txt",
        "a175706.dm04.txt",
        "a008388.dm04.txt",
        "a019570.dm04.txt",
        "a018098.dm04.txt",
        "a032222.dm04.txt",
        "a029731.dm04.txt",
        "a000001.dm04.txt",
        "a105840.dm04.txt",
        "a007057.dm04.txt",
        "a028218.dm04.txt",
        "a001036.dm04.txt",
    ],
    "sp105": [
        "au1994ck1.sp105.txt",
        "a032906.sp105.txt",
        "au2011ya.sp105.txt",
        "au2006mj10.sp105.txt",
        "a000433.sp105.txt",
        "a175706-obsB.sp105.txt",
        "a289315.sp105.txt",
        "a175706-obsA.sp105.txt",
        "au2001ye4.sp105.txt",
        "a099799.sp105.txt",
        "au2006vy13.sp105.txt",
        "a001620.sp105.txt",
        "au2011sl102.sp105.txt",
        "au1999xp35.sp105.txt",
        "a024475.sp105.txt",
    ],
    "dm05": [
        "a040573.dm05.txt",
        "a002234.dm05.txt",
        "a031393.dm05.txt",
        "a016520.dm05.txt",
    ],
    "dm06": [
        "a040537.dm06.txt",
        "a007302.dm06.txt",
    ],
    "dm07": [
        "a006067.dm07.txt",
        "a017818.dm07.txt",
        "a092516.dm07.txt",
        "a141018.dm07.txt",
        "a010977.dm07.txt",
        "a036256.dm07.txt",
        "a075810.dm07.txt",
        "a163243.dm07.txt",
        "a002234.dm07.txt",
        "au2012dh4.dm07.txt",
    ],
    "sp106": [
        "au2006vy13.sp106.txt",
        "a312473.sp106.txt",
        "a009400.sp106.txt",
        "au2012aa11.sp106.txt",
        "a003352.sp106.txt",
    ],
    "sp107": [
        "a312473.sp107.txt",
        "a152754.sp107.txt",
        "au2006vy13.sp107.txt",
        "a004402.sp107.txt",
        "au2011yv15.sp107.txt",
        "au2012aa11.sp107.txt",
        "a006455.sp107.txt",
        "au2001sz269.sp107.txt",
        "a005026.sp107.txt",
        "a086667.sp107.txt",
        "au2012fy23.sp107.txt",
        "a000334.sp107.txt",
    ],
    "sp108": [
        "a014390.sp108.txt",
        "au1998he3.sp108.txt",
        "au2010kx7.sp108.txt",
        "a019652.sp108.txt",
        "a005836.sp108.txt",
        "au2003wh166.sp108.txt",
        "a297274.sp108.txt",
    ],
    "sp109": [
        "au2012lz1.sp109.txt",
        "a004674.sp109.txt",
        "au2012kp24.sp109.txt",
        "au2012kt42.sp109.txt",
        "au2011wv134.sp109.txt",
    ],
    "dm08": [
        "a024436.dm08.txt",
        "a000636.dm08.txt",
        "au2007le.dm08.txt",
        "a000212.dm08.txt",
        "au2001lb.dm08.txt",
    ],
    "dm09": [
        "a021646.dm09.txt",
        "a003283.dm09.txt",
        "a024673.dm09.txt",
        "a013577.dm09.txt",
        "a045876.dm09.txt",
    ],
    "sp110": [
        "au2012nj.sp110.txt",
        "a007172.sp110.txt",
        "a153958.sp110.txt",
        "a005836.sp110.txt",
        "au2012oq.sp110.txt",
        "a084480.sp110.txt",
        "a326732.sp110.txt",
        "a276397.sp110.txt",
        "au2006cf.sp110.txt",
        "a329291.sp110.txt",
        "a154007.sp110.txt",
        "a329614.sp110.txt",
    ],
    "dm10": [
        "a013724.dm10.txt",
        "a003573.dm10.txt",
        "a326732.dm10.txt",
    ],
    "sp111": [
        "a004769.sp111.txt",
        "a000069.sp111.txt",
        "a000887.sp111.txt",
        "a000067.sp111.txt",
        "a060716.sp111.txt",
        "au2012qv2.sp111.txt",
        "a000010.sp111.txt",
    ],
    "dm11": [
        "a045876.dm11.txt",
        "a003573.dm11.txt",
        "a011952.dm11.txt",
        "a330825.dm11.txt",
        "a009570.dm11.txt",
        "a030698.dm11.txt",
        "a009898.dm11.txt",
        "au2012rm15.dm11.txt",
        "a009404.dm11.txt",
        "a005202.dm11.txt",
        "a035102.dm11.txt",
        "a003385.dm11.txt",
        "a014291.dm11.txt",
        "a091450.dm11.txt",
        "a017906.dm11.txt",
        "a247264.dm11.txt",
    ],
    "sp112": [
        "a021809.sp112p2.txt",
        "a329520.sp112p2.txt",
        "a010302.sp112p2.txt",
        "au2012QE50.sp112p2.txt",
        "a005646.sp112p2.txt",
        "a155334.sp112p2.txt",
        "au2012QG42.sp112p2.txt",
        "a326732.sp112p2.txt",
        "au2012QG42.sp112p1.txt",
        "a214869.sp112p2.txt",
    ],
    "sp113": [
        "au2005GQ21.sp113n1.txt",
        "au2012TC4.sp113n1.txt",
        "a045610.sp113n2.txt",
        "a007350.sp113n1.txt",
        "au2012RV16.sp113n1.txt",
        "au2002ua31.sp113n2.txt",
        "a027268.sp113n1.txt",
        "au2002tr190.sp113n2.txt",
        "a136993.sp113n2.txt",
        "a214869.sp113n2.txt",
        "au2012QE50.sp113n1.txt",
    ],
    "sp114": [
        "a162911.sp114n1.txt",
        "a099907.sp114n2.txt",
        "a099907.sp114n1.txt",
        "a329338.sp114n1.txt",
    ],
    "dm12": [
        "a001474.dm12.txt",
        "a013137.dm12.txt",
        "a015112.dm12.txt",
        "a021205.dm12.txt",
        "a035925.dm12.txt",
    ],
    "sp115": [
        "a053424.sp115n2.txt",
        "a192563.sp115n4.txt",
        "aANA-1127.sp115n3.txt",
        "aANA-0032.sp115n4.txt",
        "a002062.sp115n2.txt",
        "au2000WL63.sp115n4.txt",
        "au2001YM2.sp115n3.txt",
        "au1994XD.sp115n2.txt",
        "a345705.sp115n1.txt",
        "aANA-0920.sp115n4.txt",
        "a333358.sp115n3.txt",
        "au2003UC20.sp115n2.txt",
        "au2012XQ55.sp115n2.txt",
        "aANA-0934.sp115n2.txt",
        "a054041.sp115n4.txt",
        "au2012XH112.sp115n3.txt",
        "aHD23169.sp115n4.txt",
        "a277127.sp115n2.txt",
        "a004179.sp115n1.txt",
        "a005646.sp115n1.txt",
        "aHD23169.sp115n2.txt",
        "a052762.sp115n4.txt",
        "aANA-0934.sp115n3.txt",
        "a154347.sp115n4.txt",
        "a203471.sp115n2.txt",
        "au2012XJ134.sp115n3.txt",
        "aANA-1127.sp115n2.txt",
        "aANA-0920.sp115n2.txt",
    ],
    "sp116": [
        "a099942.sp116.txt",
        "aANA-1002.sp116.txt",
        "au2010BB.sp116.txt",
        "aANA-0356.sp116.txt",
        "a004179.sp116.txt",
        "aANA-0419.sp116.txt",
        "aANA-1133.sp116.txt",
        "a137199.sp116.txt",
        "au2009DV42.sp116.txt",
        "au2013AH11.sp116.txt",
        "au2002AY1.sp116.txt",
        "a262623.sp116.txt",
    ],
    "dm13": [
        "a008838.dm13.txt",
        "a097335.dm13.txt",
        "a017889.dm13.txt",
    ],
    "sp117": [
        "au2013AT72.sp117.txt",
        "a015107.sp117.txt",
        "a343098.sp117.txt",
        "a020439.sp117.txt",
        "aANA-0615.sp117.txt",
        "aANA-0419.sp117.txt",
        "a007468.sp117.txt",
        "a011284.sp117.txt",
        "au2006YT13.sp117.txt",
        "a099942.sp117.txt",
        "aANA-0330.sp117.txt",
        "a042946.sp117.txt",
        "au2005DD.sp117.txt",
        "a052762.sp117.txt",
    ],
    "dm14": [
        "a125102.dm14.txt",
        "a022693.dm14.txt",
        "a019312.dm14.txt",
        "a020452.dm14.txt",
        "a033917.dm14.txt",
        "a060631.dm14.txt",
        "a059115.dm14.txt",
        "a029731.dm14.txt",
        "a011406.dm14.txt",
        "a016025.dm14.txt",
        "a004264.dm14.txt",
    ],
    "sp118": [
        "au2009UM3.sp118n2.txt",
        "au2013EC20.sp118.txt",
        "au2013ET.dp08.txt",
        "a096315.sp118n2.txt",
        "au2005EM30.sp118n1.txt",
        "au2013ED68.sp118n2.txt",
        "au2003GW.sp118n2.txt",
        "a001627.sp118n1.txt",
    ],
    "sp119": [
        "au2013FY13.sp119n1.txt",
        "a009983.sp119n2.txt",
        "a004034.sp119n2.txt",
        "a005645.sp119n1.txt",
        "a285263.sp119n2.txt",
        "au2009UM3.sp119n2.txt",
        "au1993UC.sp119n1.txt",
        "au2005TS15.sp119n1.txt",
        "au2010VA1.sp119n1.txt",
        "a017288.sp119n2.txt",
        "au2013EO126.sp119n1.txt",
        "a010295.sp119n2.txt",
    ],
    "sp120": [
        "au2013HT15.sp120n2.txt",
        "a137199.sp120n2.txt",
        "au2006FJ9.sp120n1.txt",
        "aANA-1425.sp120n2.txt",
        "a077578.sp120n2.txt",
        "aHD170379.sp120n2.txt",
        "a274855.sp120n2.txt",
        "au2011WU95.sp120n1.txt",
        "au2005NZ6.sp120n1.txt",
        "a359592.sp120n2.txt",
        "aPluto.sp120n2.txt",
        "au2009SQ104.sp120n1.txt",
        "a285263.sp120n1.txt",
        "a152756.sp120n1.txt",
        "a048049.sp120n2.txt",
        "a242643.sp120n1.txt",
        "au2009BL2.sp120n1.txt",
        "a360191.sp120n1.txt",
        "a111114.sp120n2.txt",
        "aHIP89046.sp120n1.txt",
        "a038707.sp120n2.txt",
        "aANA-1744.sp120n2.txt",
    ],
    "dm15": [
        "a018853.dm15.txt",
        "a031455.dm15.txt",
        "a007888.dm15.txt",
        "a038699.dm15.txt",
    ],
    "sp121": [
        "aPluto.sp121n1.txt",
        "a285263.sp121n1.txt",
        "a285263.sp121n2.txt",
        "a007753.sp121n1.txt",
        "a096189.sp121n1.txt",
        "aANA-2144.sp121n1.txt",
        "a163364.sp121n1.txt",
        "aANA-2043.sp121n2.txt",
        "au2013KL6.sp121n1.txt",
        "aANA-2133.sp121n1.txt",
        "au2002JR9.sp121n2.txt",
        "a017188.sp121n2.txt",
        "au2013JR28.sp121n1.txt",
        "a088604.sp121n1.txt",
        "a034931.sp121n2.txt",
        "aANA-2133.sp121n2.txt",
        "a003749.sp121n2.txt",
        "a163249.sp121n1.txt",
        "aANA-2025.sp121n2.txt",
        "aANA-2144.sp121n2.txt",
    ],
    "sp122": [
        "a017188.sp122.txt",
        "a243147.sp122.txt",
        "aANA-2043.sp122.txt",
        "a294739.sp122.txt",
        "a285263.sp122.txt",
        "aANA-2025.sp122.txt",
        "a052762.sp122.txt",
        "au2007XY9.sp122.txt",
        "a163364.sp122.txt",
        "a014390.sp122.txt",
        "a009068.sp122.txt",
        "a361123.sp122.txt",
        "au2006NL.sp122.txt",
        "aPluto.sp122.txt",
    ],
    "dm16": [
        "a129761.dm16.txt",
        "au2010TN54.dm16.txt",
        "a077653.dm16.txt",
        "a055543.dm16.txt",
        "a111186.dm16.txt",
        "a034931.dm16.txt",
        "a010715.dm16.txt",
        "a023615.dm16.txt",
        "a044028.dm16.txt",
        "a081850.dm16.txt",
        "a017716.dm16.txt",
        "a018810.dm16.txt",
        "a083463.dm16.txt",
        "a046456.dm16.txt",
        "a175158.dm16.txt",
        "a052726.dm16.txt",
        "a109068.dm16.txt",
        "a052760.dm16.txt",
    ],
    "sp123": [
        "au2013PS13.sp123.txt",
        "a052760.sp123.txt",
        "au2006AO4.sp123.txt",
        "au2013PJ10.sp123.txt",
        "a140039.sp123.txt",
        "a004905.sp123.txt",
        "a006037.sp123.txt",
        "a354952.sp123.txt",
        "a277475.sp123.txt",
        "a001589.sp123.txt",
        "a285263.sp123.txt",
        "au2010TN54.sp123.txt",
        "aPluto.sp123.txt",
        "au2013BO76.sp123.txt",
    ],
    "sp124": [
        "aANA-0032.sp124.txt",
        "a008306.sp124.txt",
        "au2010CF19.sp124.txt",
        "aPluto.sp124.txt",
        "au2013HG20.sp124.txt",
        "a004450.sp124.txt",
        "a152627.sp124.txt",
        "a329437.sp124.txt",
        "au2007CN26.sp124.txt",
        "au2010UX95.sp124.txt",
        "a137126.sp124.txt",
        "au2013QJ10.sp124.txt",
    ],
    "sp125": [
        "a006063.sp125n2.txt",
        "a368664.sp125n1.txt",
        "a360433.sp125n2.txt",
        "a007745.sp125n2.txt",
        "a269690.sp125n2.txt",
        "a065733.sp125n2.txt",
        "a101703.sp125n1.txt",
        "au2013SU24.sp125n2.txt",
        "a006070.sp125n1.txt",
        "a251346.sp125n2.txt",
        "a009950.sp125n1.txt",
        "a088710.sp125n2.txt",
        "au2012TS.sp125n1.txt",
        "au2008HB38.sp125n1.txt",
        "a074096.sp125n2.txt",
        "a168378.sp125n2.txt",
        "au2012ER14.sp125n1.txt",
        "au2013SW24.sp125n1.txt",
        "a162566.sp125n2.txt",
        "au2013RH74.sp125n1.txt",
        "aANA-0032.sp125n2.txt",
        "a039565.sp125n1.txt",
    ],
    "sp126": [
        "a001981.sp126.txt",
        "au2006HU30.sp126.txt",
        "au2013UQ4.sp126.txt",
        "au2003WX25.sp126.txt",
        "a006070.sp126.txt",
        "a251346.sp126.txt",
        "a017511.sp126.txt",
        "a008014.sp126.txt",
    ],
    "sp127": [
        "au2013XY8.sp127.txt",
        "a000216.sp127.txt",
        "au2013NJ.sp127.txt",
        "au2010CL19.sp127.txt",
    ],
    "sp128": [
        "au2010XZ67.sp128.txt",
        "au2009XZ1.sp128.txt",
    ],
    "dm17": [
        "au2011BT15.dm17.txt",
        "a020596.dm17.txt",
        "a004067.dm17.txt",
        "a095560.dm17.txt",
        "a012264.dm17.txt",
    ],
    "sp129": [
        "au2010SL13.sp129.txt",
    ],
    "sp130": [
        "a085990.sp130.txt",
    ],
    "sp201": [
        "au2014QO33.sp201n1.txt",
        "au2014PR62.sp201n1.txt",
        "au2014RC.sp201n2.txt",
        "au2003EG16.sp201n1.txt",
        "au2014HF177.sp201n1.txt",
        "a000433.sp201n1.txt",
        "au2014NE63.sp201n1.txt",
        "au2008OB9.sp201n1.txt",
        "a005159.sp201n1.txt",
        "a125739.sp201n1.txt",
        "au2010MR.sp201n1.txt",
        "au2008RG1.sp201n1.txt",
        "a143624.sp201n1.txt",
    ],
    "sp202": [
        "a090075.sp202.txt",
        "a275611.sp202.txt",
        "a399307.sp202.txt",
        "a143624.sp202.txt",
        "au2014SF304.sp202.txt",
        "au2014PL51.sp202.txt",
        "a410778.sp202.txt",
        "au2008OB9.sp202.txt",
        "au2014SS1.sp202.txt",
    ],
    "dm18": [
        "a168381.dm18.txt",
    ],
    "sp203": [
        "a036017.sp203.txt",
        "au2014UR.sp203.txt",
        "a002340.sp203.txt",
        "au2007DL41.sp203.txt",
        "a159533.sp203.txt",
        "a241676.sp203.txt",
        "au2014HR178.sp203.txt",
        "a205640.sp203.txt",
        "a004401.sp203.txt",
        "a410195.sp203.txt",
        "a313591.sp203.txt",
    ],
    "sp204": [
        "a403049.sp204n1.txt",
        "a068548.sp204n1.txt",
        "au2014WZ120.sp204n1.txt",
        "au2014VQ.sp204n1.txt",
        "au2014WF201.sp204n1.txt",
        "a012538.sp204n1.txt",
        "a003200.sp204n2.txt",
        "a003200.sp204n1.txt",
        "au2008VF.sp204n1.txt",
        "a099248.sp204n1.txt",
        "a068063.sp204n1.txt",
        "a171730.sp204n1.txt",
        "au1998WT7.sp204n1.txt",
        "a214088.sp204n1.txt",
    ],
    "dm19": [
        "au1998GL10.dm19n2.txt",
        "a190208.dm19n2.txt",
        "a283319.dm19n2.txt",
        "a018620.dm19n1.txt",
        "au1997VM4.dm19n2.txt",
        "au2014WQ201.dm19n2.txt",
        "a283377.dm19n1.txt",
        "a306453.dm19n1.txt",
        "a164670.dm19n1.txt",
        "a409995.dm19n1.txt",
        "a003200.dm19n2.txt",
        "a316934.dm19n1.txt",
        "au2014VH2.dm19n2.txt",
        "a363019.dm19n1.txt",
        "a412976.dm19n2.txt",
        "a138852.dm19n1.txt",
        "a363505.dm19n1.txt",
        "a138947.dm19n2.txt",
    ],
    "sp205": [
        "au2014UF206.sp205.txt",
        "au2014QK434.sp205.txt",
        "au2011WK15.sp205.txt",
        "a410088.sp205.txt",
        "a085804.sp205.txt",
        "au2007EC.sp205.txt",
        "au2015BC.sp205.txt",
        "au2014RL12.sp205.txt",
    ],
    "sp206": [
        "au2013BK18.sp206.txt",
        "a002062.sp206.txt",
        "a422686.sp206.txt",
        "au2015AZ43.sp206.txt",
        "a376246.sp206.txt",
        "au2005NA56.sp206.txt",
        "au2014YM9.sp206.txt",
        "au2015BF92.sp206.txt",
    ],
    "sp207": [
        "a306453.sp207.txt",
        "a311554.sp207.txt",
        "a416151.sp207.txt",
        "a134746.sp207.txt",
        "au2015DB.sp207.txt",
        "a090416.sp207.txt",
        "au2014EK24.sp207.txt",
        "a137924.sp207.txt",
    ],
    "sp208": [
        "au2010XY72.sp208.txt",
        "a004055.sp208.txt",
    ],
    "sp209": [
        "a416071.sp209.txt",
        "a001036.sp209.txt",
        "a294739.sp209.txt",
        "a159504.sp209.txt",
        "au2012tm139.sp209.txt",
        "au2010pr66.sp209.txt",
        "au2015jj2.sp209.txt",
        "au2015jy1.sp209.txt",
        "a438429.sp209.txt",
        "a206378.sp209.txt",
        "a380981.sp209.txt",
        "a348400.sp209.txt",
        "a385186.sp209.txt",
    ],
    "dm20": [
        "a042946.dm20n2.txt",
        "a281396.dm20n1.txt",
        "a333889.dm20n1.txt",
        "a053426.dm20n2.txt",
        "a007822.dm20n1.txt",
        "a437975.dm20n2.txt",
    ],
    "sp210": [
        "au2005lw7.sp210.txt",
        "au2012sg32.sp210.txt",
        "a317699.sp210.txt",
        "au2012tm139.sp210.txt",
        "au2015rf36.sp210.txt",
        "au2015sy.sp210.txt",
        "au2001rb12.sp210.txt",
        "a389694.sp210.txt",
    ],
    "sp211": [
        "a438071.sp211.txt",
        "au2015FS332.sp211.txt",
        "au2015SZ.sp211.txt",
        "au2015OL35.sp211.txt",
        "a440022.sp211.txt",
        "a348400.sp211.txt",
        "au2009TK.sp211.txt",
        "a187828.sp211.txt",
        "au2001XT4.sp211.txt",
    ],
    "sp212": [
        "a194268.sp212.txt",
        "a410777.sp212.txt",
        "a163000.sp212.txt",
        "au2006uy64.sp212.txt",
        "a001620.sp212.txt",
    ],
    "dm22": [
        "a348400.dm22.txt",
        "a377052.dm22.txt",
        "a155110.dm22.txt",
        "a011318.dm22.txt",
        "a413577.dm22.txt",
        "a443892.dm22.txt",
        "au2015ta25.dm22.txt",
        "a156015.dm22.txt",
    ],
    "sp213": [
        "a159504.sp213.txt",
        "au2011YS62.sp213.txt",
        "a033342.sp213.txt",
        "a163899.sp213.txt",
    ],
    "sp214": [
        "a138937.sp214.txt",
        "a163899.sp214.txt",
        "a363218.sp214.txt",
        "a194386.sp214.txt",
        "a033342.sp214.txt",
        "a159504.sp214.txt",
        "au2011WN15.sp214.txt",
        "au2015SV2.sp214.txt",
        "a163696.sp214.txt",
    ],
    "sp215": [
        "a337866.sp215.txt",
        "a154807.sp215.txt",
        "a009400.sp215.txt",
        "a324387.sp215.txt",
        "a085990.sp215.txt",
        "au2007BZ72.sp215.txt",
        "a210669.sp215.txt",
        "au2014EK24.sp215.txt",
        "a001685.sp215.txt",
        "a013551.sp215.txt",
        "au2015WH9.sp215.txt",
    ],
    "sp216": [
        "a303959.sp216.txt",
        "a250706.sp216.txt",
        "a130206.sp216.txt",
        "a411201.sp216.txt",
        "a455148.sp216.txt",
        "a007822.sp216.txt",
        "a154807.sp216.txt",
        "a162510.sp216.txt",
        "au2015XB379.sp216.txt",
        "au2016BA14.sp216.txt",
        "au2016ES155.sp216.txt",
        "a144332.sp216.txt",
        "a100553.sp216.txt",
    ],
    "sp217": [
        "a388945.sp217.txt",
        "au2016GU.sp217.txt",
        "a141354.sp217.txt",
        "au2009EC.sp217.txt",
        "a411201.sp217.txt",
        "a035396.sp217.txt",
        "a005863.sp217.txt",
        "a450160.sp217.txt",
        "a001685.sp217.txt",
        "au2005GR33.sp217.txt",
        "a099278.sp217.txt",
        "a363599.sp217.txt",
        "a085628.sp217.txt",
    ],
    "sp218": [
        "a321356.sp218.txt",
        "a458198.sp218.txt",
        "au2016LG.sp218.txt",
        "au2016LV.sp218.txt",
        "au2002LY1.sp218.txt",
        "a331471.sp218.txt",
    ],
    "sp219": [
        "a138970.sp219.txt",
        "a441987.sp219.txt",
        "a464798.sp219.txt",
        "a068346.sp219.txt",
        "a173664.sp219.txt",
        "a129470.sp219.txt",
        "a154244n1.sp219.txt",
        "a210635.sp219.txt",
        "a154244n3.sp219.txt",
        "a385343n1.sp219.txt",
        "a385343n2.sp219.txt",
        "a331471.sp219.txt",
        "a010150.sp219.txt",
    ],
    "sp220": [
        "a213812.sp220.txt",
        "a357024.sp220.txt",
        "a219093.sp220.txt",
    ],
    "sp221": [
        "au2008CJ116.sp221.txt",
        "a001443.sp221.txt",
        "au2016NA1.sp221.txt",
        "a154244.sp221.txt",
        "a005836.sp221.txt",
        "a452389.sp221.txt",
        "au2011BX18.sp221.txt",
    ],
    "sp222": [
        "a002100.sp222.txt",
        "au2014KD91.sp222.txt",
        "a040329.sp222.txt",
        "a007088.sp222.txt",
        "a010636.sp222.txt",
        "au2016LX48.sp222.txt",
    ],
    "sp223": [
        "a162142.sp223.txt",
        "a467963.sp223.txt",
        "au2005TF.sp223.txt",
        "a226237.sp223.txt",
        "a462959.sp223.txt",
        "a137032.sp223.txt",
        "a222205.sp223.txt",
        "a000433.sp223.txt",
        "a178293.sp223.txt",
        "a007341.sp223.txt",
        "a220038.sp223.txt",
        "a004341.sp223.txt",
        "a171819.sp223.txt",
        "a330250.sp223.txt",
    ],
    "sp224": [
        "a468583.sp224.txt",
        "au2005WS3.sp224.txt",
        "a433953.sp224.txt",
        "a105406.sp224.txt",
        "a005587.sp224.txt",
        "au2005TF.sp224.txt",
        "a007341.sp224.txt",
        "a413002.sp224.txt",
        "a363831.sp224.txt",
        "a177287.sp224.txt",
        "a152563.sp224.txt",
        "au2011UU106.sp224.txt",
        "a190541.sp224.txt",
    ],
    "sp225": [
        "au2016XH1.sp225.txt",
        "a106589.sp225.txt",
        "a024447.sp225.txt",
        "a413002.sp225.txt",
        "a468583.sp225.txt",
        "a005653.sp225.txt",
        "a226514.sp225.txt",
        "a326683.sp225.txt",
    ],
    "sp226": [
        "a076818.sp226.txt",
        "au2016UE101.sp226.txt",
        "a414960.sp226.txt",
        "au2016PR8.sp226.txt",
        "a203217.sp226.txt",
        "a007343.sp226.txt",
        "au2016YK.sp226.txt",
        "au2016NL15.sp226.txt",
        "a475665.sp226.txt",
        "au2014AD17.sp226.txt",
    ],
    "sp227": [
        "a265482.sp227.txt",
        "au2015BN509.sp227.txt",
        "a438955.sp227.txt",
        "au2016UU80.sp227.txt",
    ],
    "sp228": [
        "au2011WO41.sp228.txt",
        "a443103.sp228.txt",
        "au2017DA36.sp228.txt",
        "a004179.sp228.txt",
        "au2000CE59.sp228.txt",
    ],
    "sp229": [
        "a370307.sp229.txt",
        "a143404.sp229.txt",
        "au2017CR32.sp229.txt",
        "au2003OV.sp229.txt",
        "a250620.sp229.txt",
    ],
    "sp230": [
        "a484795.sp230.txt",
        "a388838.sp230.txt",
        "a002329.sp230.txt",
        "a190166.sp230.txt",
    ],
    "sp231": [
        "au2017CS.sp231.txt",
        "a011398.sp231.txt",
    ],
    "sp232": [
        "a143992.sp232.txt",
        "a096631.sp232.txt",
        "a416584.sp232.txt",
        "au2017MB1.sp232.txt",
        "a242211.sp232.txt",
        "au2017MC4.sp232.txt",
    ],
    "sp233": [
        "a096631.sp233.txt",
        "a006053.sp233.txt",
        "a416584.sp233.txt",
        "a422699.sp233.txt",
        "a000302.sp233.txt",
        "a265962.sp233.txt",
    ],
    "sp234": [
        "a096631.sp234.txt",
        "a416584.sp234.txt",
    ],
    "sp235": [
        "a141670.sp235.txt",
        "a142464.sp235.txt",
        "a237805.sp235.txt",
    ],
    "sp236": [
        "a190208.sp236.txt",
        "au2017RL.sp236.txt",
    ],
    "sp237": [
        "a496817.sp237.txt",
        "a005189.sp237.txt",
        "a003200.sp237.txt",
        "a496005.sp237.txt",
        "a162998.sp237.txt",
        "a053319.sp237.txt",
        "a008037.sp237.txt",
        "a496816.sp237.txt",
        "au2017RR15.sp237.txt",
        "a265962.sp237.txt",
        "au2014UR116.sp237.txt",
    ],
    "sp238": [
        "a003200.sp238.txt",
        "a457768.sp238.txt",
        "a496816.sp238.txt",
        "au2017VC.sp238.txt",
        "a496818.sp238.txt",
        "a444584.sp238.txt",
        "a140158.sp238.txt",
        "a008037.sp238.txt",
    ],
    "sp239": [
        "a017511.sp239.txt",
        "a140158.sp239.txt",
        "a415029.sp239.txt",
        "a496817.sp239.txt",
        "a002368.sp239.txt",
        "a017182.sp239.txt",
        "a021374.sp239.txt",
        "a333888.sp239.txt",
        "a163696.sp239.txt",
        "a003200.sp239.txt",
    ],
    "sp240": [
        "a505657.sp240.txt",
    ],
    "sp241": [
        "a194126.sp241.txt",
    ],
    "sp242": [
        "au2015dp155.sp242.txt",
        "au2018ej4.sp242.txt",
        "au1999lk1.sp242.txt",
        "au2018ja.sp242.txt",
        "a242708.sp243n2.txt",
        "a450648.sp242.txt",
        "a015745.sp242.txt",
        "a220839.sp242.txt",
    ],
    "sp243": [
        "a067367.sp243n2.txt",
        "a068347.sp243n2.txt",
        "a066391.sp243n2.txt",
        "a001627.sp243n2.txt",
        "au2015dp155.sp243n2.txt",
        "a242708.sp243n2.txt",
        "a006569.sp243n2.txt",
        "au2017ye5.sp243n1.txt",
        "a002061.sp243n1.txt",
        "a469737.sp243n2.txt",
        "au2017ye5.sp243n2.txt",
    ],
    "sp244": [
        "au2018nb.sp244.txt",
        "a398188.sp244.txt",
    ],
    "sp245": [
        "a013553.sp245n2.txt",
        "a099799.sp245n2.txt",
        "a017511.sp239.txt",
        "a000433.sp245n2.txt",
        "a002061.sp245n2.txt",
        "au2018mm8.sp245n1.txt",
        "a013553.sp245n1.txt",
        "a017274.sp245n2.txt",
    ],
    "sp246": [
        "au2018lq2.sp246n1.txt",
        "a163373.sp246n1.txt",
        "a017511.sp246n1.txt",
        "au2018ke3.sp246n2.txt",
        "a017274.sp246n1.txt",
        "au2018qu1.sp246n2.txt",
        "a438429.sp246n1.txt",
        "a031345.sp246n1.txt",
        "a001627.sp246n1.txt",
        "a457260.sp246n1.txt",
        "au2011ua.sp246n2.txt",
        "a481394.sp246n1.txt",
        "au2015fp118.sp246n2.txt",
        "au2018qv1.sp246n2.txt",
        "au2018rc.sp246n2.txt",
    ],
    "sp247": [
        "au2011wa.sp247n2.txt",
        "a000433.sp247n1.txt",
        "a163899.sp247n2.txt",
        "a162687.sp247n2.txt",
        "au2018UQ1.sp247n1.txt",
        "a454100.sp247n1.txt",
        "au2018tt1.sp247n1.txt",
    ],
    "sp248": [
        "au2018wx1.sp248n2.txt",
        "a141053.sp248n1.txt",
        "au2018qn1.sp248n2.txt",
        "a454177.sp248n2.txt",
        "au2018wt1.sp248n2.txt",
        "a518735.sp248n1.txt",
        "au2009sv17.sp248n1.txt",
        "au2010gt7.sp248n1.txt",
        "a538.sp248n1.txt",
        "au2018wd2.sp248n2.txt",
        "a141498.sp248n2.txt",
        "a523811.sp248n1.txt",
    ],
    "sp249": [
        "a376864.sp249.txt",
        "au2007yq56.sp249.txt",
    ],
    "sp250": [
        "au2019bk.sp250.txt",
        "a515767.sp250.txt",
        "a381677.sp250.txt",
        "a523631.sp250.txt",
        "a410088.sp250.txt",
        "au2016az8.sp250.txt",
    ],
    "sp251": [
        "a275611.sp251.txt",
        "au2005yy36.sp251.txt",
        "au2013cw32.sp251.txt",
        "a523631.sp251.txt",
    ],
    "sp252": [
        "au2019cd5.sp252n1.txt",
        "au2019dn.sp252n1.txt",
        "au2015qt9.sp252n1.txt",
        "a000438.sp252n1.txt",
        "au2005fc3.sp252n1.txt",
        "au2019ap3.sp252n2.txt",
        "au2006sk134.sp252n2.txt",
        "a005189.sp252n2.txt",
        "au2009bj81.sp252n1.txt",
        "au2005ah14.sp252n1.txt",
        "a037479.sp252n2.txt",
    ],
    "sp253": [
        "a004581.sp253.txt",
        "a006478.sp253.txt",
        "a381677.sp253.txt",
        "au2019ch.sp253.txt",
        "au2016bd14.sp253.txt",
    ],
    "sp254": [
        "a107888.sp254.txt",
        "a001036.sp254.txt",
        "au2019fu.sp254.txt",
        "a031881.sp254.txt",
        "a141525.sp254.txt",
        "a006478.sp254.txt",
        "a144898.sp254.txt",
    ],
    "sp255": [
        "a112221.sp255.txt",
        "au2019hc.sp255.txt",
        "a153814.sp255.txt",
        "au2011hp.sp255.txt",
        "a297300.sp255.txt",
        "a355256.sp255.txt",
        "au2018xg5.sp255.txt",
    ],
    "sp256": [
        "a141593.sp256n2.txt",
        "a001036.sp256n1.txt",
        "a244670.sp256n1.txt",
        "a416186.sp256n2.txt",
        "a453778.sp256n1.txt",
        "a112661.sp256n2.txt",
        "au2019hc.sp256n1.txt",
        "a495615.sp256n2.txt",
        "a326777.sp256n1.txt",
        "au2008wx32.sp256n1.txt",
        "a000433.sp256n2.txt",
    ],
    "sp257": [
        "a380160.sp257.txt",
        "a002059.sp257.txt",
    ],
    "sp258": [
        "au2008sr1.sp258.txt",
        "au2007tq24.sp258.txt",
        "a002059.sp258.txt",
        "a002100.sp258.txt",
        "au2019gt3.sp258.txt",
        "a000213.sp258.txt",
        "a410778.sp258.txt",
        "a354030.sp258.txt",
    ],
    "sp259": [
        "a000054.sp259n2.txt",
        "a162082.sp259n1.txt",
        "au2019uc.sp259n1.txt",
        "a000060.sp259n2.txt",
        "a489486.sp259n2.txt",
        "a162082.sp259n2.txt",
        "a492143.sp259n1.txt",
        "a442243.sp259n1.txt",
        "a000072.sp259n2.txt",
        "a326291.sp259n1.txt",
        "a153842.sp259n2.txt",
        "a137084.sp259n1.txt",
    ],
    "sp260": [
        "a489486.sp260.txt",
        "a243025.sp260.txt",
        "a000173.sp260.txt",
    ],
    "sp261": [
        "au2018bp.sp261.txt",
        "au2019sh6.sp261.txt",
        "a349063.sp261.txt",
        "au2019yh2.sp261.txt",
    ],
    "sp262": [
        "a137924.sp262.txt",
        "au2011ct4.sp262.txt",
        "a250577.sp262.txt",
        "au2017ae5.sp262.txt",
        "a416591.sp262.txt",
        "a137120.sp262.txt",
        "au2015ox78.sp262.txt",
    ],
    "sp263": [
        "a163373.sp263n2.txt",
        "a010115.sp263n1.txt",
        "a000128.sp263n1.txt",
        "a437316.sp263n1.txt",
        "a011405.sp263n1.txt",
        "a000324.sp263n1.txt",
        "a162004.sp263n1.txt",
        "a034613.sp263n2.txt",
        "au1996ae2.sp263n2.txt",
        "a000810.sp263n2.txt",
        "a000088.sp263n1.txt",
        "au2017ae5.sp263n2.txt",
        "a089830.sp263n2.txt",
        "a002616.sp263n1.txt",
        "a329340.sp263n2.txt",
        "a512245.sp263n1.txt",
        "a001857.sp263n1.txt",
        "a203015.sp263n1.txt",
    ],
    "sp264": [
        "a531060.sp264n1.txt",
        "au2017bm123.sp264n1.txt",
        "au2000bo28.sp264n1.txt",
        "a008566.sp264n2.txt",
        "a052768.sp264n2.txt",
        "a065690.sp264n1.txt",
        "au2020dx.sp264n2.txt",
        "a154993.sp264n1.txt",
        "au2015bk509.sp264n2.txt",
    ],
    "sp265": [
        "a169675.sp265.txt",
        "a138911.sp265.txt",
        "a000050.sp265.txt",
        "a000058.sp265.txt",
        "a175189.sp265.txt",
        "a008014.sp265.txt",
    ],
    "sp266": [
        "a001685.sp266.txt",
        "a159608.sp266.txt",
    ],
    "sp267": [
        "a382503.sp267.txt",
        "a163348.sp267.txt",
        "a420302.sp267.txt",
        "a498066.sp267.txt",
        "a477885.sp267.txt",
        "a136874.sp267.txt",
    ],
    "sp268": [
        "au2020ME1.sp268.txt",
        "a437844.sp268.txt",
    ],
    "sp269": [
        "a380128.sp269n1.txt",
        "au2014lw21.sp269n1.txt",
        "au2004lu3.sp269n2.txt",
    ],
    "sp270": [
        "a005645.sp270n2.txt",
        "au2020ps.sp270n1.txt",
        "a380128.sp270n1.txt",
        "au2020ps.sp270n2.txt",
        "au2019an5.sp270n1.txt",
        "a137108.sp270n1.txt",
        "a411165.sp270n2.txt",
        "au2020qw.sp270n2.txt",
    ],
    "sp271": [
        "a271073.sp271.txt",
        "a247745.sp271.txt",
        "a366616.sp271.txt",
    ],
    "sp272": [
        "a247745.sp272.txt",
        "au2013ux14.sp272.txt",
        "au2016ed85.sp272.txt",
        "au2016nv38.sp272.txt",
        "a162173.sp272.txt",
        "a053435.sp272.txt",
        "au2020rc.sp272.txt",
        "a004953.sp272.txt",
    ],
    "sp273": [
        "au2016VM4.sp273.txt",
        "a015817.sp273.txt",
        "a085119.sp273.txt",
        "a162173.sp273.txt",
        "a069230.sp273.txt",
        "au2020RF.sp273.txt",
        "au2020RB6.sp273.txt",
    ],
    "sp274": [
        "au2020RJ3.sp274n1.txt",
        "a271073.sp274n2.txt",
        "au2020SN.sp274n1.txt",
        "au2016VM4.sp274n1.txt",
        "a005645.sp274n2.txt",
        "a474443.sp274n1.txt",
        "au2003QA108.sp274n2.txt",
        "au2020SS4.sp274n2.txt",
        "a302830.sp274n1.txt",
        "a053435.sp274n1.txt",
        "a165213.sp274n2.txt",
        "a166044.sp274n1.txt",
        "a053435.sp274n2.txt",
        "au2020SY4.sp274n2.txt",
        "a329555.sp274n1.txt",
        "au2020PM7.sp274n2.txt",
    ],
    "sp275": [
        "a159571.sp275.txt",
        "au2013py6.sp275.txt",
        "au2020pd1.sp275.txt",
        "au2004rf84.sp275.txt",
        "a004055.sp275.txt",
        "a474179.sp275.txt",
    ],
    "sp276": [
        "a306590.sp276n2.txt",
        "a465749.sp276n1.txt",
        "a168552.sp276n1.txt",
        "a306590.sp276n1.txt",
        "a066272.sp276n1.txt",
        "au2002TS67.sp276n1.txt",
        "au2020ST1.sp276n1.txt",
        "au2020TY1.sp276n1.txt",
        "au2002XZ4.sp276n2.txt",
    ],
    "sp277": [
        "a065717.sp277.txt",
        "a152978.sp277.txt",
        "a000426.sp277.txt",
    ],
    "sp278": [
        "a000196.sp278n2.txt",
        "a186822.sp278n1.txt",
        "a508907.sp278n2.txt",
        "a380359.sp278n2.txt",
        "au2013WJ.sp278n2.txt",
        "au2020WK3.sp278n1.txt",
        "au2016YM.sp278n1.txt",
        "au2020XH1.sp278n2.txt",
        "a153201.sp278n1.txt",
        "a000247.sp278n2.txt",
        "au2014WG365.sp278n1.txt",
        "au2003YJ.sp278n1.txt",
        "a154330.sp278n1.txt",
    ],
    "sp279": [
        "au2002GZ8.sp279n3.txt",
        "a438902.sp279n2.txt",
        "au2020RO6.sp279n2.txt",
        "a259776.sp279n2.txt",
        "au2015NU13.sp279n3.txt",
        "au2020WU5.sp279n2.txt",
        "a189040.sp279n1.txt",
        "a000566.sp279n2.txt",
        "a099942.sp279n3.txt",
        "a163902.sp279n3.txt",
        "a099942.sp279n4.txt",
        "a000203.sp279n3.txt",
        "a090147.sp279n3.txt",
        "au2016CO247.sp279n1.txt",
        "a000168.sp279n2.txt",
        "au2020WU5.sp279n3.txt",
        "au2020WL3.sp279n4.txt",
        "a171620.sp279n4.txt",
        "a438902.sp279n3.txt",
        "a313276.sp279n4.txt",
        "a230269.sp279n1.txt",
        "a156720.sp279n3.txt",
        "au2003AF23.sp279n1.txt",
        "au2017BQ93.sp279n2.txt",
    ],
    "sp280": [
        "au2020TB12.sp280n1.txt",
        "au2021CG.sp280n1.txt",
        "au1999RM45.sp280n2.txt",
        "au2016CL136.sp280n1.txt",
        "au2020XU6.sp280n2.txt",
        "a506459.sp280n2.txt",
        "a505657.sp280n2.txt",
        "au2019YP5.sp280n1.txt",
        "au2020WM3.sp280n2.txt",
        "a174050.sp280n2.txt",
        "au2015AS45.sp280n1.txt",
        "au2018PP10.sp280n1.txt",
        "a311554.sp280n2.txt",
        "a005879.sp280n2.txt",
    ],
    "sp281": [
        "a278462.sp281.txt",
        "au2021ec5.sp281.txt",
        "a161995.sp281.txt",
        "a068348.sp281.txt",
        "au2015kj19.sp281.txt",
        "a000164.sp281.txt",
        "a302871.sp281.txt",
        "au2014gd50.sp281.txt",
        "au2020yj5.sp281.txt",
        "a387505.sp281.txt",
        "a159857.sp281.txt",
    ],
    "sp282": [
        "a089958.sp282n2.txt",
        "a374267.sp282n1.txt",
        "a067367.sp282n2.txt",
        "au2021lg2.sp282n2.txt",
        "a366746.sp282n1.txt",
        "a000412.sp282n2.txt",
        "a480856.sp282n1.txt",
    ],
    "sp283": [
        "a159857.sp283n3.txt",
        "au2021jt5.sp283n1.txt",
        "a445267.sp283n2.txt",
        "a483471.sp283n2.txt",
        "au2006mw1.sp283n1.txt",
        "au2013xe4.sp283n2.txt",
        "a000455.sp283n2.txt",
        "a302871.sp283n3.txt",
    ],
    "sp284": [
        "au2021jq24.sp284.txt",
        "au2019nc1.sp284.txt",
        "a285571.sp284.txt",
        "au2021ch2.sp284.txt",
        "au2019ud4.sp284.txt",
    ],
    "sp285": [
        "au2021po41.sp285n2.txt",
        "a143649.sp285n1.txt",
        "a333901.sp285n2.txt",
        "a000365.sp285n2.txt",
        "au2021pr2.sp285n1.txt",
        "au2021qb3.sp285n1.txt",
        "au2001ec.sp285n1.txt",
        "au2021py40.sp285n1.txt",
        "au2008cd6.sp285n1.txt",
        "au2021jq24.sp285n1.txt",
        "a285571.sp285n2.txt",
        "a068063.sp285n1.txt",
        "a380163.sp285n1.txt",
        "au2021ln15.sp285n2.txt",
        "au2021re.sp285n2.txt",
        "a000117.sp285n1.txt",
    ],
    "sp286": [
        "a318160.sp286.txt",
    ],
    "sp287": [
        "au2021UU1.sp287.txt",
        "a100756.sp287.txt",
        "a138205.sp287.txt",
        "a005143.sp287.txt",
        "au2021US1.sp287.txt",
        "au2021LQ24.sp287.txt",
    ],
    "sp288": [
        "a162149.sp288.txt",
        "a012711.sp288.txt",
        "a523801.sp288.txt",
        "a267337.sp288.txt",
        "au2021XA6.sp288.txt",
        "a137805.sp288.txt",
    ],
    "sp289": [
        "a153591.sp289.txt",
        "a162149.sp289.txt",
        "a458122.sp289.txt",
        "a523801.sp289.txt",
        "a003103.sp289.txt",
        "au2019VG5.sp289.txt",
        "a348314.sp289.txt",
        "a012711.sp289.txt",
        "au2022AB.sp289.txt",
        "au2013YD48.sp289.txt",
        "a200473.sp289.txt",
        "au2017XC62.sp289.txt",
    ],
    "sp290": [
        "a506459.sp290n1.txt",
        "a001015.sp290n2.txt",
        "a001627.sp290n1.txt",
        "a001036.sp290n1.txt",
        "a190451.sp290n1.txt",
        "a000733.sp290n2.txt",
        "a170891.sp290n1.txt",
        "au2002LX.sp290n1.txt",
        "a138971.sp290n1.txt",
        "au2018CW13.sp290n1.txt",
        "a412983.sp290n1.txt",
        "au2010WQ7.sp290n1.txt",
        "a000209.sp290n2.txt",
        "a005330.sp290n1.txt",
        "a000747.sp290n2.txt",
        "a068350.sp290n1.txt",
        "a249886.sp290n1.txt",
    ],
    "sp291": [
        "a004660.sp291.txt",
        "au2022BX1.sp291.txt",
        "au2021VR27.sp291.txt",
        "a022099.sp291.txt",
        "a350751.sp291.txt",
        "au2021QO2.sp291.txt",
    ],
}
