"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OriginRequestQueryStringBehavior = exports.OriginRequestHeaderBehavior = exports.OriginRequestCookieBehavior = exports.OriginRequestPolicy = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const cloudfront_generated_1 = require("./cloudfront.generated");
/**
 * A Origin Request Policy configuration.
 *
 * @stability stable
 * @resource AWS::CloudFront::OriginRequestPolicy
 */
class OriginRequestPolicy extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _e, _f, _g, _h;
        super(scope, id, {
            physicalName: props.originRequestPolicyName,
        });
        const originRequestPolicyName = (_e = props.originRequestPolicyName) !== null && _e !== void 0 ? _e : core_1.Names.uniqueId(this);
        if (!core_1.Token.isUnresolved(originRequestPolicyName) && !originRequestPolicyName.match(/^[\w-]+$/i)) {
            throw new Error(`'originRequestPolicyName' can only include '-', '_', and alphanumeric characters, got: '${props.originRequestPolicyName}'`);
        }
        const cookies = (_f = props.cookieBehavior) !== null && _f !== void 0 ? _f : OriginRequestCookieBehavior.none();
        const headers = (_g = props.headerBehavior) !== null && _g !== void 0 ? _g : OriginRequestHeaderBehavior.none();
        const queryStrings = (_h = props.queryStringBehavior) !== null && _h !== void 0 ? _h : OriginRequestQueryStringBehavior.none();
        const resource = new cloudfront_generated_1.CfnOriginRequestPolicy(this, 'Resource', {
            originRequestPolicyConfig: {
                name: originRequestPolicyName,
                comment: props.comment,
                cookiesConfig: {
                    cookieBehavior: cookies.behavior,
                    cookies: cookies.cookies,
                },
                headersConfig: {
                    headerBehavior: headers.behavior,
                    headers: headers.headers,
                },
                queryStringsConfig: {
                    queryStringBehavior: queryStrings.behavior,
                    queryStrings: queryStrings.queryStrings,
                },
            },
        });
        this.originRequestPolicyId = resource.ref;
    }
    /**
     * Imports a Origin Request Policy from its id.
     *
     * @stability stable
     */
    static fromOriginRequestPolicyId(scope, id, originRequestPolicyId) {
        return new class extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.originRequestPolicyId = originRequestPolicyId;
            }
        }(scope, id);
    }
    /** Use an existing managed origin request policy. */
    static fromManagedOriginRequestPolicy(managedOriginRequestPolicyId) {
        return new class {
            constructor() {
                this.originRequestPolicyId = managedOriginRequestPolicyId;
            }
        }();
    }
}
exports.OriginRequestPolicy = OriginRequestPolicy;
_a = JSII_RTTI_SYMBOL_1;
OriginRequestPolicy[_a] = { fqn: "@aws-cdk/aws-cloudfront.OriginRequestPolicy", version: "1.113.0" };
/**
 * This policy includes only the User-Agent and Referer headers.
 *
 * It doesn’t include any query strings or cookies.
 *
 * @stability stable
 */
OriginRequestPolicy.USER_AGENT_REFERER_HEADERS = OriginRequestPolicy.fromManagedOriginRequestPolicy('acba4595-bd28-49b8-b9fe-13317c0390fa');
/**
 * This policy includes the header that enables cross-origin resource sharing (CORS) requests when the origin is a custom origin.
 *
 * @stability stable
 */
OriginRequestPolicy.CORS_CUSTOM_ORIGIN = OriginRequestPolicy.fromManagedOriginRequestPolicy('59781a5b-3903-41f3-afcb-af62929ccde1');
/**
 * This policy includes the headers that enable cross-origin resource sharing (CORS) requests when the origin is an Amazon S3 bucket.
 *
 * @stability stable
 */
OriginRequestPolicy.CORS_S3_ORIGIN = OriginRequestPolicy.fromManagedOriginRequestPolicy('88a5eaf4-2fd4-4709-b370-b4c650ea3fcf');
/**
 * This policy includes all values (query strings, headers, and cookies) in the viewer request.
 *
 * @stability stable
 */
OriginRequestPolicy.ALL_VIEWER = OriginRequestPolicy.fromManagedOriginRequestPolicy('216adef6-5c7f-47e4-b989-5492eafa07d3');
/**
 * This policy is designed for use with an origin that is an AWS Elemental MediaTailor endpoint.
 *
 * @stability stable
 */
OriginRequestPolicy.ELEMENTAL_MEDIA_TAILOR = OriginRequestPolicy.fromManagedOriginRequestPolicy('775133bc-15f2-49f9-abea-afb2e0bf67d2');
/**
 * Determines whether any cookies in viewer requests (and if so, which cookies) are included in requests that CloudFront sends to the origin.
 *
 * @stability stable
 */
class OriginRequestCookieBehavior {
    constructor(behavior, cookies) {
        this.behavior = behavior;
        this.cookies = cookies;
    }
    /**
     * Cookies in viewer requests are not included in requests that CloudFront sends to the origin.
     *
     * Any cookies that are listed in a CachePolicy are still included in origin requests.
     *
     * @stability stable
     */
    static none() { return new OriginRequestCookieBehavior('none'); }
    /**
     * All cookies in viewer requests are included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static all() { return new OriginRequestCookieBehavior('all'); }
    /**
     * Only the provided `cookies` are included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static allowList(...cookies) {
        if (cookies.length === 0) {
            throw new Error('At least one cookie to allow must be provided');
        }
        return new OriginRequestCookieBehavior('whitelist', cookies);
    }
}
exports.OriginRequestCookieBehavior = OriginRequestCookieBehavior;
_b = JSII_RTTI_SYMBOL_1;
OriginRequestCookieBehavior[_b] = { fqn: "@aws-cdk/aws-cloudfront.OriginRequestCookieBehavior", version: "1.113.0" };
/**
 * Determines whether any HTTP headers (and if so, which headers) are included in requests that CloudFront sends to the origin.
 *
 * @stability stable
 */
class OriginRequestHeaderBehavior {
    constructor(behavior, headers) {
        this.behavior = behavior;
        this.headers = headers;
    }
    /**
     * HTTP headers are not included in requests that CloudFront sends to the origin.
     *
     * Any headers that are listed in a CachePolicy are still included in origin requests.
     *
     * @stability stable
     */
    static none() { return new OriginRequestHeaderBehavior('none'); }
    /**
     * All HTTP headers in viewer requests are included in requests that CloudFront sends to the origin.
     *
     * Additionally, any additional CloudFront headers provided are included; the additional headers are added by CloudFront.
     *
     * @see https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-cloudfront-headers.html
     * @stability stable
     */
    static all(...cloudfrontHeaders) {
        if (cloudfrontHeaders.length > 0) {
            if (!cloudfrontHeaders.every(header => header.startsWith('CloudFront-'))) {
                throw new Error('additional CloudFront headers passed to `OriginRequestHeaderBehavior.all()` must begin with \'CloudFront-\'');
            }
            return new OriginRequestHeaderBehavior('allViewerAndWhitelistCloudFront', cloudfrontHeaders);
        }
        else {
            return new OriginRequestHeaderBehavior('allViewer');
        }
    }
    /**
     * Listed headers are included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static allowList(...headers) {
        if (headers.length === 0) {
            throw new Error('At least one header to allow must be provided');
        }
        if (headers.map(header => header.toLowerCase()).some(header => ['authorization', 'accept-encoding'].includes(header))) {
            throw new Error('you cannot pass `Authorization` or `Accept-Encoding` as header values; use a CachePolicy to forward these headers instead');
        }
        return new OriginRequestHeaderBehavior('whitelist', headers);
    }
}
exports.OriginRequestHeaderBehavior = OriginRequestHeaderBehavior;
_c = JSII_RTTI_SYMBOL_1;
OriginRequestHeaderBehavior[_c] = { fqn: "@aws-cdk/aws-cloudfront.OriginRequestHeaderBehavior", version: "1.113.0" };
/**
 * Determines whether any URL query strings in viewer requests (and if so, which query strings) are included in requests that CloudFront sends to the origin.
 *
 * @stability stable
 */
class OriginRequestQueryStringBehavior {
    constructor(behavior, queryStrings) {
        this.behavior = behavior;
        this.queryStrings = queryStrings;
    }
    /**
     * Query strings in viewer requests are not included in requests that CloudFront sends to the origin.
     *
     * Any query strings that are listed in a CachePolicy are still included in origin requests.
     *
     * @stability stable
     */
    static none() { return new OriginRequestQueryStringBehavior('none'); }
    /**
     * All query strings in viewer requests are included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static all() { return new OriginRequestQueryStringBehavior('all'); }
    /**
     * Only the provided `queryStrings` are included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static allowList(...queryStrings) {
        if (queryStrings.length === 0) {
            throw new Error('At least one query string to allow must be provided');
        }
        return new OriginRequestQueryStringBehavior('whitelist', queryStrings);
    }
}
exports.OriginRequestQueryStringBehavior = OriginRequestQueryStringBehavior;
_d = JSII_RTTI_SYMBOL_1;
OriginRequestQueryStringBehavior[_d] = { fqn: "@aws-cdk/aws-cloudfront.OriginRequestQueryStringBehavior", version: "1.113.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoib3JpZ2luLXJlcXVlc3QtcG9saWN5LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsib3JpZ2luLXJlcXVlc3QtcG9saWN5LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsd0NBQXVEO0FBRXZELGlFQUFnRTs7Ozs7OztBQTJCaEUsTUFBYSxtQkFBb0IsU0FBUSxlQUFROzs7O0lBNkIvQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLFFBQWtDLEVBQUU7O1FBQzVFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ2YsWUFBWSxFQUFFLEtBQUssQ0FBQyx1QkFBdUI7U0FDNUMsQ0FBQyxDQUFDO1FBRUgsTUFBTSx1QkFBdUIsU0FBRyxLQUFLLENBQUMsdUJBQXVCLG1DQUFJLFlBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDdEYsSUFBSSxDQUFDLFlBQUssQ0FBQyxZQUFZLENBQUMsdUJBQXVCLENBQUMsSUFBSSxDQUFDLHVCQUF1QixDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsRUFBRTtZQUMvRixNQUFNLElBQUksS0FBSyxDQUFDLDJGQUEyRixLQUFLLENBQUMsdUJBQXVCLEdBQUcsQ0FBQyxDQUFDO1NBQzlJO1FBRUQsTUFBTSxPQUFPLFNBQUcsS0FBSyxDQUFDLGNBQWMsbUNBQUksMkJBQTJCLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDM0UsTUFBTSxPQUFPLFNBQUcsS0FBSyxDQUFDLGNBQWMsbUNBQUksMkJBQTJCLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDM0UsTUFBTSxZQUFZLFNBQUcsS0FBSyxDQUFDLG1CQUFtQixtQ0FBSSxnQ0FBZ0MsQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUUxRixNQUFNLFFBQVEsR0FBRyxJQUFJLDZDQUFzQixDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDNUQseUJBQXlCLEVBQUU7Z0JBQ3pCLElBQUksRUFBRSx1QkFBdUI7Z0JBQzdCLE9BQU8sRUFBRSxLQUFLLENBQUMsT0FBTztnQkFDdEIsYUFBYSxFQUFFO29CQUNiLGNBQWMsRUFBRSxPQUFPLENBQUMsUUFBUTtvQkFDaEMsT0FBTyxFQUFFLE9BQU8sQ0FBQyxPQUFPO2lCQUN6QjtnQkFDRCxhQUFhLEVBQUU7b0JBQ2IsY0FBYyxFQUFFLE9BQU8sQ0FBQyxRQUFRO29CQUNoQyxPQUFPLEVBQUUsT0FBTyxDQUFDLE9BQU87aUJBQ3pCO2dCQUNELGtCQUFrQixFQUFFO29CQUNsQixtQkFBbUIsRUFBRSxZQUFZLENBQUMsUUFBUTtvQkFDMUMsWUFBWSxFQUFFLFlBQVksQ0FBQyxZQUFZO2lCQUN4QzthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLHFCQUFxQixHQUFHLFFBQVEsQ0FBQyxHQUFHLENBQUM7SUFDNUMsQ0FBQzs7Ozs7O0lBakRNLE1BQU0sQ0FBQyx5QkFBeUIsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxxQkFBNkI7UUFDakcsT0FBTyxJQUFJLEtBQU0sU0FBUSxlQUFRO1lBQXRCOztnQkFDTywwQkFBcUIsR0FBRyxxQkFBcUIsQ0FBQztZQUNoRSxDQUFDO1NBQUEsQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDZixDQUFDO0lBRUQscURBQXFEO0lBQzdDLE1BQU0sQ0FBQyw4QkFBOEIsQ0FBQyw0QkFBb0M7UUFDaEYsT0FBTyxJQUFJO1lBQUE7Z0JBQ08sMEJBQXFCLEdBQUcsNEJBQTRCLENBQUM7WUFDdkUsQ0FBQztTQUFBLEVBQUUsQ0FBQztJQUNOLENBQUM7O0FBekJILGtEQWdFQzs7Ozs7Ozs7OztBQTdEd0IsOENBQTBCLEdBQUcsbUJBQW1CLENBQUMsOEJBQThCLENBQUMsc0NBQXNDLENBQUMsQ0FBQzs7Ozs7O0FBRXhILHNDQUFrQixHQUFHLG1CQUFtQixDQUFDLDhCQUE4QixDQUFDLHNDQUFzQyxDQUFDLENBQUM7Ozs7OztBQUVoSCxrQ0FBYyxHQUFHLG1CQUFtQixDQUFDLDhCQUE4QixDQUFDLHNDQUFzQyxDQUFDLENBQUM7Ozs7OztBQUU1Ryw4QkFBVSxHQUFHLG1CQUFtQixDQUFDLDhCQUE4QixDQUFDLHNDQUFzQyxDQUFDLENBQUM7Ozs7OztBQUV4RywwQ0FBc0IsR0FBRyxtQkFBbUIsQ0FBQyw4QkFBOEIsQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDOzs7Ozs7QUF3RDdJLE1BQWEsMkJBQTJCO0lBb0J0QyxZQUFvQixRQUFnQixFQUFFLE9BQWtCO1FBQ3RELElBQUksQ0FBQyxRQUFRLEdBQUcsUUFBUSxDQUFDO1FBQ3pCLElBQUksQ0FBQyxPQUFPLEdBQUcsT0FBTyxDQUFDO0lBQ3pCLENBQUM7Ozs7Ozs7O0lBckJNLE1BQU0sQ0FBQyxJQUFJLEtBQUssT0FBTyxJQUFJLDJCQUEyQixDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQzs7Ozs7O0lBR2pFLE1BQU0sQ0FBQyxHQUFHLEtBQUssT0FBTyxJQUFJLDJCQUEyQixDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQzs7Ozs7O0lBRy9ELE1BQU0sQ0FBQyxTQUFTLENBQUMsR0FBRyxPQUFpQjtRQUMxQyxJQUFJLE9BQU8sQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ3hCLE1BQU0sSUFBSSxLQUFLLENBQUMsK0NBQStDLENBQUMsQ0FBQztTQUNsRTtRQUNELE9BQU8sSUFBSSwyQkFBMkIsQ0FBQyxXQUFXLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDL0QsQ0FBQzs7QUFiSCxrRUF3QkM7Ozs7Ozs7O0FBR0QsTUFBYSwyQkFBMkI7SUFnQ3RDLFlBQW9CLFFBQWdCLEVBQUUsT0FBa0I7UUFDdEQsSUFBSSxDQUFDLFFBQVEsR0FBRyxRQUFRLENBQUM7UUFDekIsSUFBSSxDQUFDLE9BQU8sR0FBRyxPQUFPLENBQUM7SUFDekIsQ0FBQzs7Ozs7Ozs7SUFqQ00sTUFBTSxDQUFDLElBQUksS0FBSyxPQUFPLElBQUksMkJBQTJCLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDOzs7Ozs7Ozs7SUFHakUsTUFBTSxDQUFDLEdBQUcsQ0FBQyxHQUFHLGlCQUEyQjtRQUM5QyxJQUFJLGlCQUFpQixDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDaEMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLENBQUMsRUFBRTtnQkFDeEUsTUFBTSxJQUFJLEtBQUssQ0FBQyw2R0FBNkcsQ0FBQyxDQUFDO2FBQ2hJO1lBQ0QsT0FBTyxJQUFJLDJCQUEyQixDQUFDLGlDQUFpQyxFQUFFLGlCQUFpQixDQUFDLENBQUM7U0FDOUY7YUFBTTtZQUNMLE9BQU8sSUFBSSwyQkFBMkIsQ0FBQyxXQUFXLENBQUMsQ0FBQztTQUNyRDtJQUNILENBQUM7Ozs7OztJQUdNLE1BQU0sQ0FBQyxTQUFTLENBQUMsR0FBRyxPQUFpQjtRQUMxQyxJQUFJLE9BQU8sQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ3hCLE1BQU0sSUFBSSxLQUFLLENBQUMsK0NBQStDLENBQUMsQ0FBQztTQUNsRTtRQUNELElBQUksT0FBTyxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLENBQUMsZUFBZSxFQUFFLGlCQUFpQixDQUFDLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxDQUFDLEVBQUU7WUFDckgsTUFBTSxJQUFJLEtBQUssQ0FBQywySEFBMkgsQ0FBQyxDQUFDO1NBQzlJO1FBQ0QsT0FBTyxJQUFJLDJCQUEyQixDQUFDLFdBQVcsRUFBRSxPQUFPLENBQUMsQ0FBQztJQUMvRCxDQUFDOztBQXpCSCxrRUFvQ0M7Ozs7Ozs7O0FBR0QsTUFBYSxnQ0FBZ0M7SUFvQjNDLFlBQW9CLFFBQWdCLEVBQUUsWUFBdUI7UUFDM0QsSUFBSSxDQUFDLFFBQVEsR0FBRyxRQUFRLENBQUM7UUFDekIsSUFBSSxDQUFDLFlBQVksR0FBRyxZQUFZLENBQUM7SUFDbkMsQ0FBQzs7Ozs7Ozs7SUFyQk0sTUFBTSxDQUFDLElBQUksS0FBSyxPQUFPLElBQUksZ0NBQWdDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDOzs7Ozs7SUFHdEUsTUFBTSxDQUFDLEdBQUcsS0FBSyxPQUFPLElBQUksZ0NBQWdDLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDOzs7Ozs7SUFHcEUsTUFBTSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFlBQXNCO1FBQy9DLElBQUksWUFBWSxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDN0IsTUFBTSxJQUFJLEtBQUssQ0FBQyxxREFBcUQsQ0FBQyxDQUFDO1NBQ3hFO1FBQ0QsT0FBTyxJQUFJLGdDQUFnQyxDQUFDLFdBQVcsRUFBRSxZQUFZLENBQUMsQ0FBQztJQUN6RSxDQUFDOztBQWJILDRFQXdCQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IE5hbWVzLCBSZXNvdXJjZSwgVG9rZW4gfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgQ2ZuT3JpZ2luUmVxdWVzdFBvbGljeSB9IGZyb20gJy4vY2xvdWRmcm9udC5nZW5lcmF0ZWQnO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgSU9yaWdpblJlcXVlc3RQb2xpY3kge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgb3JpZ2luUmVxdWVzdFBvbGljeUlkOiBzdHJpbmc7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgT3JpZ2luUmVxdWVzdFBvbGljeVByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IG9yaWdpblJlcXVlc3RQb2xpY3lOYW1lPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBjb21tZW50Pzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgY29va2llQmVoYXZpb3I/OiBPcmlnaW5SZXF1ZXN0Q29va2llQmVoYXZpb3I7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBoZWFkZXJCZWhhdmlvcj86IE9yaWdpblJlcXVlc3RIZWFkZXJCZWhhdmlvcjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHF1ZXJ5U3RyaW5nQmVoYXZpb3I/OiBPcmlnaW5SZXF1ZXN0UXVlcnlTdHJpbmdCZWhhdmlvcjtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgT3JpZ2luUmVxdWVzdFBvbGljeSBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSU9yaWdpblJlcXVlc3RQb2xpY3kge1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFVTRVJfQUdFTlRfUkVGRVJFUl9IRUFERVJTID0gT3JpZ2luUmVxdWVzdFBvbGljeS5mcm9tTWFuYWdlZE9yaWdpblJlcXVlc3RQb2xpY3koJ2FjYmE0NTk1LWJkMjgtNDliOC1iOWZlLTEzMzE3YzAzOTBmYScpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgQ09SU19DVVNUT01fT1JJR0lOID0gT3JpZ2luUmVxdWVzdFBvbGljeS5mcm9tTWFuYWdlZE9yaWdpblJlcXVlc3RQb2xpY3koJzU5NzgxYTViLTM5MDMtNDFmMy1hZmNiLWFmNjI5MjljY2RlMScpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IENPUlNfUzNfT1JJR0lOID0gT3JpZ2luUmVxdWVzdFBvbGljeS5mcm9tTWFuYWdlZE9yaWdpblJlcXVlc3RQb2xpY3koJzg4YTVlYWY0LTJmZDQtNDcwOS1iMzcwLWI0YzY1MGVhM2ZjZicpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBBTExfVklFV0VSID0gT3JpZ2luUmVxdWVzdFBvbGljeS5mcm9tTWFuYWdlZE9yaWdpblJlcXVlc3RQb2xpY3koJzIxNmFkZWY2LTVjN2YtNDdlNC1iOTg5LTU0OTJlYWZhMDdkMycpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgRUxFTUVOVEFMX01FRElBX1RBSUxPUiA9IE9yaWdpblJlcXVlc3RQb2xpY3kuZnJvbU1hbmFnZWRPcmlnaW5SZXF1ZXN0UG9saWN5KCc3NzUxMzNiYy0xNWYyLTQ5ZjktYWJlYS1hZmIyZTBiZjY3ZDInKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBmcm9tT3JpZ2luUmVxdWVzdFBvbGljeUlkKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIG9yaWdpblJlcXVlc3RQb2xpY3lJZDogc3RyaW5nKTogSU9yaWdpblJlcXVlc3RQb2xpY3kge1xuICAgIHJldHVybiBuZXcgY2xhc3MgZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElPcmlnaW5SZXF1ZXN0UG9saWN5IHtcbiAgICAgIHB1YmxpYyByZWFkb25seSBvcmlnaW5SZXF1ZXN0UG9saWN5SWQgPSBvcmlnaW5SZXF1ZXN0UG9saWN5SWQ7XG4gICAgfShzY29wZSwgaWQpO1xuICB9XG5cbiAgLyoqIFVzZSBhbiBleGlzdGluZyBtYW5hZ2VkIG9yaWdpbiByZXF1ZXN0IHBvbGljeS4gKi9cbiAgcHJpdmF0ZSBzdGF0aWMgZnJvbU1hbmFnZWRPcmlnaW5SZXF1ZXN0UG9saWN5KG1hbmFnZWRPcmlnaW5SZXF1ZXN0UG9saWN5SWQ6IHN0cmluZyk6IElPcmlnaW5SZXF1ZXN0UG9saWN5IHtcbiAgICByZXR1cm4gbmV3IGNsYXNzIGltcGxlbWVudHMgSU9yaWdpblJlcXVlc3RQb2xpY3kge1xuICAgICAgcHVibGljIHJlYWRvbmx5IG9yaWdpblJlcXVlc3RQb2xpY3lJZCA9IG1hbmFnZWRPcmlnaW5SZXF1ZXN0UG9saWN5SWQ7XG4gICAgfSgpO1xuICB9XG5cbiAgcHVibGljIHJlYWRvbmx5IG9yaWdpblJlcXVlc3RQb2xpY3lJZDogc3RyaW5nO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBPcmlnaW5SZXF1ZXN0UG9saWN5UHJvcHMgPSB7fSkge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwge1xuICAgICAgcGh5c2ljYWxOYW1lOiBwcm9wcy5vcmlnaW5SZXF1ZXN0UG9saWN5TmFtZSxcbiAgICB9KTtcblxuICAgIGNvbnN0IG9yaWdpblJlcXVlc3RQb2xpY3lOYW1lID0gcHJvcHMub3JpZ2luUmVxdWVzdFBvbGljeU5hbWUgPz8gTmFtZXMudW5pcXVlSWQodGhpcyk7XG4gICAgaWYgKCFUb2tlbi5pc1VucmVzb2x2ZWQob3JpZ2luUmVxdWVzdFBvbGljeU5hbWUpICYmICFvcmlnaW5SZXF1ZXN0UG9saWN5TmFtZS5tYXRjaCgvXltcXHctXSskL2kpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYCdvcmlnaW5SZXF1ZXN0UG9saWN5TmFtZScgY2FuIG9ubHkgaW5jbHVkZSAnLScsICdfJywgYW5kIGFscGhhbnVtZXJpYyBjaGFyYWN0ZXJzLCBnb3Q6ICcke3Byb3BzLm9yaWdpblJlcXVlc3RQb2xpY3lOYW1lfSdgKTtcbiAgICB9XG5cbiAgICBjb25zdCBjb29raWVzID0gcHJvcHMuY29va2llQmVoYXZpb3IgPz8gT3JpZ2luUmVxdWVzdENvb2tpZUJlaGF2aW9yLm5vbmUoKTtcbiAgICBjb25zdCBoZWFkZXJzID0gcHJvcHMuaGVhZGVyQmVoYXZpb3IgPz8gT3JpZ2luUmVxdWVzdEhlYWRlckJlaGF2aW9yLm5vbmUoKTtcbiAgICBjb25zdCBxdWVyeVN0cmluZ3MgPSBwcm9wcy5xdWVyeVN0cmluZ0JlaGF2aW9yID8/IE9yaWdpblJlcXVlc3RRdWVyeVN0cmluZ0JlaGF2aW9yLm5vbmUoKTtcblxuICAgIGNvbnN0IHJlc291cmNlID0gbmV3IENmbk9yaWdpblJlcXVlc3RQb2xpY3kodGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgb3JpZ2luUmVxdWVzdFBvbGljeUNvbmZpZzoge1xuICAgICAgICBuYW1lOiBvcmlnaW5SZXF1ZXN0UG9saWN5TmFtZSxcbiAgICAgICAgY29tbWVudDogcHJvcHMuY29tbWVudCxcbiAgICAgICAgY29va2llc0NvbmZpZzoge1xuICAgICAgICAgIGNvb2tpZUJlaGF2aW9yOiBjb29raWVzLmJlaGF2aW9yLFxuICAgICAgICAgIGNvb2tpZXM6IGNvb2tpZXMuY29va2llcyxcbiAgICAgICAgfSxcbiAgICAgICAgaGVhZGVyc0NvbmZpZzoge1xuICAgICAgICAgIGhlYWRlckJlaGF2aW9yOiBoZWFkZXJzLmJlaGF2aW9yLFxuICAgICAgICAgIGhlYWRlcnM6IGhlYWRlcnMuaGVhZGVycyxcbiAgICAgICAgfSxcbiAgICAgICAgcXVlcnlTdHJpbmdzQ29uZmlnOiB7XG4gICAgICAgICAgcXVlcnlTdHJpbmdCZWhhdmlvcjogcXVlcnlTdHJpbmdzLmJlaGF2aW9yLFxuICAgICAgICAgIHF1ZXJ5U3RyaW5nczogcXVlcnlTdHJpbmdzLnF1ZXJ5U3RyaW5ncyxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICB0aGlzLm9yaWdpblJlcXVlc3RQb2xpY3lJZCA9IHJlc291cmNlLnJlZjtcbiAgfVxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIE9yaWdpblJlcXVlc3RDb29raWVCZWhhdmlvciB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBub25lKCkgeyByZXR1cm4gbmV3IE9yaWdpblJlcXVlc3RDb29raWVCZWhhdmlvcignbm9uZScpOyB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgYWxsKCkgeyByZXR1cm4gbmV3IE9yaWdpblJlcXVlc3RDb29raWVCZWhhdmlvcignYWxsJyk7IH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBhbGxvd0xpc3QoLi4uY29va2llczogc3RyaW5nW10pIHtcbiAgICBpZiAoY29va2llcy5sZW5ndGggPT09IDApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQXQgbGVhc3Qgb25lIGNvb2tpZSB0byBhbGxvdyBtdXN0IGJlIHByb3ZpZGVkJyk7XG4gICAgfVxuICAgIHJldHVybiBuZXcgT3JpZ2luUmVxdWVzdENvb2tpZUJlaGF2aW9yKCd3aGl0ZWxpc3QnLCBjb29raWVzKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBiZWhhdmlvcjogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgY29va2llcz86IHN0cmluZ1tdO1xuXG4gIHByaXZhdGUgY29uc3RydWN0b3IoYmVoYXZpb3I6IHN0cmluZywgY29va2llcz86IHN0cmluZ1tdKSB7XG4gICAgdGhpcy5iZWhhdmlvciA9IGJlaGF2aW9yO1xuICAgIHRoaXMuY29va2llcyA9IGNvb2tpZXM7XG4gIH1cbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgT3JpZ2luUmVxdWVzdEhlYWRlckJlaGF2aW9yIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBub25lKCkgeyByZXR1cm4gbmV3IE9yaWdpblJlcXVlc3RIZWFkZXJCZWhhdmlvcignbm9uZScpOyB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGFsbCguLi5jbG91ZGZyb250SGVhZGVyczogc3RyaW5nW10pIHtcbiAgICBpZiAoY2xvdWRmcm9udEhlYWRlcnMubGVuZ3RoID4gMCkge1xuICAgICAgaWYgKCFjbG91ZGZyb250SGVhZGVycy5ldmVyeShoZWFkZXIgPT4gaGVhZGVyLnN0YXJ0c1dpdGgoJ0Nsb3VkRnJvbnQtJykpKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignYWRkaXRpb25hbCBDbG91ZEZyb250IGhlYWRlcnMgcGFzc2VkIHRvIGBPcmlnaW5SZXF1ZXN0SGVhZGVyQmVoYXZpb3IuYWxsKClgIG11c3QgYmVnaW4gd2l0aCBcXCdDbG91ZEZyb250LVxcJycpO1xuICAgICAgfVxuICAgICAgcmV0dXJuIG5ldyBPcmlnaW5SZXF1ZXN0SGVhZGVyQmVoYXZpb3IoJ2FsbFZpZXdlckFuZFdoaXRlbGlzdENsb3VkRnJvbnQnLCBjbG91ZGZyb250SGVhZGVycyk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHJldHVybiBuZXcgT3JpZ2luUmVxdWVzdEhlYWRlckJlaGF2aW9yKCdhbGxWaWV3ZXInKTtcbiAgICB9XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGFsbG93TGlzdCguLi5oZWFkZXJzOiBzdHJpbmdbXSkge1xuICAgIGlmIChoZWFkZXJzLmxlbmd0aCA9PT0gMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdBdCBsZWFzdCBvbmUgaGVhZGVyIHRvIGFsbG93IG11c3QgYmUgcHJvdmlkZWQnKTtcbiAgICB9XG4gICAgaWYgKGhlYWRlcnMubWFwKGhlYWRlciA9PiBoZWFkZXIudG9Mb3dlckNhc2UoKSkuc29tZShoZWFkZXIgPT4gWydhdXRob3JpemF0aW9uJywgJ2FjY2VwdC1lbmNvZGluZyddLmluY2x1ZGVzKGhlYWRlcikpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ3lvdSBjYW5ub3QgcGFzcyBgQXV0aG9yaXphdGlvbmAgb3IgYEFjY2VwdC1FbmNvZGluZ2AgYXMgaGVhZGVyIHZhbHVlczsgdXNlIGEgQ2FjaGVQb2xpY3kgdG8gZm9yd2FyZCB0aGVzZSBoZWFkZXJzIGluc3RlYWQnKTtcbiAgICB9XG4gICAgcmV0dXJuIG5ldyBPcmlnaW5SZXF1ZXN0SGVhZGVyQmVoYXZpb3IoJ3doaXRlbGlzdCcsIGhlYWRlcnMpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGJlaGF2aW9yOiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGhlYWRlcnM/OiBzdHJpbmdbXTtcblxuICBwcml2YXRlIGNvbnN0cnVjdG9yKGJlaGF2aW9yOiBzdHJpbmcsIGhlYWRlcnM/OiBzdHJpbmdbXSkge1xuICAgIHRoaXMuYmVoYXZpb3IgPSBiZWhhdmlvcjtcbiAgICB0aGlzLmhlYWRlcnMgPSBoZWFkZXJzO1xuICB9XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIE9yaWdpblJlcXVlc3RRdWVyeVN0cmluZ0JlaGF2aW9yIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIG5vbmUoKSB7IHJldHVybiBuZXcgT3JpZ2luUmVxdWVzdFF1ZXJ5U3RyaW5nQmVoYXZpb3IoJ25vbmUnKTsgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGFsbCgpIHsgcmV0dXJuIG5ldyBPcmlnaW5SZXF1ZXN0UXVlcnlTdHJpbmdCZWhhdmlvcignYWxsJyk7IH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGFsbG93TGlzdCguLi5xdWVyeVN0cmluZ3M6IHN0cmluZ1tdKSB7XG4gICAgaWYgKHF1ZXJ5U3RyaW5ncy5sZW5ndGggPT09IDApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQXQgbGVhc3Qgb25lIHF1ZXJ5IHN0cmluZyB0byBhbGxvdyBtdXN0IGJlIHByb3ZpZGVkJyk7XG4gICAgfVxuICAgIHJldHVybiBuZXcgT3JpZ2luUmVxdWVzdFF1ZXJ5U3RyaW5nQmVoYXZpb3IoJ3doaXRlbGlzdCcsIHF1ZXJ5U3RyaW5ncyk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBiZWhhdmlvcjogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgcXVlcnlTdHJpbmdzPzogc3RyaW5nW107XG5cbiAgcHJpdmF0ZSBjb25zdHJ1Y3RvcihiZWhhdmlvcjogc3RyaW5nLCBxdWVyeVN0cmluZ3M/OiBzdHJpbmdbXSkge1xuICAgIHRoaXMuYmVoYXZpb3IgPSBiZWhhdmlvcjtcbiAgICB0aGlzLnF1ZXJ5U3RyaW5ncyA9IHF1ZXJ5U3RyaW5ncztcbiAgfVxufVxuIl19