# Exploration

DPGEN2 allows developers to contribute exploration strategies. The exploration strategy defines how the configuration space is explored by molecular simulations in each DPGEN iteration. Notice that we are not restricted to molecular dynamics, any molecular simulation is, in priciple, allowed. For example, Monte Carlo, enhanced sampling, structure optimization, and so on.

An [exploration strategy](developer.md#the-exploration-strategy) takes the history of exploration as input, and gives back DPGEN the exploration tasks (we call it **task group**) and the rule to select configurations from the trajectories generated by the tasks (we call it **configuration selector**). 

One can contribute from three aspects: 

- [The stage scheduler](#stage-scheduler)
- [The exploration task groups](#exploration-task-groups)
- [Configuration selector](#configuration-selector)


## Stage scheduler 

The stage scheduler takes an exploration report passed from the exploration scheduler as input, and tells the exploration scheduler if the exploration in the stage is converged, if not, returns a group of exploration tasks and a configuration selector that are used in the next DPGEN iteration. 

Detailed explanation of the concepts are found [here](developer.md#the-exploration-strategy). 

All the stage schedulers are derived from the abstract base class `StageScheduler`. The only interface to be implemented is `StageScheduler.plan_next_iteration`. One may check the doc string for the explanation of the interface.
```python
class StageScheduler(ABC):
    """
    The scheduler for an exploration stage.
    """

    @abstractmethod
    def plan_next_iteration(
            self,
            hist_reports : List[ExplorationReport],
            report : ExplorationReport,
            confs : List[Path],
    ) -> Tuple[bool, ExplorationTaskGroup, ConfSelector] :
        """
        Make the plan for the next iteration of the stage.

        It checks the report of the current and all historical iterations of the stage, 
        and tells if the iterations are converged. 
        If not converged, it will plan the next ieration for the stage. 

        Parameters
        ----------
        hist_reports: List[ExplorationReport]
            The historical exploration report of the stage. If this is the first iteration of the stage, this list is empty.
        report : ExplorationReport
            The exploration report of this iteration.
        confs: List[Path]
            A list of configurations generated during the exploration. May be used to generate new configurations for the next iteration. 

        Returns
        -------
        converged: bool
            If the stage converged.
        task: ExplorationTaskGroup
            A `ExplorationTaskGroup` defining the exploration of the next iteration. Should be `None` if the stage is converged.
        conf_selector: ConfSelector
            The configuration selector for the next iteration. Should be `None` if the stage is converged.

        """
```

One may check more details on the [exploratin task group](#exploration-task-groups) and the [configuration selector]().


## Exploration task groups

DPGEN2 defines a python class `ExplorationTask` to manage all necessry files needed to run a exploration task. It can be used as the example provided in the doc string.
```python
class ExplorationTask():
    """Define the files needed by an exploration task. 

    Examples
    --------
    >>> # this example dumps all files needed by the task.
    >>> files = exploration_task.files()
    ... for file_name, file_content in files.items():
    ...     with open(file_name, 'w') as fp:
    ...         fp.write(file_content)    

    """	
```

A collection of the exploration tasks is called exploration task group. All tasks groups are derived from the base class `ExplorationTaskGroup`. The exploration task group can be viewd as a list of `ExplorationTask`s, one may get the list by using property `ExplorationTaskGroup.task_list`. One may add tasks, or ExplorationTaskGroup to the group by methods `ExplorationTaskGroup.add_task` and `ExplorationTaskGroup.add_group`, respectively.

```python
class ExplorationTaskGroup(Sequence):
    @property
    def task_list(self) -> List[ExplorationTask]:
        """Get the `list` of `ExplorationTask`""" 
        ...

    def add_task(self, task: ExplorationTask):
        """Add one task to the group."""
        ...

    def add_group(
            self,
            group : 'ExplorationTaskGroup',
    ):
        """Add another group to the group."""
        ...
```

[An example of generating a group of NPT MD simulations](https://github.com/deepmodeling/dpgen2/blob/master/dpgen2/exploration/task/npt_task_group.py) may illustrate how to implement the `ExplorationTaskGroup`s.


## Configuration selector

The configuration selectors are derived from the abstract base class `ConfSelector`

```python
class ConfSelector(ABC):
    """Select configurations from trajectory and model deviation files.
    """
    @abstractmethod
    def select (
            self,
            trajs : List[Path],
            model_devis : List[Path],
            traj_fmt : str = 'deepmd/npy',
            type_map : List[str] = None,
    ) -> Tuple[List[ Path ], ExplorationReport]:
```

The abstractmethod to implement is `ConfSelector.select`. `trajs` and `model_devis` are lists of files that recording the simulations trajectories and model deviations respectively. `traj_fmt` and `type_map` are parameters that may be needed for loading the trajectories by [`dpdata`](https://github.com/deepmodeling/dpdata).

The `ConfSelector.select` returns a Path, each of which can be treated as a `dpdata.MultiSystems`, and a [`ExplorationReport`](https://github.com/deepmodeling/dpgen2/blob/master/dpgen2/exploration/report/report.py). 

[An example of selecting configurations from LAMMPS trajectories](https://github.com/deepmodeling/dpgen2/blob/master/dpgen2/exploration/selector/conf_selector_frame.py) may illustrate how to implement the `ConfSelector`s.
