"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssertionsProvider = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
let SDK_METADATA = undefined;
/**
 * integ-tests can only depend on '@aws-cdk/core' so
 * this construct creates a lambda function provider using
 * only CfnResource
 */
class LambdaFunctionProvider extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.policies = [];
        const staging = new aws_cdk_lib_1.AssetStaging(this, 'Staging', {
            sourcePath: path.join(__dirname, 'lambda-handler.bundle'),
        });
        const stack = aws_cdk_lib_1.Stack.of(this);
        const asset = stack.synthesizer.addFileAsset({
            fileName: staging.relativeStagedPath(stack),
            sourceHash: staging.assetHash,
            packaging: aws_cdk_lib_1.FileAssetPackaging.ZIP_DIRECTORY,
        });
        const role = new aws_cdk_lib_1.CfnResource(this, 'Role', {
            type: 'AWS::IAM::Role',
            properties: {
                AssumeRolePolicyDocument: {
                    Version: '2012-10-17',
                    Statement: [{ Action: 'sts:AssumeRole', Effect: 'Allow', Principal: { Service: 'lambda.amazonaws.com' } }],
                },
                ManagedPolicyArns: [
                    { 'Fn::Sub': 'arn:${AWS::Partition}:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole' },
                ],
                Policies: aws_cdk_lib_1.Lazy.any({
                    produce: () => {
                        const policies = this.policies.length > 0 ? [
                            {
                                PolicyName: 'Inline',
                                PolicyDocument: {
                                    Version: '2012-10-17',
                                    Statement: this.policies,
                                },
                            },
                        ] : undefined;
                        return policies;
                    },
                }),
            },
        });
        const handler = new aws_cdk_lib_1.CfnResource(this, 'Handler', {
            type: 'AWS::Lambda::Function',
            properties: {
                Runtime: 'nodejs14.x',
                Code: {
                    S3Bucket: asset.bucketName,
                    S3Key: asset.objectKey,
                },
                Timeout: aws_cdk_lib_1.Duration.minutes(2).toSeconds(),
                Handler: props?.handler ?? 'index.handler',
                Role: role.getAtt('Arn'),
            },
        });
        this.serviceToken = aws_cdk_lib_1.Token.asString(handler.getAtt('Arn'));
        this.roleArn = role.getAtt('Arn');
    }
    addPolicies(policies) {
        this.policies.push(...policies);
    }
}
/**
 * Mimic the singletonfunction construct in '@aws-cdk/aws-lambda'
 */
class SingletonFunction extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.lambdaFunction = this.ensureFunction(props);
        this.serviceToken = this.lambdaFunction.serviceToken;
    }
    ensureFunction(props) {
        const constructName = 'SingletonFunction' + slugify(props.uuid);
        const existing = aws_cdk_lib_1.Stack.of(this).node.tryFindChild(constructName);
        if (existing) {
            return existing;
        }
        return new LambdaFunctionProvider(aws_cdk_lib_1.Stack.of(this), constructName, {
            handler: props.handler,
        });
    }
    /**
     * Add an IAM policy statement to the inline policy of the
     * lambdas function's role
     *
     * **Please note**: this is a direct IAM JSON policy blob, *not* a `iam.PolicyStatement`
     * object like you will see in the rest of the CDK.
     *
     *
     * singleton.addToRolePolicy({
     *   Effect: 'Allow',
     *   Action: 's3:GetObject',
     *   Resources: '*',
     * });
     */
    addToRolePolicy(statement) {
        this.lambdaFunction.addPolicies([statement]);
    }
    /**
     * Create a policy statement from a specific api call
     */
    addPolicyStatementFromSdkCall(service, api, resources) {
        if (SDK_METADATA === undefined) {
            // eslint-disable-next-line
            SDK_METADATA = require('./sdk-api-metadata.json');
        }
        const srv = service.toLowerCase();
        const iamService = (SDK_METADATA[srv] && SDK_METADATA[srv].prefix) || srv;
        const iamAction = api.charAt(0).toUpperCase() + api.slice(1);
        this.lambdaFunction.addPolicies([{
                Action: [`${iamService}:${iamAction}`],
                Effect: 'Allow',
                Resource: resources || ['*'],
            }]);
    }
}
/**
 * Represents an assertions provider. The creates a singletone
 * Lambda Function that will create a single function per stack
 * that serves as the custom resource provider for the various
 * assertion providers
 */
class AssertionsProvider extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_integ_tests_alpha_AssertionsProviderProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, AssertionsProvider);
            }
            throw error;
        }
        this.handler = new SingletonFunction(this, 'AssertionsProvider', {
            handler: props?.handler,
            uuid: props?.uuid ?? '1488541a-7b23-4664-81b6-9b4408076b81',
        });
        this.handlerRoleArn = this.handler.lambdaFunction.roleArn;
        this.serviceToken = this.handler.serviceToken;
    }
    /**
     * Encode an object so it can be passed
     * as custom resource parameters. Custom resources will convert
     * all input parameters to strings so we encode non-strings here
     * so we can then decode them correctly in the provider function
     */
    encode(obj) {
        if (!obj) {
            return obj;
        }
        return JSON.parse(JSON.stringify(obj), (_k, v) => {
            switch (v) {
                case true:
                    return 'TRUE:BOOLEAN';
                case false:
                    return 'FALSE:BOOLEAN';
                default:
                    return v;
            }
        });
    }
    /**
     * Create a policy statement from a specific api call
     */
    addPolicyStatementFromSdkCall(service, api, resources) {
        this.handler.addPolicyStatementFromSdkCall(service, api, resources);
    }
    /**
     * Add an IAM policy statement to the inline policy of the
     * lambdas function's role
     *
     * **Please note**: this is a direct IAM JSON policy blob, *not* a `iam.PolicyStatement`
     * object like you will see in the rest of the CDK.
     *
     *
     * @example
     * declare const provider: AssertionsProvider;
     * provider.addToRolePolicy({
     *   Effect: 'Allow',
     *   Action: ['s3:GetObject'],
     *   Resource: ['*'],
     * });
     */
    addToRolePolicy(statement) {
        this.handler.addToRolePolicy(statement);
    }
    /**
     * Grant a principal access to invoke the assertion provider
     * lambda function
     *
     * @param principalArn the ARN of the principal that should be given
     *  permission to invoke the assertion provider
     */
    grantInvoke(principalArn) {
        new aws_cdk_lib_1.CfnResource(this, 'Invoke', {
            type: 'AWS::Lambda::Permission',
            properties: {
                Action: 'lambda:InvokeFunction',
                FunctionName: this.serviceToken,
                Principal: principalArn,
            },
        });
    }
}
_a = JSII_RTTI_SYMBOL_1;
AssertionsProvider[_a] = { fqn: "@aws-cdk/integ-tests-alpha.AssertionsProvider", version: "2.72.0-alpha.0" };
exports.AssertionsProvider = AssertionsProvider;
function slugify(x) {
    return x.replace(/[^a-zA-Z0-9]/g, '');
}
//# sourceMappingURL=data:application/json;base64,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