"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssertionHandler = void 0;
/* eslint-disable no-console */
// eslint-disable-next-line import/no-extraneous-dependencies
const helpers_internal_1 = require("aws-cdk-lib/assertions/lib/helpers-internal");
const base_1 = require("./base");
class AssertionHandler extends base_1.CustomResourceHandler {
    async processEvent(request) {
        let actual = decodeCall(request.actual);
        const expected = decodeCall(request.expected);
        let result;
        const matcher = new MatchCreator(expected).getMatcher();
        console.log(`Testing equality between ${JSON.stringify(request.actual)} and ${JSON.stringify(request.expected)}`);
        const matchResult = matcher.test(actual);
        matchResult.finished();
        if (matchResult.hasFailed()) {
            result = {
                failed: true,
                assertion: JSON.stringify({
                    status: 'fail',
                    message: matchResult.renderMismatch(),
                }),
            };
            if (request.failDeployment) {
                throw new Error(result.assertion);
            }
        }
        else {
            result = {
                assertion: JSON.stringify({
                    status: 'success',
                }),
            };
        }
        return result;
    }
}
exports.AssertionHandler = AssertionHandler;
class MatchCreator {
    constructor(obj) {
        this.parsedObj = {
            matcher: obj,
        };
    }
    /**
     * Return a Matcher that can be tested against the actual results.
     * This will convert the encoded matchers into their corresponding
     * assertions matcher.
     *
     * For example:
     *
     * ExpectedResult.objectLike({
     *   Messages: [{
     *     Body: Match.objectLike({
     *       Elements: Match.arrayWith([{ Asdf: 3 }]),
     *       Payload: Match.serializedJson({ key: 'value' }),
     *     }),
     *   }],
     * });
     *
     * Will be encoded as:
     * {
     *   $ObjectLike: {
     *     Messages: [{
     *       Body: {
     *         $ObjectLike: {
     *           Elements: {
     *             $ArrayWith: [{ Asdf: 3 }],
     *           },
     *           Payload: {
     *             $SerializedJson: { key: 'value' }
     *           }
     *         },
     *       },
     *     }],
     *   },
     * }
     *
     * Which can then be parsed by this function. For each key (recursively)
     * the parser will check if the value has one of the encoded matchers as a key
     * and if so, it will set the value as the Matcher. So,
     *
     * {
     *   Body: {
     *     $ObjectLike: {
     *       Elements: {
     *         $ArrayWith: [{ Asdf: 3 }],
     *       },
     *       Payload: {
     *         $SerializedJson: { key: 'value' }
     *       }
     *     },
     *   },
     * }
     *
     * Will be converted to
     * {
     *   Body: Match.objectLike({
     *     Elements: Match.arrayWith([{ Asdf: 3 }]),
     *     Payload: Match.serializedJson({ key: 'value' }),
     *   }),
     * }
     */
    getMatcher() {
        try {
            const final = JSON.parse(JSON.stringify(this.parsedObj), function (_k, v) {
                const nested = Object.keys(v)[0];
                switch (nested) {
                    case '$ArrayWith':
                        return helpers_internal_1.Match.arrayWith(v[nested]);
                    case '$ObjectLike':
                        return helpers_internal_1.Match.objectLike(v[nested]);
                    case '$StringLike':
                        return helpers_internal_1.Match.stringLikeRegexp(v[nested]);
                    case '$SerializedJson':
                        return helpers_internal_1.Match.serializedJson(v[nested]);
                    default:
                        return v;
                }
            });
            if (helpers_internal_1.Matcher.isMatcher(final.matcher)) {
                return final.matcher;
            }
            return helpers_internal_1.Match.exact(final.matcher);
        }
        catch {
            return helpers_internal_1.Match.exact(this.parsedObj.matcher);
        }
    }
}
function decodeCall(call) {
    if (!call) {
        return undefined;
    }
    try {
        const parsed = JSON.parse(call);
        return parsed;
    }
    catch {
        return call;
    }
}
//# sourceMappingURL=data:application/json;base64,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