import os, sys, multiprocessing
from fractions import Fraction
from typing import Union
import numpy as np
from .kshell_exceptions import KshellDataStructureError
from .general_utilities import level_plot, level_density, gamma_strength_function_average

atomic_numbers = {
    "oxygen": 8, "fluorine": 9, "neon": 10, "sodium": 11, "magnesium": 12,
    "aluminium": 13, "silicon": 14, "phosphorus": 15, "sulfur": 16,
    "chlorine": 17, "argon": 18
}

atomic_numbers_reversed = {
    8: 'oxygen', 9: 'fluorine', 10: 'neon', 11: 'sodium', 12: 'magnesium',
    13: 'aluminium', 14: 'silicon', 15: 'phosphorus', 16: 'sulfur',
    17: 'chlorine', 18: 'argon'
}

def generate_states(
    start: int = 0,
    stop: int = 14,
    n_states: int = 100,
    parity: Union[str, int] = "both"
    ):
    """
    Generate correct string for input to `kshell_ui.py` when asked for
    which states to calculate. Copy the string generated by this
    function and paste it into `kshell_ui.py` when it prompts for
    states.

    Parameters
    ----------
    start : int
        The lowest spin value.

    stop : int
        The largest spin value.

    n_states : int
        The number of states per spin value.

    parity : Union[str, int]
        The parity of the states. Allowed values are: 1, -1, 'both',
        'positive', 'negative', 'pos', 'neg', '+', '-'.

    Examples
    --------
    ``` python
    >>> import kshell_utilities as ksutil
    >>> ksutil.generate_states(start=0, stop=3, n_states=100, parity="both")
    0+100, 0.5+100, 1+100, 1.5+100, 2+100, 2.5+100, 3+100, 0-100, 0.5-100, 1-100, 1.5-100, 2-100, 2.5-100, 3-100,
    ```
    """
    allowed_positive_parity_inputs = ["positive", "pos", "+", "1", "+1", 1, "both"]
    allowed_negative_parity_inputs = ["negative", "neg", "-", "-1", -1, "both"]
    
    def correct_syntax(lst):
        for elem in lst:
            print(elem, end=", ")
    
    if parity in allowed_positive_parity_inputs:
        positive = [f"{i:g}{'+'}{n_states}" for i in np.arange(start, stop+0.5, 0.5)]
        correct_syntax(positive)
    
    if parity in allowed_negative_parity_inputs:
        negative = [f"{i:g}{'-'}{n_states}" for i in np.arange(start, stop+0.5, 0.5)]
        correct_syntax(negative)

class ReadKshellOutput:
    """
    Read `KSHELL` data files and store the values as instance
    attributes.

    Attributes
    ----------
    Ex : np.ndarray
        Array of excitation energies zeroed at the ground state energy.

    BM1 : np.ndarray
        Array of [[E, B_decay_prob, E_gamma], ...].
        Reduced transition probabilities for M1.

    BE2 : np.ndarray
        Array of [[E, B_decay_prob, E_gamma], ...].
        Reduced transition probabilities for E2.

    levels : np.ndarray
        Array containing energy, spin, and parity for each excited
        state. [[E, 2*spin, parity], ...].

    transitions : np.ndarray
        OLD:
        Mx8 array containing [2*spin_final, parity_initial, Ex_final,
        2*spin_initial, parity_initial, Ex_initial, E_gamma, B(.., i->f)].
        NEW:
        [2*spin_initial, parity_initial, Ex_initial, 2*spin_final,
        parity_final, Ex_final, E_gamma, B(.., i->f), B(.., f<-i)]
    """
    def __init__(self, path: str, load_and_save_to_file: bool):
        """
        Parameters
        ----------
        path : string
            Path of `KSHELL` output file directory, or path to a
            specific `KSHELL` data file.

        load_and_save_to_file : bool
            Toggle saving data as `.npy` files on / off. If `overwrite`,
            saved `.npy` files are overwritten.
        """

        self.path = path
        self.load_and_save_to_file = load_and_save_to_file
        # Some attributes might not be altered, depending on the input file.
        self.fname_summary = None
        self.fname_ptn = None
        self.nucleus = None
        self.model_space = None
        self.proton_partition = None
        self.neutron_partition = None
        self.Ex = None
        self.BM1 = None
        self.BE2 = None
        self.levels = None
        self.transitions = None
        self.transitions_BM1 = None
        self.transitions_BE2 = None
        self.truncation = None
        # Debug.
        self.minus_one_spin_counts = np.array([0, 0])  # The number of skipped -1 spin states for [levels, transitions].

        if isinstance(self.load_and_save_to_file, str) and (self.load_and_save_to_file != "overwrite"):
            msg = "Allowed values for 'load_and_save_to_file' are: 'True', 'False', 'overwrite'."
            msg += f" Got '{self.load_and_save_to_file}'."
            raise ValueError(msg)

        if os.path.isdir(path):
            """
            If input 'path' is a directory containing KSHELL files,
            extract info from both summary and .ptn file.
            """
            for elem in os.listdir(path):
                if elem.startswith("summary"):
                    self.fname_summary = f"{path}/{elem}"
                    self._extract_info_from_summary_fname()
                    self._read_summary()

                elif elem.endswith(".ptn"):
                    self.fname_ptn = f"{path}/{elem}"
                    self._extract_info_from_ptn_fname()
                    self.read_ptn()

        else:
            """
            'path' is a single file, not a directory.
            """
            fname = path.split("/")[-1]

            if fname.startswith("summary"):
                self.fname_summary = path
                self._extract_info_from_summary_fname()
                self._read_summary()

            elif fname.endswith(".ptn"):
                self.fname_ptn = path
                self._extract_info_from_ptn_fname()
                self._read_ptn()

            else:
                msg = f"Handling for file {fname} is not implemented."
                raise KshellDataStructureError(msg)

    def _extract_info_from_ptn_fname(self):
        """
        Extract nucleus and model space name.
        """
        fname_split = self.fname_ptn.split("/")[-1]
        fname_split = fname_split.split("_")
        self.nucleus = fname_split[0]
        self.model_space = fname_split[1]

    def _read_ptn(self):
        """
        Read `KSHELL` partition file (.ptn) and extract proton
        partition, neutron partition, and particle-hole truncation data.
        Save as instance attributes.
        """

        line_number = 0
        line_number_inner = 0
        self.truncation = []

        with open(self.fname_ptn, "r") as infile:
            for line in infile:
                line_number += 1
                
                if line.startswith("# proton partition"):
                    for line_inner in infile:
                        """
                        Read until next '#'.
                        """
                        line_number_inner += 1
                        if line_inner.startswith("#"):
                            line = line_inner
                            break
                    
                    self.proton_partition = np.loadtxt(
                        fname = self.fname_ptn,
                        skiprows = line_number,
                        max_rows = line_number_inner
                    )
                    line_number += line_number_inner
                    line_number_inner = 0
                
                if line.startswith("# neutron partition"):
                    for line_inner in infile:
                        """
                        Read until next '#'.
                        """
                        line_number_inner += 1
                        if line_inner.startswith("#"):
                            line = line_inner
                            break
                    
                    self.neutron_partition = np.loadtxt(
                        fname = self.fname_ptn,
                        skiprows = line_number,
                        max_rows = line_number_inner
                    )
                    line_number += line_number_inner
                    line_number_inner = 0

                if line.startswith("# particle-hole truncation"):
                    for line_inner in infile:
                        """
                        Loop over all particle-hole truncation lines.
                        """
                        line_number += 1
                        line_inner_split = line_inner.split()

                        if (len(line_inner_split) < 2):
                            """
                            Condition will probably not get fulfilled.
                            Safety precaution due to indexing in this
                            loop.
                            """
                            break

                        if (line_inner_split[1]).startswith("["):
                            """
                            '[' indicates that 'line_inner' is still
                            containing truncation information.
                            """
                            for colon_index, elem in enumerate(line_inner_split):
                                """
                                Find the index of the colon ':' to
                                decide the orbit numbers and occupation
                                numbers.
                                """
                                if (elem == ":"): break

                            occupation = [int(occ) for occ in line_inner_split[colon_index + 1:]]   # [min, max].
                            orbit_numbers = "".join(line_inner_split[1:colon_index])
                            orbit_numbers = orbit_numbers.replace("[", "")
                            orbit_numbers = orbit_numbers.replace("]", "")
                            orbit_numbers = orbit_numbers.replace(" ", "")  # This can prob. be removed because of the earlier split.
                            orbit_numbers = orbit_numbers.split(",")
                            orbit_numbers = [int(orbit) for orbit in orbit_numbers]
                            
                            for orbit in orbit_numbers:
                                self.truncation.append((orbit, occupation))
                        
                        else:
                            """
                            Line does not contain '[' and thus does not
                            contain truncation information.
                            """
                            break

    def _extract_info_from_summary_fname(self):
        """
        Extract nucleus and model space name.
        """
        fname_split = self.fname_summary.split("/")[-1]  # Remove path.
        fname_split = fname_split.split("_")
        self.nucleus = fname_split[1]
        self.model_space = fname_split[2][:-4]  # Remove .txt and keep model space name.

    def _read_summary(self):
        """
        Read energy level data, transition probabilities and transition
        strengths from `KSHELL` output files.

        TODO: Change all the substring indexing to something more
        rigorous, like string.split and similar.

        Raises
        ------
        KshellDataStructureError
            If the `KSHELL` file has unexpected structure / syntax.
        """
        npy_path = "tmp"
        base_fname = self.path.split("/")[-1][:-4]

        try:
            os.mkdir(npy_path)
        except FileExistsError:
            pass
            
        levels_fname = f"{npy_path}/{base_fname}_levels.npy"
        transitions_fname = f"{npy_path}/{base_fname}_transitions.npy"
        transitions_BM1_fname = f"{npy_path}/{base_fname}_transitions_BM1.npy"
        transitions_BE2_fname = f"{npy_path}/{base_fname}_transitions_BE2.npy"
        Ex_fname = f"{npy_path}/{base_fname}_Ex.npy"
        BM1_fname = f"{npy_path}/{base_fname}_BM1.npy"
        BE2_fname = f"{npy_path}/{base_fname}_BE2.npy"
        debug_fname = f"{npy_path}/{base_fname}_debug.npy"

        fnames = [
            levels_fname, transitions_fname, Ex_fname, BM1_fname, BE2_fname,
            transitions_BM1_fname, transitions_BE2_fname, debug_fname
        ]

        if self.load_and_save_to_file != "overwrite":
            """
            Do not load files if overwrite parameter has been passed.
            """
            if all([os.path.isfile(fname) for fname in fnames]) and self.load_and_save_to_file:
                """
                If all files exist, load them. If any of the files do
                not exist, all will be generated.
                """
                self.Ex = np.load(file=Ex_fname, allow_pickle=True)
                self.BM1 = np.load(file=BM1_fname, allow_pickle=True)
                self.BE2 = np.load(file=BE2_fname, allow_pickle=True)
                self.levels = np.load(file=levels_fname, allow_pickle=True)
                self.transitions = np.load(file=transitions_fname, allow_pickle=True)
                self.transitions_BM1 = np.load(file=transitions_BM1_fname, allow_pickle=True)
                self.transitions_BE2 = np.load(file=transitions_BE2_fname, allow_pickle=True)
                self.debug = np.load(file=debug_fname, allow_pickle=True)
                msg = "Summary data loaded from .npy!"
                msg += " Use loadtxt parameter load_and_save_to_file = 'overwrite'"
                msg += " to re-read data from the summary file."
                print(msg)
                return

        def load_energy_levels(infile):
            """
            Example
            -------
            Energy levels

            N    J prty N_Jp    T     E(MeV)  Ex(MeV)  log-file

            1   5/2 +     1   3/2    -16.565    0.000  log_O19_sdpf-mu_m1p.txt 
            2   3/2 +     1   3/2    -15.977    0.588  log_O19_sdpf-mu_m1p.txt 
            3   1/2 +     1   3/2    -15.192    1.374  log_O19_sdpf-mu_m1p.txt 
            4   9/2 +     1   3/2    -13.650    2.915  log_O19_sdpf-mu_m1p.txt 
            5   7/2 +     1   3/2    -13.267    3.298  log_O19_sdpf-mu_m1p.txt 
            6   5/2 +     2   3/2    -13.074    3.491  log_O19_sdpf-mu_m1p.txt
            """
            for _ in range(3): infile.readline()
            for line in infile:
                try:
                    tmp = line.split()
                    
                    if tmp[1] == "-1":
                        """
                        -1 spin states in the KSHELL data file indicates
                        bad states which should not be included.
                        """
                        self.minus_one_spin_counts[0] += 1  # Debug.
                        continue
                    
                    self.Ex.append(float(tmp[6]))
                    parity = 1 if tmp[2] == "+" else -1
                    self.levels.append([float(tmp[5]), 2*float(Fraction(tmp[1])), parity])
                except IndexError:
                    """
                    End of energies.
                    """
                    break

        def load_transition_probabilities(infile, reduced_transition_prob_decay_list):
            """
            Parameters
            ----------
            infile:
                The KSHELL summary file.

            reduced_transition_prob_decay_list:
                List for storing B(M1) or B(E2) values.
            """
            for _ in range(2): infile.readline()
            for line in infile:
                try:
                    """
                    Example of possible lines in file:
                    J_i    Ex_i     J_f    Ex_f   dE        B(M1)->         B(M1)<- 
                    2+(11) 18.393 2+(10) 17.791 0.602 0.1(  0.0) 0.1( 0.0)
                    3/2+( 1) 0.072 5/2+( 1) 0.000 0.071 0.127( 0.07) 0.084( 0.05)
                    2+(10) 17.791 2+( 1) 5.172 12.619 0.006( 0.00) 0.006( 0.00)
                    3+( 8) 19.503 2+(11) 18.393 1.111 0.000( 0.00) 0.000( 0.00)
                    1+( 7) 19.408 2+( 9) 16.111 3.297 0.005( 0.00) 0.003( 0.00)
                    5.0+(60) 32.170  4.0+(100) 31.734  0.436    0.198( 0.11)    0.242( 0.14)
                    0.0+(46)', '47.248', '1.0+(97)', '45.384', '1.864', '23.973(13.39)', '7.991(', '4.46)
                    """
                    tmp = line.split()
                    len_tmp = len(tmp)
                    case_ = None # Used for identifying which if-else case reads wrong.
                    
                    # Location of initial parity is common for all cases.
                    parity_idx = tmp[0].index("(") - 1 # Find index of initial parity.
                    parity_initial = 1 if tmp[0][parity_idx] == "+" else -1
                    parity_initial_symbol = tmp[0][parity_idx]
                    
                    # Location of initial spin is common for all cases.
                    spin_initial = float(Fraction(tmp[0][:parity_idx]))
                    
                    if (tmp[1][-1] != ")") and (tmp[3][-1] != ")") and (len_tmp == 9):
                        """
                        Example:
                        J_i    Ex_i     J_f    Ex_f   dE        B(M1)->         B(M1)<- 
                        2+(11)   18.393  2+(10)    17.791  0.602    0.1(    0.0)    0.1(    0.0)
                        5.0+(60) 32.170  4.0+(100) 31.734  0.436    0.198( 0.11)    0.242( 0.14)
                        """
                        case_ = 0
                        E_gamma = float(tmp[4])
                        Ex_initial = float(tmp[1])
                        reduced_transition_prob_decay = float(tmp[5][:-1])
                        reduced_transition_prob_excite = float(tmp[7][:-1])
                        spin_final = float(Fraction(tmp[2].split(parity_initial_symbol)[0]))
                        Ex_final = float(tmp[3])
                        parity_final = tmp[2].split("(")[0][-1]

                    elif (tmp[1][-1] != ")") and (tmp[3][-1] == ")") and (len_tmp == 10):
                        """
                        Example:
                        J_i    Ex_i     J_f    Ex_f   dE        B(M1)->         B(M1)<- 
                        2+(10) 17.791 2+( 1) 5.172 12.619 0.006( 0.00) 0.006( 0.00)
                        """
                        case_ = 1
                        E_gamma = float(tmp[5])
                        Ex_initial = float(tmp[1])
                        reduced_transition_prob_decay = float(tmp[6][:-1])
                        reduced_transition_prob_excite = float(tmp[8][:-1])
                        spin_final = float(Fraction(tmp[2][:-2]))
                        Ex_final = float(tmp[4])
                        parity_final = tmp[2].split("(")[0][-1]
                    
                    elif (tmp[1][-1] == ")") and (tmp[4][-1] != ")") and (len_tmp == 10):
                        """
                        Example:
                        J_i    Ex_i     J_f    Ex_f   dE        B(M1)->         B(M1)<- 
                        3+( 8)   19.503 2+(11)    18.393 1.111 0.000( 0.00) 0.000( 0.00)
                        1.0+( 1) 5.357  0.0+(103) 0.000  5.357 0.002( 0.00) 0.007( 0.00)
                        """
                        case_ = 2
                        E_gamma = float(tmp[5])
                        Ex_initial = float(tmp[2])
                        reduced_transition_prob_decay = float(tmp[6][:-1])
                        reduced_transition_prob_excite = float(tmp[8][:-1])
                        spin_final = float(Fraction(tmp[3].split(parity_initial_symbol)[0]))
                        Ex_final = float(tmp[4])
                        parity_final = tmp[3].split("(")[0][-1]

                    elif (tmp[1][-1] == ")") and (tmp[4][-1] == ")") and (len_tmp == 11):
                        """
                        Example:
                        J_i    Ex_i     J_f    Ex_f   dE        B(M1)->         B(M1)<- 
                        1+( 7) 19.408 2+( 9) 16.111 3.297 0.005( 0.00) 0.003( 0.00)
                        """
                        case_ = 3
                        E_gamma = float(tmp[6])
                        Ex_initial = float(tmp[2])
                        reduced_transition_prob_decay = float(tmp[7][:-1])
                        reduced_transition_prob_excite = float(tmp[9][:-1])
                        spin_final = float(Fraction(tmp[3][:-2]))
                        Ex_final = float(tmp[5])
                        parity_final = tmp[3].split("(")[0][-1]

                    elif (tmp[5][-1] == ")") and (tmp[2][-1] == ")") and (len_tmp == 8):
                        """
                        Example:
                        J_i    Ex_i     J_f    Ex_f   dE        B(M1)->         B(M1)<- 
                        0.0+(46) 47.248  1.0+(97) 45.384  1.864   23.973(13.39)    7.991( 4.46)
                        """
                        case_ = 4
                        E_gamma = float(tmp[4])
                        Ex_initial = float(tmp[1])
                        reduced_transition_prob_decay = float(tmp[5].split("(")[0])
                        reduced_transition_prob_excite = float(tmp[6][:-1])
                        spin_final = float(Fraction(tmp[2].split(parity_initial_symbol)[0]))
                        Ex_final = float(tmp[3])
                        parity_final = tmp[2].split("(")[0][-1]

                    else:
                        msg = "ERROR: Structure not accounted for!"
                        msg += f"\n{line=}"
                        raise KshellDataStructureError(msg)

                    if (spin_final == -1) or (spin_initial == -1):
                        """
                        -1 spin states in the KSHELL data file indicates
                        bad states which should not be included.
                        """
                        self.minus_one_spin_counts[1] += 1  # Debug.
                        continue
                    
                    reduced_transition_prob_decay_list.append([
                        Ex_initial, reduced_transition_prob_decay, E_gamma
                    ])
                    if parity_final == "+":
                        parity_final = 1
                    elif parity_final == "-":
                        parity_final = -1
                    else:
                        msg = "Could not properly read the final parity!"
                        raise KshellDataStructureError(msg)

                    self.transitions.append([
                        2*spin_initial, parity_initial, Ex_initial, 2*spin_final,
                        parity_final, Ex_final, E_gamma, reduced_transition_prob_decay,
                        reduced_transition_prob_excite
                    ])

                except ValueError as err:
                    """
                    One of the float conversions failed indicating that
                    the structure of the line is not accounted for.
                    """
                    msg = "\n" + err.__str__() + f"\n{case_=}" + f"\n{line=}"
                    raise KshellDataStructureError(msg)

                except IndexError:
                    """
                    End of probabilities.
                    """
                    break

        with open(self.fname_summary, "r") as infile:
            for line in infile:
                tmp = line.split()
                try:
                    if tmp[0] == "Energy":
                        self.Ex = []    # NOTE: Remove this?
                        self.levels = [] # [Ei, 2*spin_initial, parity].
                        load_energy_levels(infile)
                    
                    elif tmp[0] == "B(E2)":
                        self.BE2 = []
                        if self.transitions is None:
                            self.transitions = []
                        load_transition_probabilities(infile, self.BE2)
                    
                    elif tmp[0] == "B(M1)":
                        self.BM1 = []
                        if self.transitions is None:
                            self.transitions = []
                        load_transition_probabilities(infile, self.BM1)
                
                except IndexError:
                    """
                    Skip blank lines.
                    """
                    continue

        self.levels = np.array(self.levels)
        self.transitions = np.array(self.transitions)
        self.Ex = np.array(self.Ex)
        self.BM1 = np.array(self.BM1)
        self.BE2 = np.array(self.BE2)
        self.debug = "DEBUG\n"
        self.debug += f"skipped -1 states in levels: {self.minus_one_spin_counts[0]}\n"
        self.debug += f"skipped -1 states in transitions: {self.minus_one_spin_counts[1]}\n"
        self.debug = np.array(self.debug)

        try:
            self.transitions_BE2 = self.transitions[:len(self.BE2)]
            self.transitions_BM1 = self.transitions[len(self.BE2):]
        except TypeError:
            """
            TypeError: len() of unsized object because self.BE2 = None.
            """
            self.transitions_BE2 = np.array(None)
            self.transitions_BM1 = np.array(None)

        if self.load_and_save_to_file:
            np.save(file=levels_fname, arr=self.levels, allow_pickle=True)
            np.save(file=transitions_fname, arr=self.transitions, allow_pickle=True)
            np.save(file=transitions_BM1_fname, arr=self.transitions_BM1, allow_pickle=True)
            np.save(file=transitions_BE2_fname, arr=self.transitions_BE2, allow_pickle=True)
            np.save(file=Ex_fname, arr=self.Ex, allow_pickle=True)
            np.save(file=BM1_fname, arr=self.BM1, allow_pickle=True)
            np.save(file=BE2_fname, arr=self.BE2, allow_pickle=True)
            np.save(file=debug_fname, arr=self.debug, allow_pickle=True)

    def level_plot(self,
        max_spin_states: int = 1000,
        filter_spins: Union[None, list] = None
        ):
        """
        Wrapper method to include level plot as an attribute to this
        class. Generate a level plot for a single isotope. Spin on the x
        axis, energy on the y axis.

        Parameters
        ----------
        max_spin_states : int
            The maximum amount of states to plot for each spin. Default
            set to a large number to indicate ≈ no limit.

        filter_spins : Union[None, list]
            Which spins to include in the plot. If `None`, all spins are
            plotted. Defaults to `None`
        """
        level_plot(
            levels = self.levels,
            max_spin_states = max_spin_states,
            filter_spins = filter_spins
        )

    def level_density_plot(self,
            bin_size: Union[int, float],
            plot: bool = True
        ):
        """
        Wrapper method to include level density plotting as
        an attribute to this class. Generate the level density with the
        input bin size.

        Parameters
        ----------
        bin_size : Union[int, float]
            Energy interval of which to calculate the density.

        plot : bool
            Toogle plotting on / off.

        Returns
        -------
        bins : np.ndarray
            The corresponding bins (x value for plotting).

        density : np.ndarray
            The level density.
        """
        bins, density = level_density(
            energy_levels = self.levels[:, 0],
            bin_size = bin_size,
            plot = plot
        )

        return bins, density

    def gamma_strength_function_average_plot(self,
        bin_width: Union[float, int],
        Ex_min: Union[float, int],
        Ex_max: Union[float, int],
        multipole_type: str = "M1",
        plot: bool = True
        ):
        """
        Wrapper method to include gamma ray strength function
        calculations as an attribute to this class.

        Parameters
        ----------
        bin_width : Union[float, int]
            The width of the energy bins. A bin width of 0.2 contains 20
            states of uniform spacing of 0.01.

        Ex_min : Union[float, int]
            Lower limit for initial level excitation energy, usually in MeV.

        Ex_max : Union[float, int]
            Upper limit for initial level excitation energy, usually in MeV.

        multipole_type : str
            Choose whether to calculate for 'E1', 'M1' or 'E2'. NOTE:
            Currently only M1 is implemented.

        plot : bool
            Toogle plotting on / off.
        """
        transitions_dict = {
            "M1": self.transitions_BM1,
            "E2": self.transitions_BE2
        }
        bins, gsf = gamma_strength_function_average(
            levels = self.levels,
            transitions = transitions_dict[multipole_type],
            bin_width = bin_width,
            Ex_min = Ex_min,
            Ex_max = Ex_max,
            multipole_type = multipole_type,
            plot = plot
        )

        return bins, gsf

    def gsf(self,
        bin_width: Union[float, int],
        Ex_min: Union[float, int],
        Ex_max: Union[float, int],
        multipole_type: str = "M1",
        plot: bool = True
        ):
        """
        Alias for gamma_strength_function_average_plot.
        """
        return self.gamma_strength_function_average_plot(bin_width, Ex_min, Ex_max, multipole_type, plot)

    @property
    def help(self):
        """
        Generate a list of instance attributes without magic and private
        methods.

        Returns
        -------
        help_list : list
            A list of non-magic instance attributes.
        """
        help_list = []
        for elem in dir(self):
            if not elem.startswith("_"):   # Omit magic and private methods.
                help_list.append(elem)
        
        return help_list

def _process_kshell_output_in_parallel(args):
    """
    Simple wrapper for parallelizing loading of KSHELL files.
    """
    filepath, load_and_save_to_file = args
    print(filepath)
    return ReadKshellOutput(filepath, load_and_save_to_file)

def loadtxt(
    path: str,
    is_directory: bool = False,
    filter_: Union[None, str] = None,
    load_and_save_to_file: Union[bool, str] = True
    ) -> list:
    """
    Wrapper for using ReadKshellOutput class as a function.
    TODO: Consider changing 'path' to 'fname' to be the same as
    np.loadtxt.

    Parameters
    ----------
    path : str
        Filename (and path) of `KSHELL` output data file, or path to
        directory containing sub-directories with `KSHELL` output data.
    
    is_directory : bool
        If True, and 'path' is a directory containing sub-directories
        with `KSHELL` data files, the contents of 'path' will be scanned
        for `KSHELL` data files. Currently supports only summary files.

    filter_ : Union[None, str]
        NOTE: Shouldnt the type be list, not str?

    load_and_save_to_file : Union[bool, str]
        Toggle saving data as `.npy` files on / off. If 'overwrite',
        saved `.npy` files are overwritten.

    Returns
    -------
    data : list
        List of instances with data from `KSHELL` data file as
        attributes.
    """
    all_fnames = None
    data = []
    if (is_directory) and (not os.path.isdir(path)):
        msg = f"{path} is not a directory"
        raise NotADirectoryError(msg)

    elif (not is_directory) and (not os.path.isfile(path)):
        msg = f"{path} is not a file"
        raise FileNotFoundError(msg)

    elif (is_directory) and (os.path.isdir(path)):
        all_fnames = {}

        for element in sorted(os.listdir(path)):
            """
            List all content in path.
            """
            if os.path.isdir(path + element):
                """
                If element is a directory, enter it to find data files.
                """
                all_fnames[element] = []    # Create blank list entry in dict for current element.
                for isotope in os.listdir(path + element):
                    """
                    List all content in the element directory.
                    """
                    if isotope.startswith("summary") and isotope.endswith(".txt"):
                        """
                        Extract summary data files.
                        """
                        try:
                            """
                            Example: O16.
                            """
                            n_neutrons = int(isotope[9:11])
                        except ValueError:
                            """
                            Example: Ne20.
                            """
                            n_neutrons = int(isotope[10:12])

                        n_neutrons -= atomic_numbers[element.split("_")[1]]
                        all_fnames[element].append([element + "/" + isotope, n_neutrons])
        
        pool = multiprocessing.Pool()
        for key in all_fnames:
            """
            Sort each list in the dict by the number of neutrons. Loop
            over all directories in 'all_fnames' and extract KSHELL data
            and append to a list.
            """
            if filter_ is not None:
                if key.split("_")[1] not in filter_:
                    """
                    Skip elements not in filter_.
                    """
                    continue

            all_fnames[key].sort(key=lambda tup: tup[1])   # Why not do this when directory is listed?
            sub_fnames = all_fnames[key]
            arg_list = [(path + i[0], load_and_save_to_file) for i in sub_fnames]
            data += pool.map(_process_kshell_output_in_parallel, arg_list)

    else:
        """
        Only a single KSHELL data file.
        """
        data.append(ReadKshellOutput(path, load_and_save_to_file))

    if not data:
        msg = "No KSHELL data loaded. Most likely error is that the given"
        msg += f" directory has no KSHELL data files. {path=}"
        raise RuntimeError(msg)

    return data

def _get_timing_data(path: str):
    """
    Get timing data from KSHELL log files.

    Parameters
    ----------
    path : str
        Path to log file.

    Examples
    --------
    Last 10 lines of log_Ar30_usda_m0p.txt:
    ```
          total      20.899         2    10.44928   1.0000
    pre-process       0.029         1     0.02866   0.0014
        operate       3.202      1007     0.00318   0.1532
     re-orthog.      11.354       707     0.01606   0.5433
  thick-restart       0.214        12     0.01781   0.0102
   diag tri-mat       3.880       707     0.00549   0.1857
           misc       2.220                         0.1062

           tmp        0.002       101     0.00002   0.0001
    ```
    """

    if "log" not in path:
        msg = "Unknown log file name!"
        raise KshellDataStructureError(msg)

    if not os.path.isfile(path):
        raise FileNotFoundError(path)

    res = os.popen(f'tail -n 20 {path}').read()    # Get the final 10 lines.
    res = res.split("\n")
    total = None
    
    if "tr" not in path:
        """
        KSHELL log.
        """
        for elem in res:
            tmp = elem.split()
            try:
                if tmp[0] == "total":
                    total = float(tmp[1])
                    break
            except IndexError:
                continue
        
    elif "tr" in path:
        """
        Transit log.
        """
        for elem in res:
            tmp = elem.split()
            try:
                if tmp[0] == "total":
                    total = float(tmp[3])
                    break
            except IndexError:
                continue

    if total is None:
        msg = f"Not able to extract timing data from '{path.split('/')[-1]}'!"
        raise KshellDataStructureError(msg)
    
    return total

def get_timing_data(path: str):
    """
    Wrapper for _get_timing_data. Input a single log filename and get
    the timing data. Input a path to a directory several log files and
    get the summed timing data. In units of seconds.

    Parameters
    ----------
    path : str
        Path to a single log file or path to a directory of log files.
    """
    if os.path.isfile(path):
        return _get_timing_data(path)
    
    elif os.path.isdir(path):
        total = 0
        for elem in os.listdir(path):
            if ("log_" in elem) and elem.endswith(".txt"):
                total += _get_timing_data(f"{path}/{elem}")
        
        return total

    else:
        msg = f"'{path}' is neither a file nor a directory!"
        raise NotADirectoryError(msg)