# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['coleo']

package_data = \
{'': ['*']}

install_requires = \
['ptera>=0.2.1,<0.3.0']

setup_kwargs = {
    'name': 'coleo',
    'version': '0.1.3',
    'description': 'The nicest way to develop a command-line interface',
    'long_description': '\n# Coleo\n\nColeo is a minimum-effort way to create a command-line interface in Python.\n\n* Declare options where they are used.\n* Scale easily to extensive CLIs with dozens of subcommands and options.\n\n\n## Basic usage\n\nFirst, define a command line interface as follows:\n\n```python\nfrom coleo import Argument, auto_cli, default\n\ndef main():\n    # The greeting\n    greeting: Argument = default("Hello")\n\n    # The name to greet\n    name: Argument = default("you")\n\n    return f"{greeting}, {name}!"\n\nif __name__ == "__main__":\n    auto_cli(main)\n```\n\nThen you may run it like this on the command line:\n\n```bash\n$ python hello.py\nHello, you!\n$ python hello.py --name Luke\nHello, Luke!\n$ python hello.py --name Luke --greeting "Happy birthday"\nHappy birthday, Luke!\n$ python hello.py -h\nusage: hello.py [-h] [--greeting VALUE] [--name VALUE]\n\noptional arguments:\n  -h, --help        show this help message and exit\n  --greeting VALUE  The greeting\n  --name VALUE      The name to greet\n```\n\n* Any variable annotated with `Argument` will become an option.\n* You can provide a default value with `default(value)`, although you don\'t have to, if the argument is required.\n* If there is a comment above the variable, it will be used as documentation for the option.\n\n\n## Argument types\n\nBy default, all arguments are interpreted as strings, but you can easily give a different type to an argument:\n\n```python\ndef main():\n    # This argument will be converted to an int\n    x: Argument & int\n    # This argument will be converted to a float\n    y: Argument & float\n    return x + y\n```\n\n**Boolean flags**\n\nIf the type is bool, the option will take no argument, and the `--no-<optname>` option is added to negate the value. For example:\n\n```python\ndef main():\n    flag: Argument & bool\n    return "yes!" if flag else "no!"\n```\n\nUse it like this:\n\n```bash\n$ python script.py --flag\nyes!\n$ python script.py --no-flag\nno!\n```\n\n**Files**\n\nUse `coleo.FileType` (or `argparse.FileType`, it\'s the same thing) to open a file to read from or to write to:\n\n```python\ndef main():\n    grocery_list: Argument & coleo.FileType("r")\n    with grocery_list as f:\n        for food in f.readlines():\n            print(f"Gotta buy some {food}")\n```\n\n**Config**\n\nYou can manipulate configuration files with `coleo.config` or `coleo.ConfigFile`:\n\n```python\ndef main():\n    # ConfigFile lets you read or write a configuration file\n    book: Argument & ConfigFile\n    contents = book.read()\n    contents["xyz"] = "abc"\n    book.write(contents)\n\n    # config will read the file for you or parse the argument as JSON\n    magazine: Argument & config\n    print(magazine)\n```\n\nUse it simply like this:\n\n```bash\n$ python librarian.py --book alice.json --magazine vogue.json\n$ python librarian.py --book history.yaml --magazine gamez.toml\n$ python librarian.py --book physics.json --magazine \'{"a": 1, "b": 2}\'\n# etc\n```\n\nSupported extensions are `json`, `yaml` and `toml` (the latter two require installing the `pyyaml` or `toml` packages).\n\n\n**Other**\n\nAny function can be used as a "type" for an argument. So for example, if you want to be able to provide lists and dictionaries on the command line you can simply use `json.loads` (although `coleo.config` is usually better, because it can also read files, in various formats):\n\n```python\ndef main():\n    obj: Argument & json.loads\n    return type(obj).__name__\n```\n\n```bash\n$ python json.py --obj 1\nint\n$ python json.py --obj \'"hello"\'\nstr\n$ python json.py --obj \'{"a": 1, "b": 2}\'\ndict\n```\n\nIf you\'re feeling super feisty and care nothing about safety, you can even use `eval`:\n\n```python\ndef main():\n    obj: Argument & eval\n    return type(obj).__name__\n```\n\n```bash\n$ python eval.py --obj "1 + 2"\nint\n$ python eval.py --obj "lambda x: x + 1"\nfunction\n```\n\n\n## Customization\n\nUsing comments of the form `# [<instruction>: <args ...>]` you can customize the option parser:\n\n```python\ndef main():\n    # This argument can be given as either --greeting or -g\n    # [alias: -g]\n    greeting: Argument = default("Hello")\n\n    # This argument is positional\n    # [positional]\n    name: Argument = default("you")\n\n    # This argument can only be given as -n\n    # [options: -n]\n    ntimes: Argument & int = default(1)\n\n    for i in range(ntimes):\n        print(f"{greeting}, {name}!")\n```\n\nThe above would be used like this:\n\n```bash\n$ python hello.py Alice -g Greetings -n 2\nGreetings, Alice!\nGreetings, Alice!\n```\n\nThe following customizations are available:\n\n* `[alias: ...]` defines one or several options that are aliases for the main one. Options are separated by spaces, commas or semicolons.\n* `[options: ...]` defines one or several options for this argument, which *override* the default one. Options are separated by spaces, commas or semicolons.\n* `[positional]` defines one positional argument.\n  * `[positional: n]`: n positional arguments (a list is returned).\n  * `[positional: ?]`: one optional positional argument\n  * `[positional: *]`: zero or more positional arguments\n  * `[positional: +]`: one or more positional arguments\n\nIt is currently not possible to define multiple arguments as positional, although you can simulate it by using e.g. `[positional: 2]` on a single argument to get a pair, and then you can get the elements of the list.\n\n\n## Subcommands\n\nYou can create an interface with a hierarchy of subcommands by passing a dictionary to `auto_cli`:\n\n```python\ndef add():\n    x: Argument & int\n    y: Argument & int\n    return x + y\n\ndef mul():\n    x: Argument & int\n    y: Argument & int\n    return x * y\n\ndef pow():\n    base: Argument & int\n    exponent: Argument & int\n    return base ** exponent\n\ndef greet():\n    greeting: Argument = default("Hello")\n    name: Argument = default("you")\n    return f"{greeting}, {name}!"\n\nif __name__ == "__main__":\n    auto_cli({\n        "calc": {\n            "__doc__": "Calculate something!",\n            "add": add,\n            "mul": mul,\n            "pow": pow,\n        },\n        "greet": greet,\n    })\n```\n\nThen you may use it like this:\n\n```bash\n$ python multi.py greet --name Alice --greeting Hi\nHi, Alice!\n$ python multi.py calc add --x=3 --y=8\n11\n```\n\n\n## Sharing arguments\n\nIt is possible to share behavior and arguments between subcommands, or to split complex functionality into multiple pieces. For example, maybe multiple subcommands in your application require an API key, which can either be given on the command line or can be read from a file. This is how you would share this behavior across all subcommands:\n\n```python\nfrom coleo import Argument, auto_cli, default, tooled\n\n@tooled\ndef apikey():\n    # The API key to use\n    key: Argument = default(None)\n    if key is None:\n        # If no key parameter is given on the command line, try to read it from\n        # some standard location.\n        key = config("~/.config/myapp/config.json")["key"]\n    return key\n\ndef search():\n    interface = Application(apikey())\n    query: Argument\n    return interface.search(query)\n\ndef install():\n    interface = Application(apikey())\n    package: Argument\n    return interface.install(package)\n\nif __name__ == "__main__":\n    auto_cli({"search": search, "install": install})\n```\n\nIf a function is decorated with `@tooled` and is called from one of the main functions (or from another tooled function), Coleo will search for arguments in that function too. Thus any subcommand that calls `apikey()` will gain a `--key` option.\n\nIn addition to this, you can "share" arguments by defining the same argument with the same type in multiple functions. Coleo will set all of them to the same value.\n\nFor example, in the example above you could easily let the user specify the path to the file that contains the key, simply by replacing\n\n```python\nkey = config("~/.config/myapp/config.json")["key"]\n\n# ==>\n\nconfig_path: Argument = default("~/.config/myapp/config.json")\nkey = config(config_path)["key"]\n```\n\nAnd that `config_path` argument could, of course, be declared in any other function that needs to read some configuration value.\n\n\n## Non-CLI usage\n\nIt is possible to set arguments without `auto_cli` with `setvars`:\n\n```python\nfrom coleo import Argument, setvars, tooled\n\n@tooled\ndef greet():\n    greeting: Argument = default("Hello")\n    name: Argument = default("you")\n    return f"{greeting} {name}!"\n\nwith setvars(greeting="Hi", name="Bob"):\n    assert greet() == "Hi bob!"\n```\n\nNote:\n\n* With `setvars`, you *must* decorate the function with `@tooled` (this is something `auto_cli` does on your behalf).\n* `setvars` entirely bypasses the option parsing and the type annotations will not be used to wrap these values. In other words, if a variable is annotated `Argument & int` and you provide the value "1", it will remain a string.\n\n\n### Using with Ptera\n\nColeo is based on [Ptera](https://github.com/mila-iqia/ptera) and all of Ptera\'s functionality is de facto available on functions marked as `@tooled`. For example, using the example above:\n\n```python\n# Set the variables in the greet function -- it\'s a bit like making an object\nhibob = greet.new(greeting="Hi", name="Bob")\nassert hibob() == "Hi Bob!"\n\n# Same as above but this would also change greeting/name in any other function\n# that is called by greet, and so on recursively (a bit like dynamic scoping)\nhibob = greet.tweaking({"greeting": "Hi", "name": "Bob"})\nassert hibob() == "Hi Bob!"\n\n# More complex behavior\nfrom ptera import overlay\nwith overlay.tweaking({\n    "greet(greeting=\'Bonjour\') > name": "Toto"\n}):\n    assert greet() == "Hello you!"\n    assert greet.new(greeting="Hi")() == "Hi you!"\n    assert greet.new(greeting="Bonjour")() == "Bonjour toto!"\n```\n\nRead the documentation for [Ptera](https://github.com/mila-iqia/ptera) for more information. Note that Ptera is not limited to variables tagged `Argument`, it can manipulate *any* variable in a tooled function.\n',
    'author': 'Olivier Breuleux',
    'author_email': 'breuleux@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/breuleux/coleo',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
