#!/usr/bin/env python
#
# -*- coding: utf-8 -*-
# pylint: disable=invalid-name, line-too-long, too-many-nested-blocks, too-many-branches, too-many-locals
#

"""
PUG: 'Pug, the Uefi Guidedog', or 'the Programmer's Uefi Guide'.
A front-end to build the Uefi driver(s) with a .PY as the configuration file.

(c) 2019-2021 Timothy Lin <timothy.gh.lin@gmail.com>, BSD 3-Clause License.

Prerequisites:
1. Python 3.7.0+
2. git 2.25.0+

Generic prerequisites for the UEFI build:
0. Ref. https://github.com/tianocore/tianocore.github.io/wiki/Getting%20Started%20with%20EDK%20II
        Xcode: https://github.com/tianocore/tianocore.github.io/wiki/Xcode
1. nasm (2.0 or above)
2. iasl (version 2018xxxx or later)
3. GCC(Posix) or MSVC(Windows)
4. build-essential uuid-dev (Posix)
5. py -3 -m pip install future (Windows)
   python3 -m pip install future
6. motc (Xcode)

Tool installation for any Debian Based Linux:
1. sudo apt-get update && sudo apt-get install nasm iasl build-essential uuid-dev python3-pip python-is-python3

TODO:
- The driver's specific settings can reside in its major .C file for self-contained purpose. -- i.e. relocate them from config.py
- keyword list of the supported section names of DSC and INF.
- X64/IA32/ARM/... section differentiation.
- automate the tool-chain for Windows/Linux/Mac.

"""

from __future__ import print_function
from __future__ import absolute_import

__all__ = ['build', 'build_basetools', 'run', 'setup_codetree', 'main']

import os
import sys
import time
import shutil
import threading
import subprocess
import multiprocessing

from . import config             # Invoke config.py in the same folder

sys.dont_write_bytecode = True      # To inhibit the creation of .pyc file

dry_run = False

if dry_run:
    VERBOSE_THRESHOLD = -1
else:
    VERBOSE_THRESHOLD = config.VERBOSE_THRESHOLD


UDKBUILD_MAKETOOL = 'nmake' if (os.name == 'nt') else 'make'
UDKBUILD_COMMAND_JOINTER = '&&' if (os.name == 'nt') else ';'

default_pug_signature = '#\n# Do not edit this file.\n# It is automatically created by PUG.\n#\n'

edk2_actions = ['all', 'fds', 'genc', 'genmake', 'clean', 'cleanall', 'cleanlib', 'modules', 'libraries', 'run']
pug_action_clean = ['clean', 'cleanall']
pug_action_all = ['build', 'setup', 'clean', 'cleanall', 'init', 'init-basetools', 'clean-basetools', 'help']


def pwdpopd(target_dir=''):
    """pwd | popd
    when a directory is assigned, (1) the current directory is pushed to the stack, (2) chdir() to that assigned directory.
    when a directory is not assigned, (1) a directory is popped from the stack, (2) chdir() to that popped directory.
    It's the caller's responsibility to maintain the stack's balance.
    """
    if target_dir:
        pwdpopd.pushd += [os.getcwd()]
    else:
        if pwdpopd.pushd:
            target_dir = pwdpopd.pushd.pop()
    if target_dir:
        os.chdir(target_dir)
pwdpopd.pushd = []


def bowwow(msg, noise_pitch=0, no_clobber=False):
    """display some tagged progress messages when iPug is running."""
    if noise_pitch >= VERBOSE_THRESHOLD:
        if no_clobber:
            print(msg)
        else:
            pugsay = 'PUG: '
            if msg.startswith('\n'):
                print('\n')
                msg = msg[1:]
            msg = msg.replace('\n', '\n{}'.format(pugsay))
            print('{0}{1}'.format(pugsay, str(msg)))
        sys.stdout.flush()


def abs_path(sub_dir, base_dir):
    """return an absolute path."""
    return sub_dir if os.path.isabs(sub_dir) else os.path.join(base_dir, sub_dir)


def write_file(path, content, signature=''):
    """update a platform's dsc file content.
    - create the folder when it does not exist.
    - skip write attempt when the contents are identical"""

    if isinstance(content, (list, tuple)):
        content = '\n'.join(content)
        if not content.endswith('\n'):
            content += '\n'
    if signature:
        content = signature + content
    path_dir = os.path.dirname(path)
    content0 = ''
    if not os.path.exists(path_dir):
        os.makedirs(path_dir)
    else:
        if os.path.exists(path):
            with open(path, 'r') as pf:
                content0 = pf.read()
            if content0 == content:
                return
    with open(path, 'w') as pf:
        pf.write(content)


def conf_files(files, dest_conf_dir, cmd_arg, verbose=False):
    """Ref. BaseTools/BuildEnv for build_rule.txt , tools_def.txt and target.txt"""
    dest_conf_dir = os.path.abspath(dest_conf_dir)
    if not os.path.exists(dest_conf_dir):
        os.makedirs(dest_conf_dir)
    os.environ['CONF_PATH'] = dest_conf_dir
    if cmd_arg[0] in {'setup', 'init'}:
        src_conf_dir = os.path.join(os.environ.get('EDK_TOOLS_PATH', os.path.join(os.environ['WORKSPACE'], 'BaseTools')), 'Conf')
        for f in files:
            src_conf_path = os.path.join(src_conf_dir, '%s.template' % f)
            dest_conf_path = os.path.join(dest_conf_dir, '%s.txt' % f)
            if verbose:
                bowwow('Copy %s\nTo   %s' % (src_conf_path, dest_conf_path), noise_pitch=1)
            shutil.copyfile(src_conf_path, dest_conf_path)


def gen_section(items, override=None, section='', sep='=', ident=0):
    """generate a section's content"""
    ret_list = []
    if section:
        ret_list += ['\n%s[%s]' % (' '*ident*2, section)]
    if items:
        if isinstance(items, (tuple, list)) or (override in {list, tuple}):
            for d in items:
                if d:
                    if isinstance(d, (list, tuple)):
                        ret_list += ['%s%s' % (' '*(ident+1)*2, sep.join(d))]
                    else:
                        ret_list += ['%s%s' % (' '*(ident+1)*2, str(d))]
        elif isinstance(items, dict):
            ret_list += ['%s%s %s %s' % (' '*(ident+1)*2, str(d), sep, str(items[d])) for d in sorted(items) if d]
    return ret_list


def gen_target_txt(target_txt):
    """generate the content of Conf/target.txt"""
    tt = []
    for s in sorted(target_txt):
        if s.isupper():
            tt += ['%s = %s' % (s, target_txt[s])]
    write_file(target_txt['path'], tt)


def run(Command, WorkingDir='.', verbose=VERBOSE_THRESHOLD<=1):
    """A derivative of EDK2's BaseTools/build/build.py::launch_command

        returns
        [0] - error code
        [1] - buffered stdout content, list
        [2] - buffered stderr content, list
    """
    stdout_buffer = []
    stderr_buffer = []

    def ReadMessage(From, To, ExitFlag):
        """read message fro stream"""
        while True:
            Line = From.readline()
            if Line:
                To(Line.rstrip())
            if not Line or ExitFlag.isSet():
                break

    def __logger(msg, buf):
        if isinstance(msg, bytes):
            msg = msg.decode('utf-8')
        if verbose:
            bowwow('%s' % msg, noise_pitch=1)
        buf += [msg]

    def logger_stdout(msg):
        """print message from stdout"""
        __logger(msg, stdout_buffer)

    def logger_stderr(msg):
        """print message from stderr"""
        __logger(msg, stderr_buffer)

    if isinstance(Command, (list, tuple)):
        Command = ' '. join(Command)

    WorkingDir = os.path.abspath(WorkingDir)
    pwdpopd(WorkingDir)
    bowwow('Run: [%s] @ [%s]' % (Command, WorkingDir), noise_pitch=2)

    if dry_run:
        return 0, ['dry-run-stdout'], ['']

    Proc = EndOfProcedure = StdOutThread = StdErrThread = None
    _stdout = sys.stdout if verbose else subprocess.PIPE
    _stderr = sys.stderr if verbose else subprocess.PIPE
    Proc = subprocess.Popen(Command, stdout=_stdout, stderr=_stderr, env=os.environ, cwd=WorkingDir, bufsize=-1, shell=True)
    if not verbose:
        EndOfProcedure = threading.Event()
        EndOfProcedure.clear()
        if Proc.stdout:
            StdOutThread = threading.Thread(target=ReadMessage, args=(Proc.stdout, logger_stdout, EndOfProcedure))
            StdOutThread.setName('STDOUT-Redirector')
            StdOutThread.setDaemon(False)
            StdOutThread.start()
        if Proc.stderr:
            StdErrThread = threading.Thread(target=ReadMessage, args=(Proc.stderr, logger_stderr, EndOfProcedure))
            StdErrThread.setName('STDERR-Redirector')
            StdErrThread.setDaemon(False)
            StdErrThread.start()
        # waiting for program exit
    Proc.wait()

    return_code = -1
    if Proc:
        if Proc.stdout and StdOutThread:
            StdOutThread.join()
        if Proc.stderr and StdErrThread:
            StdErrThread.join()
        return_code = Proc.returncode
    pwdpopd()
    return return_code, stdout_buffer, stderr_buffer


def print_run_result(r, prompt=''):
    """print the stdout & stderr when return code is non-zero.

       returns the caller's return code"""
    if r[0]:
        s1 = '\n'.join(r[1])
        s2 = '\n'.join(r[2])
        if s1 or s2:
            bowwow('{0}{1}'.format(prompt, '\n'.join([s1, 'STDERR:', s2])), noise_pitch=2)
    else:
        bowwow('{0}Success'.format(prompt), noise_pitch=2)
    return r[0]


def locate_nasm():
    """Try to locate the nasm's installation directory. For Windows only."""
    for d in [
            'C:\\Program Files\\NASM\\nasm.exe',
            'C:\\Program Files (x86)\\NASM\\nasm.exe',
            os.environ.get('LOCALAPPDATA', '') + '\\bin\\NASM\\nasm.exe',
            'C:\\NASM\\nasm.exe',
    ]:
        if os.path.exists(d):
            return os.path.dirname(d)
    return ''


def env_var(k, v):
    """Setup environment variable"""
    k0 = k[0]
    k1 = k[1:]
    if v[0] == '$':             # macro from os.environ
        v = os.environ.get(v[1:], '')
    if k0 in {'+', '*'}:
        try:
            ex = ''
            if k0 == '+':       # append
                ex = '%s%s%s' % (os.environ[k1], os.pathsep, v)
            elif k0 == '*':     # prepend
                ex = '%s%s%s' % (v, os.pathsep, os.environ[k1])
            os.environ[k1] = ex
        except KeyError:
            os.environ[k1] = v
    elif k0 == '=':             # conditional assignment
        if k1 not in os.environ:
            os.environ[k1] = v
    else:                       # unconditional assignment
        os.environ[k] = v


def setup_env_vars(workspace, codetree):
    """Setup environment variables"""
    env_var('=WORKSPACE', os.path.abspath(workspace))
    udk_home = config.CODETREE['edk2']['path']
    env_var('=UDK_ABSOLUTE_DIR', os.path.abspath(udk_home))
    env_var('=EDK_TOOLS_PATH', os.path.join(os.environ['UDK_ABSOLUTE_DIR'], 'BaseTools'))
    env_var('=CONF_PATH', os.path.join(os.environ['WORKSPACE'], 'Conf'))
    env_var('=BASE_TOOLS_PATH', '$EDK_TOOLS_PATH')
    env_var('=PYTHONPATH', os.path.join(os.environ['EDK_TOOLS_PATH'], 'Source', 'Python'))
    env_var('=EDK_TOOLS_PATH_BIN', os.path.join(os.environ['EDK_TOOLS_PATH'], 'BinWrappers', 'WindowsLike' if os.name == 'nt' else 'PosixLike'))

    nasm_path = ''
    if os.name == 'nt':
        env_var('*PATH', os.path.join(os.environ['EDK_TOOLS_PATH'], 'Bin', 'Win32'))
        env_var('=PYTHON_HOME', os.path.dirname(sys.executable))
        env_var('=PYTHONHOME', os.path.dirname(sys.executable))
        nasm_path = locate_nasm()
        if nasm_path:
            env_var('=NASM_PREFIX', nasm_path + os.sep)
            env_var('*PATH', nasm_path)
        env_var('=PYTHON_COMMAND', '"%s"' % sys.executable)
    env_var('*PATH', '$EDK_TOOLS_PATH_BIN')
    env_var('+PACKAGES_PATH', '$UDK_ABSOLUTE_DIR')
    env_var('+PACKAGES_PATH', os.getcwd())
    for c in codetree:
        if c == 'edk2':
            continue
        if codetree[c].get('multiworkspace', False):
            env_var('+PACKAGES_PATH', codetree[c]['path'])


def build_basetools(cmd=''):
    """build the C-Lang executable binaries in BaseTools."""
    if cmd[:2] == ['build', 'clean']:
        return 0
    home_dir = os.environ['EDK_TOOLS_PATH']
    cmds = [UDKBUILD_MAKETOOL]

    if UDKBUILD_MAKETOOL == 'make':
        cmds += [
            '--jobs', '%d' % multiprocessing.cpu_count()
        ]
    if (cmd[0] == 'clean-basetools') or (cmd[:2] == ['build', 'cleanall']):
        cmds += ['clean']

    r = run(cmds, home_dir)
    return print_run_result(r, 'build_basetools(): ')


def apply_patch(codetree, workspace):
    """apply patches to the code tree."""
    r0, r1, r2 = 0, [], []
    for c in codetree.values():
        cPatch = c.get('patch', None)
        if cPatch is None:
            continue
        s = run(cPatch, workspace)
        r0 |= s[0]
        r1 += s[1]
        r2 += s[2]
    return print_run_result((r0, r1, r2), 'apply_patch(): ')


def setup_codetree(codetree):
    """pull the edk2 code tree when it does not locally/correctly exist.
        1. git clone
        2. git checkout tag/branch/master"""

    def _get_code(node):
        """get code using git clone/checkout"""
        r = 0, [], []
        local_dir = node['path']
        dot_git = os.path.join(local_dir, '.git')
        if not os.path.exists(local_dir):
            os.makedirs(local_dir)

        nsource = node.get('source', None)
        if nsource is None:
            return r
        nsource_url = nsource.get('url', None)
        nsource_cmd = nsource.get('command', None)
        if nsource_url is None and nsource_cmd is None:
            return r

        nsource_signature = nsource.get('signature', '')
        if not nsource_signature:
            nsource_signature = 'master'
            branch_sig = []
        else:
            branch_sig = ['-b', nsource_signature]

        recurse_submodule = '--recurse-submodules' if node.get('recursive', '') else ''
        new_clone = False
        if nsource_cmd:
            if isinstance(nsource_cmd, type("")):
                r = run(nsource_cmd, local_dir, verbose=True)
            elif hasattr(nsource_cmd, '__iter__'):
                for ncmd in nsource_cmd:
                    r = run(ncmd, local_dir, verbose=True)
                    if r[0]:
                        break
            else:
                pass # TODO: this should be an error
            if r[0]:
                return r
        elif nsource_url:
            if not os.path.exists(dot_git):
                r = run(['git', 'clone', recurse_submodule, nsource_url, local_dir] + branch_sig, local_dir, verbose=True)
                if r[0]:
                    return r
                new_clone = True
            else:
                pwdpopd(local_dir)
                r = run(['git', 'fetch', '--tags', '--all', recurse_submodule], local_dir, verbose=True)
                pwdpopd()
                if r[0]:
                    return r
                r = run(['git', 'checkout', nsource_signature], local_dir, verbose=True)
                if r[0]:
                    return r
            if node.get('recursive', ''):
                if not new_clone:
                    r = run(['git', 'submodule sync --recursive'], local_dir, verbose=True)
                r = run(['git', 'submodule update --recursive'], local_dir, verbose=True)
        return r

    r0, r1, r2 = _get_code(codetree['edk2'])
    for c in codetree:
        if c == 'edk2':
            continue
        s = _get_code(codetree[c])
        r0 |= s[0]
        r1 += s[1]
        r2 += s[2]
    return print_run_result((r0, r1, r2), 'setup_codetree(): ')


def platform_dsc(platform, components, workspace):
    """generate a platform's dsc file."""

    dsc_path = abs_path(platform['path'], workspace)
    bowwow('PLATFORM_DSC = %s' % dsc_path, noise_pitch=1)
    if not platform.get('update', False):
        return
    sections = ['Defines', 'Components']
    overrides = {'LibraryClasses', 'PcdsFixedAtBuild'}  # , 'BuildOptions'}
    pfile = []
    for s in sections:
        if s == 'Components':
            pfile += gen_section(None, section=s)
            for compc in components:
                pfile += ['  %s' % compc['path']]
                in_override = False
                ovs = overrides.intersection(set(compc.keys()))
                for ov in ovs:
                    # print('Override: %s' % ov)
                    if not in_override:
                        pfile[-1] += ' {'
                        in_override = True
                    pfile += ['    <%s>' % ov]
                    sep = '|' if ov in {'LibraryClasses', 'PcdsFixedAtBuild'} else '='
                    for d in compc[ov]:
                        if d and d[0]:
                            pfile += ['      %s %s %s' % (d[0], sep, d[1])]
                if in_override:
                    pfile += ['  }']
        else:
            pfile += gen_section(platform[s], section=s)
    write_file(dsc_path, pfile, default_pug_signature)


def component_inf(components, workspace):
    """generate INF files of components."""
    sections = [
        'Sources', 'Packages', 'LibraryClasses', 'Protocols', 'Ppis',
        'Guids', 'FeaturePcd', 'Pcd', 'BuildOptions', 'Depex', 'UserExtensions',
    ]
    for comp in components:
        cfile = []
        inf_path = abs_path(comp.get('path', ''), workspace)
        bowwow('COMPONENT: %s' % inf_path, noise_pitch=1)
        if not comp.get('update', False):
            continue
        defines = comp.get('Defines', '')
        if not defines:
            raise Exception('INF must contain [Defines] section.')
        cfile += gen_section(defines, section='Defines')
        for s in comp:
            s0 = s.split('.')[0]
            if s0 not in sections:
                continue
            if s0 == 'LibraryClasses':
                cfile += gen_section([v[0] for v in comp[s] if v[0] != 'NULL'], section=s, override=list)
            else:
                cfile += gen_section(comp[s], section=s)
        write_file(inf_path, cfile, default_pug_signature)


def build(cmd_arg):
    """0. prepare the EDK2 code tree.
       1. setup environment variables.
       2. build C-Lang executable binaries in BaseTools.
       3. EDK2 build."""

    workspace = os.path.abspath(config.WORKSPACE['path'])

    # 1. (1) check the external repos and (2) apply the patches.
    if cmd_arg[0] in {'setup', 'init'}:
        r = setup_codetree(config.CODETREE)
        if r:
            bowwow('setup_codetree(0) returns: %s' % str(r), noise_pitch=2)
            bowwow('Unable to setup the EDK2 code tree correctly.', 2)
            bowwow('Please check the access permission or the sanity of the external folder(s).', 2)
            return r
        r = apply_patch(config.CODETREE, workspace)
        if r:
            bowwow('apply_patch() returns: %s' % str(r), 2)
            bowwow('The path is not applied successfully.', 2)
            bowwow('Maybe the patch has been applied before. Ignoring the error.\n', 2)
            # return r

    # 2. setup the THREE basic text files for the EDK2 build.
    setup_env_vars(workspace, config.CODETREE)
    conf_files(['build_rule', 'tools_def', 'target'], config.WORKSPACE['conf_path'], cmd_arg)
    if cmd_arg[0] in {'setup', 'init'}:
        gen_target_txt(config.TARGET_TXT)

    # 2.1 dump the essential environment variables when requested.
    if '--pug:environ' in cmd_arg[2]:
        bowwow(config.dump_env_vars(), 3, no_clobber=True)

    # 3. build/clean the BaseTools binaries
    if cmd_arg[0] in pug_action_all:
        r = build_basetools(cmd_arg)
        # BaseTools build failure is ignored quietly.
        # leave it to the EDK2's build logic to control the failure.

        if cmd_arg[0] in {'init-basetools', 'clean-basetools'}:
            return r

    # 4. [TODO] generate the temporary DSC/INF files
    cPlatform = getattr(config, 'PLATFORM', None)
    cComponent = getattr(config, 'COMPONENT', None)
    if cmd_arg[0] in {'setup', 'init'}:
        if cPlatform and cComponent:
            platform_dsc(cPlatform, cComponent, workspace)
        if cComponent:
            component_inf(cComponent, workspace)

    if cmd_arg[0] in {'setup', 'init'}:
        return 0

    # 5. run (1) the customized "build" command or (2) the default one to build the code base.
    if config.DEFAULT_BUILD_COMMAND:
        cmds = [config.DEFAULT_BUILD_COMMAND] + sys.argv[1:]
        r = run(cmds, os.environ['WORKSPACE'])
    else:
        cmds = []
        if os.name == 'nt':
            cmds += [
                os.path.join(os.environ['EDK_TOOLS_PATH'], 'toolsetup.bat'), UDKBUILD_COMMAND_JOINTER,
            ]

        cmds += ['build']

        if cmd_arg[0] != '--help':
            # -n Explicitly define the maximum threads number.
            if '-n' not in sys.argv[1:]:
                cmds += ['-n', '%d' % multiprocessing.cpu_count()]

            # platform DSC file's path (relative to WORKSPACE)
            if '-p' not in sys.argv[1:]:
                ppdsc = ''
                try:
                    ppdsc = config.DEFAULT_ACTIVE_PLATFORM
                    ppdsc = config.ACTIVE_PLATFORM
                except AttributeError:
                    pass
                if ppdsc:
                    cmds += [
                        '-p', ppdsc
                    ]

        cmds += sys.argv[1:]
        r = run(cmds, os.environ['WORKSPACE'])
    return print_run_result(r, 'build(): ')


def usage():
    """help message"""
    msg = f"""Usage: ipug [pug_action [edk2_build_argument] | [defines] ]

    pug's action:
        {'|'.join(sorted(pug_action_all))}
        -- the default action is 'build'

    edk2's build argument
        [options] [all|fds|genc|genmake|clean|cleanall|cleanlib|modules|libraries|run]
        -- run `ipug --help` for more info

    defines
        e.g TARGET=RELEASE
"""

    print(msg)


def main():
    """main"""
    cmd_arg = ['build', '', set()]

    if config.project is None:
        bowwow('Ignoring the missing project.py.', noise_pitch=0)

    start_time = time.time()

    try:
        if sys.argv[1] == 'help':
            usage()
            return 0

        if '--pug:dry-run' in sys.argv[1:]:
            global dry_run, VERBOSE_THRESHOLD
            dry_run = True
            VERBOSE_THRESHOLD = -1
            bowwow('A DRY RUN!', 3)
            cmd_arg[2].add('--pug:dry-run')
            sys.argv.remove('--pug:dry-run')

        if '--pug:config' in sys.argv[1:]:
            bowwow(config.dump_config(), 3, no_clobber=True)
            cmd_arg[2].add('--pug:config')
            sys.argv.remove('--pug:config')

        if '--pug:environ' in sys.argv[1:]:
#            bowwow(config.dump_env_vars(), 3, no_clobber=True)
            cmd_arg[2].add('--pug:environ')
            sys.argv.remove('--pug:environ')

        if '--help' in sys.argv[1:]:
            cmd_arg[0] = '--help'

        else:
            if sys.argv[1] in pug_action_clean:
                cmd_arg[1] = sys.argv[1]
                sys.argv[1:1] = [sys.argv.pop(1)]
            elif sys.argv[1] in pug_action_all:
                cmd_arg[0] = sys.argv.pop(1)
            elif sys.argv[1][0] not in {'/', '-'}:
                usage()
                return 0
    except IndexError:
        pass

    bowwow('sys.argv: %s' % str(sys.argv), 0)
    bowwow('cmd_arg: %s' % str(cmd_arg), 0)
    ret = build(cmd_arg)

    elapsed_time = time.gmtime(int(round(time.time() - start_time)))
    elapsed_time_str = time.strftime('%H:%M:%S', elapsed_time)
    if elapsed_time.tm_yday > 1:
        elapsed_time_str += ', %d day(s)' % (elapsed_time.tm_yday - 1)
    bowwow("\nPug's running elapsed time: %s" % elapsed_time_str, noise_pitch=1)
    return ret


if __name__ == '__main__':
    sys.exit(main())
