# AUTOGENERATED! DO NOT EDIT! File to edit: notebooks/aggregates_from_pote.ipynb (unless otherwise specified).

__all__ = [
    "config",
    "tc",
    "yaml",
    "extraction_date",
    "get_fake_upper_bound",
    "keep_upper_bound_secret",
    "save_agg",
    "save_agg_from_list",
    "ref",
    "labels",
    "perimeter",
    "row_to_dict",
    "get_key_value_aggregate",
    "get_key_value_aggregates",
    "metadata",
    "year",
    "aggregates",
    "liste_des_variables_csg_2018",
    "liste_des_variables_csg_2019",
    "pote",
    "get_aggregats_from_row",
    "get_aggregate_from_df",
    "get_aggregats_from_json",
    "process_all_json",
]

# Cell

import glob
import json
import time
import unittest
from pathlib import Path
from typing import List, Union

import numpy as np
import pandas as pd
from leximpact_socio_fisca_simu_etat.config import Configuration
from leximpact_socio_fisca_simu_etat.logger import logger
from ruamel.yaml import YAML

from .aggregate import Aggregate, AggregateManager, Data, Perimeter, Reference

config = Configuration(project_folder="leximpact-aggregates")
tc = unittest.TestCase()


yaml = YAML()  # typ='unsafe' for testing

extraction_date = time.strftime("%Y-%m-%d")

# Cell


def get_fake_upper_bound(val: Union[float, int]):
    """
    ::val::
    ::return::
    """
    if val == 1e15:
        return 1e15
    else:
        try:
            return 10 ** (len(str(int(val))))
        except Exception as e:
            print(val)
            raise e


# Cell
def keep_upper_bound_secret(json_content):
    if json_content[-1].get("buckets"):
        # We are in a copulas
        val = json_content[-1].get("upper_bound")
        if val:
            json_content[-1]["upper_bound"] = get_fake_upper_bound(val)
        for buck in json_content:
            val = buck["buckets"][-1].get("seuil_var_supp")
            if val:
                buck["buckets"][-1]["seuil_var_supp"] = get_fake_upper_bound(val)
    else:
        # In a calib
        val = json_content[-1].get("seuil_var_supp")
        if val:
            if type(val) is not str:
                json_content[-1]["seuil_var_supp"] = get_fake_upper_bound(val)


# Cell


def save_agg(filepath: Path, aggregate: Aggregate):
    # # Extract path without filename
    # path = Path(os.path.basename(filepath))
    # # Create directory if missing
    # path.mkdir(parents=True, exist_ok=True)
    # # Use AggregateManager to save file
    aggm = AggregateManager(aggregate)
    aggm.save_to_yaml(filepath)
    aggm.save_to_json(filepath.with_suffix(".json"))


# Cell


def save_agg_from_list(agg_list: List[Aggregate], year="2019") -> None:
    for agg in agg_list:
        var_name = (
            agg.source_variable
            if agg.openfisca_variable == ""
            else agg.openfisca_variable
        )
        path = Path(config.get("AGREGATS")) / "POTE" / year / (var_name + ".yaml")
        # print(agg,"\n-------------------------")
        save_agg(path, agg)


# Cell

ref = Reference(
    title="POTE 2019 (DGFIP)",
    href="https://www.casd.eu/source/declarations-dimpot-sur-le-revenu-des-foyers-fiscaux-formulaire-2042-et-annexes/",
)
labels = {
    "V": "Veuf(ve)",
    "C": "Célibataire",
    "O": "Pacsé(e)s",
    "D": "Divorcé(e)/séparé(e)",
    "M": "Marié(e)s",
}
perimeter = Perimeter(entity="foyer", period="year", geographic="France entière")

# Cell


def row_to_dict(row, dest_dict: dict):
    """
    Return the first column as key and second columns as row
    WARNING : Use only with 2 columns
    """
    # Force cast of float to avoid problems with ramel.YAML() dump
    key = row[0] if not isinstance(row[0], np.float64) else float(row[0])
    val = row[1] if not isinstance(row[1], np.float64) else float(row[1])
    val = "secret" if np.isnan(val) else val

    dest_dict[key] = val
    return row


# Cell


def get_key_value_aggregate(
    csv_filepath: str, var_name: str, metadata: dict
) -> Aggregate:
    df = pd.read_csv(csv_filepath)
    short_name = metadata.get("short_name")
    short_name = short_name if short_name else metadata.get("openfisca_variable")
    values = {}
    _ = df.apply(row_to_dict, args=[values], axis=1)
    data = Data(reference=[ref], date=year, values=values)
    aggregate = Aggregate(
        description=metadata.get("description"),
        short_name=short_name,
        ux_template=metadata.get("ux_template"),
        openfisca_variable=metadata.get("openfisca_variable"),
        source_variable=var_name,
        unit="foyer",
        perimeter=perimeter,
        label_of_keys=metadata.get("labels"),
        data=data,
    )
    return aggregate


# Cell


def get_key_value_aggregates(
    year: str, variables: List[str], metadata: dict
) -> List[Aggregate]:
    aggregates = []

    for var_name in variables:
        aggregate = get_key_value_aggregate(
            f"{config.get('DATASETS')}20220224_ExtractAgg/sorties/count_{year}_{var_name}.csv",
            var_name,
            metadata[var_name],
        )
        aggregates.append(aggregate)
    return aggregates


# Cell

metadata = {
    "mat": {
        "ux_name": "Situation matrimoniale",
        "description": "Situation matrimoniale du foyer fiscal",
        "openfisca_variable": "statut_marital",
        "ux_template": "Parmi les foyers français {value} sont {label}.",
        "labels": labels,
    },
    "n": {
        "ux_name": "NOMBRE D'ENFANTS MARIÉS RATTACHÉS".lower(),
        "description": "Nombre d'enfants mariés/pacsés et d'enfants non mariés chargés de famille",
        "short_name": "nb_enfants_maries",
        "openfisca_variable": "nbN",
        "ux_template": "Parmi les foyers français {value} ont fait le choix de rattacher {label} enfants mariés ou chargés de famille.",
    },
    "nbefi": {
        "ux_name": "NB TOTAL ENFANT A CHARGE I".lower(),
        "description": "Nombre d'enfants à charge non mariés, qui ne sont pas en résidence alternée, de moins de 18 ans au 1er janvier de l'année de perception des revenus, ou nés durant la même année ou handicapés quel que soit leur âge",
        "openfisca_variable": "nbF",
        "short_name": "nb_enfants_a_charge",
        "ux_template": "Parmi les foyers français {value} sont {label}.",
    },
}
year = "2019"
aggregates = get_key_value_aggregates(year, ["mat", "n", "nbefi"], metadata)

# Cell
pd.set_option("display.max_colwidth", 80)
pd.options.display.float_format = "{:,.7f}".format

liste_des_variables_csg_2018 = pd.read_csv(
    config.get("DATASETS") + "agregats_des_variables_csg-POTE_2018.csv"
)
liste_des_variables_csg_2019 = pd.read_csv(
    config.get("DATASETS") + "agregats_des_variables_csg-POTE_2019.csv"
)


pote = {"2019": liste_des_variables_csg_2019, "2018": liste_des_variables_csg_2018}

# Cell


def get_aggregats_from_row(row, agregats, year, col_name):
    """Parcours les lignes du dataframe et enregistre les agrégats dans le
    dictionnaire.

    ::row:: une ligne de dataframe
    ::year:: year of the data
    """
    values = {}
    for i, cell in enumerate(row):
        if i == 0:
            var_name = cell
        else:
            values[col_name[i]] = cell
    data = Data(reference=[ref], date=year, values=values)
    aggregate = Aggregate(
        source_variable=var_name,
        unit="foyer",
        perimeter=perimeter,
        data=data,
    )
    agregats.append(aggregate)


# Cell


def get_aggregate_from_df(df, variable_name):
    """Permet de récupérer un agrégat de type somme dans le dataframe.

    ::df:: Dataframe à lire
    ::variable_name:: Nom de la variable
    """
    return df.loc[df["name"] == variable_name, "sum"].iloc[0]


# Cell


def get_aggregats_from_json(
    var_name: str, calib_filepath: str, year: str = "2019", unit: str = "euros"
) -> Aggregate:
    """Convert calib file to an Aggregate.

    ::var_name:: une ligne de dataframe
    ::calib_filepath:: Path to calib file
    """
    global ref, perimeter
    with open(calib_filepath) as myfile:
        json_content = json.loads(myfile.read())
    keep_upper_bound_secret(json_content)
    data = Data(reference=[ref], date=year, values=json_content)
    aggregate = Aggregate(
        source_variable=var_name,
        unit=unit,
        perimeter=perimeter,
        data=data,
    )
    return aggregate


# Cell


def process_all_json(path: str) -> List[Aggregate]:
    aggregates = []
    files = glob.glob(path)
    if len(files) == 0:
        logger.warning(f"No file for {path} !")
        return []
    for file in files:
        #    Parcours les fichier de calibration et enregistre les agrégats dans le dictionnaire.
        var_name = file.split("-")[-1].split(".")[0]
        print(f"Processing {var_name} from {file}")
        aggregate = get_aggregats_from_json(var_name, file)
        aggregates.append(aggregate)
    return aggregates
