# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['leo_optimizer']

package_data = \
{'': ['*']}

install_requires = \
['stringcase>=1.2.0,<2.0.0']

setup_kwargs = {
    'name': 'leo-optimizer',
    'version': '0.0.5',
    'description': 'Package for on fly optimizing SQL queries generated by Django ORM. This package targets the N+1 problem so well known to Django.',
    'long_description': '# Leo Optimizer\n\nThis Django plugin allows you to optimize SQL queries and solve [N+1](https://scoutapm.com/blog/django-and-the-n1-queries-problem) bottleneck for queries resolved in GraphQL, as well as in admin pages.\n\n# Install\n\n```bash\npip install leo-optimizer\n```\n\n# Context\n\nLet\'s say that the business context that you are modeling is around the concept of a city. You modeled your solution following way:\n\n* City has one mayor\n* City has many districts\n* City belong to one state\n* State is in the country\n* Country is located in a continent\n\nOutcome of these relations allows you to build a complex GraphQL query that might look like this.\n\n```\nquery CityQuery {\n  allCities{\n    mayor {\n      city {\n        name\n      }\n    }\n    state {\n      country {\n        continent {\n          name\n        }\n      }\n    }\n    district {\n      city {\n        mayor {\n          city {\n            district {\n              name\n            }\n          }\n        }\n      }\n    }\n  }\n}\n```\n\n# Problem\n\nThe problem is that if you didn\'t use [select_related](https://docs.djangoproject.com/en/3.2/ref/models/querysets/#select-related) and [prefetch_related](https://docs.djangoproject.com/en/3.2/ref/models/querysets/#prefetch-related) on the model that you are resolving then by default you will get [N+1](https://scoutapm.com/blog/django-and-the-n1-queries-problem) problem.\n\n\n# Solution\n\nYou can fix this problem by importing `gql_optimizer` function from `leo_optimizer` package.\n\nIn your GraphQL resolver, wrap your model along with FiledNode that you would like to resolve against and return Django QuerySet. Your code should look like the code below.\n\n```python\nfrom leo_optimizer import gql_optimizer\n\nfrom app.models import City\n\ndef resolve_all_city(self, info):\n    queryset = gql_optimizer(City.objects.all(), info.field_nodes[0])\n    return queryset\n```\n\n# Admin\n\nAutomatically generated django admin pages suffer from the same [N+1](https://scoutapm.com/blog/django-and-the-n1-queries-problem) problem. To solve slow loading admin pages you can import `QuickAdmin` class in your `admin.py` file as shown on the example below.\n\n\n```python\nfrom django.apps import apps\nfrom django.contrib import admin\n\nfrom leo_optimizer import QuickAdmin\n\n\nfor model in apps.get_app_config("name_of_your_application").get_models():\n    admin.site.register(model, QuickAdmin)\n```',
    'author': 'Martin Andersen',
    'author_email': 'zdtdk@leo-pharma.com',
    'maintainer': 'Martin Andersen',
    'maintainer_email': 'zdtdk@leo-pharma.com',
    'url': 'https://github.com/LEO-Pharma-R-D-Data-Analytics/leo-optimizer',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
