"""Rebase script generation and execution."""

import os
from datetime import datetime
from typing import List

from ..models import CommitGroup


class RebaseScriptGenerator:
    """Generates rebase scripts for dry-run mode."""
    
    def __init__(self, repo_path: str = "."):
        self.repo_path = repo_path
    
    def generate_rebase_script(self, groups: List[CommitGroup], output_path: str) -> str:
        """Generate a shell script that performs the rebase operations."""
        script_lines = [
            "#!/bin/bash",
            "# Generated by git-smart-squash",
            f"# Generated on: {datetime.now().isoformat()}",
            "",
            "set -e  # Exit on any error",
            "",
            "echo 'Starting git smart squash operations...'",
            ""
        ]
        
        # Add backup creation
        script_lines.extend([
            "# Create backup branch",
            'BACKUP_BRANCH="backup_$(git branch --show-current)_$(date +%Y%m%d_%H%M%S)"',
            'git branch "$BACKUP_BRANCH"',
            'echo "Created backup branch: $BACKUP_BRANCH"',
            ""
        ])
        
        # Generate rebase instructions for each group
        for i, group in enumerate(groups):
            if len(group.commits) > 1:
                script_lines.extend([
                    f"# Group {i+1}: {group.rationale}",
                    f"# Files: {', '.join(list(group.files_touched)[:3])}",
                    f"# New message: {group.suggested_message}",
                ])
                
                # Create the git rebase interactive sequence
                oldest_commit = group.commits[0]
                script_lines.extend([
                    f'git rebase -i {oldest_commit.parent_hash} --exec \'echo "Processing group {i+1}"\'',
                    ""
                ])
        
        script_lines.extend([
            "echo 'Git smart squash completed successfully!'",
            'echo "Backup branch created: $BACKUP_BRANCH"',
            'echo "To restore backup: git reset --hard $BACKUP_BRANCH"'
        ])
        
        # Write script to file
        with open(output_path, 'w') as f:
            f.write('\n'.join(script_lines))
        
        # Make script executable
        os.chmod(output_path, 0o755)
        
        return output_path


class InteractiveRebaseExecutor:
    """Executes interactive rebase operations directly."""
    
    def __init__(self, repo_path: str = "."):
        self.repo_path = repo_path
    
    def execute_squash_plan(self, groups: List[CommitGroup], create_backup: bool = True) -> bool:
        """Execute the squash plan using git rebase interactive."""
        import subprocess
        import tempfile
        from ..git_operations.safety_checks import GitSafetyChecker
        from datetime import datetime
        
        # Create backup if requested
        backup_name = None
        if create_backup:
            try:
                checker = GitSafetyChecker(self.repo_path)
                timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
                current_branch = self._get_current_branch()
                backup_name = f"backup_{current_branch}_{timestamp}"
                checker.create_backup_branch(backup_name)
            except Exception:
                # If backup creation fails, continue but warn
                pass
        
        # Execute rebase for each group that has multiple commits
        try:
            for i, group in enumerate(groups):
                if len(group.commits) <= 1:
                    continue  # Skip single-commit groups
                    
                # Execute rebase for this group
                success = self._execute_group_rebase(group, i)
                if not success:
                    print(f"Warning: Failed to rebase group {i+1}")
                    # Continue with other groups rather than failing completely
            
            return True
            
        except Exception as e:
            print(f"Error during rebase execution: {e}")
            if backup_name:
                print(f"Your work is safe - restore with: git reset --hard {backup_name}")
            return False
    
    def _execute_group_rebase(self, group: CommitGroup, group_index: int) -> bool:
        """Execute rebase for a single group of commits."""
        import subprocess
        import tempfile
        import os
        
        if len(group.commits) <= 1:
            return True  # Nothing to squash
            
        try:
            # Sort commits by timestamp to ensure proper chronological order
            # Oldest commit should be the base for squashing
            sorted_commits = sorted(group.commits, key=lambda c: c.timestamp)
            oldest_commit = sorted_commits[0]
            newest_commit = sorted_commits[-1]
            
            print(f"Debug: Squashing {len(sorted_commits)} commits:")
            for i, commit in enumerate(sorted_commits):
                print(f"  {i}: {commit.short_hash} {commit.message} ({commit.timestamp})")
            
            # Verify we have the parent commit hash
            if not hasattr(oldest_commit, 'parent_hash') or not oldest_commit.parent_hash:
                # Get parent hash using git
                parent_result = subprocess.run(
                    ['git', '-C', self.repo_path, 'rev-parse', f'{oldest_commit.hash}^'],
                    capture_output=True, text=True
                )
                if parent_result.returncode != 0:
                    print(f"Failed to get parent commit for {oldest_commit.hash}")
                    return False
                parent_hash = parent_result.stdout.strip()
            else:
                parent_hash = oldest_commit.parent_hash
            
            # Create a temporary rebase todo file
            with tempfile.NamedTemporaryFile(mode='w', suffix='.txt', delete=False) as f:
                todo_file = f.name
                
                # Write rebase instructions in chronological order
                # First (oldest) commit: pick (keep as-is)
                f.write(f"pick {oldest_commit.hash} {oldest_commit.message}\n")
                
                # Subsequent commits: squash into the first
                for commit in sorted_commits[1:]:
                    f.write(f"squash {commit.hash} {commit.message}\n")
                
                print(f"Debug: Rebase todo file content:")
                with open(todo_file, 'r') as rf:
                    print(rf.read())
            
            # Create commit message file
            with tempfile.NamedTemporaryFile(mode='w', suffix='.txt', delete=False) as f:
                msg_file = f.name
                if group.suggested_message:
                    f.write(group.suggested_message)
                else:
                    f.write(f"Squashed {len(group.commits)} commits\n\n")
                    for commit in sorted_commits:
                        f.write(f"- {commit.message}\n")
                
                print(f"Debug: Commit message file content:")
                with open(msg_file, 'r') as rf:
                    print(rf.read())
            
            # Execute git rebase with our todo file
            env = os.environ.copy()
            env['GIT_SEQUENCE_EDITOR'] = f'cp {todo_file}'
            env['GIT_EDITOR'] = f'cp {msg_file}'
            
            # Use parent hash for rebase range
            cmd = [
                'git', '-C', self.repo_path, 'rebase', '-i', parent_hash
            ]
            
            print(f"Debug: Executing rebase command: {' '.join(cmd)}")
            print(f"Debug: Rebase range: {parent_hash}..{newest_commit.hash}")
            
            result = subprocess.run(
                cmd, 
                env=env,
                capture_output=True, 
                text=True,
                timeout=120  # Increased timeout
            )
            
            print(f"Debug: Rebase result code: {result.returncode}")
            if result.stdout:
                print(f"Debug: Rebase stdout: {result.stdout}")
            if result.stderr:
                print(f"Debug: Rebase stderr: {result.stderr}")
            
            # Verify the squash actually happened
            if result.returncode == 0:
                # Check if we actually have fewer commits now
                verify_result = subprocess.run(
                    ['git', '-C', self.repo_path, 'rev-list', '--count', f'{parent_hash}..HEAD'],
                    capture_output=True, text=True
                )
                if verify_result.returncode == 0:
                    commit_count = int(verify_result.stdout.strip())
                    expected_count = 1  # Should be 1 commit after squashing
                    print(f"Debug: Commit count after rebase: {commit_count} (expected: {expected_count})")
                    
                    if commit_count != expected_count:
                        print(f"Warning: Expected {expected_count} commit after squashing, but found {commit_count}")
                        return False
            
            # Clean up temporary files
            try:
                os.unlink(todo_file)
                os.unlink(msg_file)
            except:
                pass
            
            if result.returncode == 0:
                return True
            else:
                print(f"Git rebase failed for group {group_index + 1}: {result.stderr}")
                return False
                
        except Exception as e:
            print(f"Error executing rebase for group {group_index + 1}: {e}")
            import traceback
            traceback.print_exc()
            return False
    
    def _get_current_branch(self) -> str:
        """Get current branch name."""
        try:
            import subprocess
            cmd = ['git', '-C', self.repo_path, 'branch', '--show-current']
            result = subprocess.run(cmd, capture_output=True, text=True, check=True)
            return result.stdout.strip()
        except:
            return "unknown"