#!/usr/bin/env python
# -*- coding: utf-8 -*-
# Copyright (C) 2022 David Amrani Hernandez
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import sys
import json
import fire
import dome9
from tabulate import tabulate


class Dome9CLI():

    _tablefmt = 'github'
    _export_assessment_filename = 'assessment.json'

    def __init__(self):
        self._dome9 = dome9.Dome9()

    # ------------------------------------------
    # 			  SYSTEM METHODS
    # ------------------------------------------

    def _read_file(self, file):
        with open(file, 'r') as f:
            return f.read()

    def _write_file(self, data, file):
        with open(file, 'w') as f:
            f.write(data)

    def _find(self, element, json):
        keys = element.split('.')
        rv = json
        for key in keys:
            rv = rv[key]
        return rv

    def _pprint(self, dataset, headers):
        output = list()
        for item in dataset:
            rows = list()
            for key in headers:
                rows.append(self._find(key, item))
            output.append(rows)
        print(tabulate(output, headers=headers, tablefmt=self._tablefmt))

    # ------------------------------------------
    # 			  CLOUD ACCOUNTS
    # ------------------------------------------

    def list_aws_accounts(self):
        """List AWS (Amazon Web Services) accounts
        """
        data = self._dome9.list_aws_accounts()
        self._pprint(data, ['id', 'name', 'externalAccountNumber'])

    def list_azure_accounts(self):
        """List AZURE accounts
        """
        data = self._dome9.list_azure_accounts()
        self._pprint(data, ['id', 'name'])

    def list_google_accounts(self):
        """List GCP (Google Cloud Platform) accounts
        """
        data = self._dome9.list_google_accounts()
        self._pprint(data, ['id', 'name'])

    def list_kubernetes_accounts(self):
        """List Kubernetes accounts
        """
        data = self._dome9.list_kubernetes_accounts()
        self._pprint(data, ['id', 'name'])

    def list_cloud_accounts(self):
        """List all types of cloud accounts
        """
        data = self._dome9.list_cloud_accounts()
        self._pprint(data, ['id', 'name', 'vendor'])

    def get_cloud_account(self, accountId):
        """Get a cloud account by ID

        Args:
            accountId (int): Cloud account ID
        """
        data = self._dome9.get_cloud_account(accountId)
        self._pprint([data], ['id', 'name'])

    # ------------------------------------------
    # 				  RULESETS
    # ------------------------------------------

    def list_rulesets(self):
        """List all Compliance Rulesets
        """
        data = self._dome9.list_rulesets()
        self._pprint(data, ['id', 'cloudVendor', 'name'])

    def get_ruleset(self, rulesetId):
        """Get a ruleset by ID

        Args:
            rulesetId (int): Id of the ruleset
        """
        data = self._dome9.get_ruleset(rulesetId)
        self._pprint([data], ['id', 'cloudVendor', 'name', 'createdTime', 'updatedTime'])

    def create_ruleset(self, jsonFile):
        """Create a compliance ruleset

        Args:
            jsonFile (str): Absolute or relative path to a JSON file with the ruleset
        """
        ruleset = self._read_file(jsonFile)
        data = self._dome9.create_ruleset(ruleset)
        print('Ruleset created with ID: {}'.format(data['id']))

    def update_ruleset(self, jsonFile):
        """Update ruleset data

        Args:
            jsonFile (str): Absolute or relative path to a JSON file with the ruleset
        """
        ruleset = json.loads(self._read_file(jsonFile))
        remote_ruleset = filter(lambda x: ruleset['name'] in x['name'], self._dome9.list_rulesets())
        if remote_ruleset:
            ruleset['id'] = remote_ruleset[0]['id']
            self._dome9.update_ruleset(ruleset)
            print('Ruleset updated with ID: {}'.format(ruleset['id']))
        else:
            print('Ruleset "{}" not found'.format(ruleset['name']))

    def delete_ruleset(self, rulesetId):
        """Delete a compliance

        Args:
            rulesetId (int): Id of the ruleset
        """
        data = self._dome9.delete_ruleset(rulesetId)
        print('Ruleset deleted') if data else ('Resource not deleted')

    def generate_ruleset(self, name, cloud, rulesFile=None, desc=''):
        """Generate a ruleset template

        Args:
            cloud (str): Name of the cloud vendor. Accepted values: [aws, azure, google, kubernetes]
            name (str): Name of the ruleset
            rulesFile (str, optional): Absolute or relative path to a JSON file with Dome9 rules. Defaults to None.
            desc (str, optional): Description of the ruleset. Defaults to ''.
        """
        rules = []
        if rulesFile:
            rules.extend(json.loads(self._read_file(rulesFile)))
        ruleset = {
            "id": 0,
            "name": name,
            "description": desc,
            "hideInCompliance": True,
            "cloudVendor": cloud,
            "minFeatureTier": "Advanced",
            "rules": rules,
        }
        print(json.dumps(ruleset))

    # ------------------------------------------
    # 			  COMPLIANCE RULES
    # ------------------------------------------

    def list_rules(self, rulesetId):
        """List compliance rulesets
        """
        data = self._dome9.get_ruleset(rulesetId=rulesetId)['rules']
        self._pprint(data, ['name', 'severity', 'complianceTag'])

    # ------------------------------------------
    # 			  REMEDIATIONS
    # ------------------------------------------

    def list_remediations(self):
        """List all remediations
        """
        data = self._dome9.list_remediations()
        self._pprint(data, ['id', 'ruleName', 'cloudBots', 'comment'])

    def get_remediation(self, remediationId):
        """Get a remediation by ID

        Args:
            remediationId (int): Id of the remediation
        """
        data = self._dome9.get_remediation(remediationId)
        self._pprint([data], ['id', 'ruleName', 'cloudBots', 'comment'])

    def create_remediation(self, remediationFile):
        """Create a new remediation

        Args:
            remediationFile (str): Absolute or relative path to a JSON file with Dome9 remediation.
        """
        remediation = json.loads(self._read_file(remediationFile))
        data = self._dome9.create_remediation(remediation)
        print('Remediation created with ID: {}'.format(data['id']))

    def delete_remediation(self, remediationId):
        """Delete a specific remediation

        Args:
            remediationId (int): Id of the remediation
        """
        data = self._dome9.delete_remediation(remediationId)
        print('Remediation deleted') if data else ('Resource not deleted')

    # ------------------------------------------
    # 			  EXCLUSIONS
    # ------------------------------------------

    def list_exclusions(self):
        """List all exclusions
        """
        data = self._dome9.list_exclusions()
        self._pprint(data, ['id', 'cloudAccountId', 'logic', 'comment'])

    def get_exclusion(self, exclusionId):
        """Get a specific exclusion

        Args:
            exclusionId (int): Id of the exclusion
        """
        data = self._dome9.get_exclusion(exclusionId)
        print(json.dumps(data, indent=4))
        self._pprint([data], ['id', 'cloudAccountType', 'cloudAccountId', 'logic', 'comment'])

    def create_exclusion(self, exclusionFile):
        """Create a new exclusion

        Args:
            exclusion (str): Absolute or relative path to a JSON file with Dome9 exclusion.
        """
        exclusion = json.loads(self._read_file(exclusionFile))
        data = self._dome9.create_exclusion(exclusion)
        print('Exclusion created with ID: {}'.format(data['id']))

    def delete_exclusion(self, exclusionId):
        """Delete a specific exclusion

        Args:
            exclusionId (id): Id of the exclusion
        """
        data = self._dome9.delete_exclusion(exclusionId)
        print('Exclusion deleted') if data else ('Resource not deleted')

    # ------------------------------------------
    # 			  ASSESSMENTS
    # ------------------------------------------

    def _print_assessment_results(self, assessment, stats=False, export=None):
        if export:
            print('\n---------------- Export ----------------')
            self._write_file(json.dumps(assessment, indent=4), self._export_assessment_filename)
            print('[i] Report saved: ./%s' % self._export_assessment_filename)
            print('[ ] File size: %i KB' % (sys.getsizeof(json.dumps(assessment, indent=4))/1024))

        if stats:
            print('\n---------------- Stats ----------------')
            print('[-] Report URL: https://secure.dome9.com/v2/compliance-engine/result/%s' % assessment['id'])
            print('[-] Report ID: %s' % assessment['id'])
            print('[-] Account ID: %s' % assessment['request']['cloudAccountId'])
            print('[-] Ruleset: %s' % assessment['request']['name'])
            print('[-] Checks executed: %s' % len(assessment['tests']))
            print('[-] Entities scanned: %s' % len(assessment['testEntities']))
            for k, v in assessment['testEntities'].items():
                print('    - %s\t%s' % (k.title().ljust(18), len(v)))
            print('\n[!] Exam Passed: %s' % assessment['assessmentPassed'])

        print('\n---------------- Results ----------------\n')
        self._pprint(assessment['tests'], ['rule.name', 'rule.severity', 'testedCount', 'nonComplyingCount'])

    def run_assessment(self, rulesetId, cloudAccountId, stats=False, export=False):
        """Run assessment and show results

        Args:
            rulesetId (int): ID of the ruleset
            cloudAccountId (str): ID of the cloud account which will be scanned
            exportFile (bool): Export assessment to JSON file. Defaults to False.
        """
        assessment = self._dome9.run_assessment(rulesetId, cloudAccountId)
        if assessment:
            self._print_assessment_results(assessment, stats=stats, export=export)

    def get_assessment(self, assessmentId, stats=False, export=False):
        """Get assessment by ID

        Args:
            assessmentId (str): Id of the assessment
            stats (bool, optional): Show statistics of the assessment. Defaults to False.
            export (bool, optional): Export results to output file. Defaults to False.
        """
        assessment = self._dome9.get_assessment(assessmentId)
        if assessment:
            self._print_assessment_results(assessment, stats=stats, export=export)


def run(*args, **kwargs):
    try:
        fire.Fire(Dome9CLI())
        sys.exit(0)
    # Ctrl-C
    except KeyboardInterrupt as e:
        raise e
    # sys.exit()
    except SystemExit as e:
        raise e
    except Exception as e:
        print(e)
        sys.exit(1)


if __name__ == '__main__':
    run()
