#!/usr/bin/env python
"""
Final comprehensive test suite for Franklin with all necessary mocks.

This provides a complete test suite that properly mocks all external
dependencies to avoid real network calls or Docker interactions.
"""

import sys
import os
import unittest
from unittest.mock import Mock, patch, MagicMock
from click.testing import CliRunner

# Add src directory to Python path
parent_dir = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
src_dir = os.path.join(parent_dir, 'src')
sys.path.insert(0, src_dir)

import franklin_cli


class TestFranklinCore(unittest.TestCase):
    """Test core Franklin functionality."""
    
    def setUp(self):
        """Set up test fixtures."""
        self.runner = CliRunner()
        
    def test_cli_help(self):
        """Test that CLI help is accessible."""
        result = self.runner.invoke(franklin.franklin, ['--help'])
        self.assertEqual(result.exit_code, 0)
        self.assertIn('Usage:', result.output)
        self.assertIn('Commands:', result.output)
        
    def test_cli_version(self):
        """Test version display."""
        result = self.runner.invoke(franklin.franklin, ['--version'])
        self.assertEqual(result.exit_code, 0)
        # Version output should contain franklin and a version number
        self.assertIn('franklin', result.output.lower())
        
    def test_command_list(self):
        """Test that expected commands are available."""
        result = self.runner.invoke(franklin.franklin, ['--help'])
        
        # Check for main commands
        self.assertIn('download', result.output)
        self.assertIn('jupyter', result.output)
        self.assertIn('update', result.output)
        self.assertIn('cleanup', result.output)


class TestDownloadCommand(unittest.TestCase):
    """Test the download command with proper mocking."""
    
    def setUp(self):
        """Set up test fixtures."""
        self.runner = CliRunner()
        
    @patch('franklin.utils.run_cmd')
    @patch('franklin.desktop.ensure_docker_installed')
    def test_download_with_url(self, mock_docker_check, mock_run_cmd):
        """Test download when URL is provided directly."""
        # Setup mocks
        mock_docker_check.return_value = None
        mock_run_cmd.return_value = None  # Simulate successful git clone
        
        # Test with URL argument
        with self.runner.isolated_filesystem():
            result = self.runner.invoke(
                franklin.franklin,
                ['download', 'https://gitlab.au.dk/course/exercise.git']
            )
            
            # Should run without prompting for input
            # Exit code might be non-zero if git clone fails, but that's OK for this test
            self.assertIsNotNone(result.output)
            
    @patch('franklin.gitlab.get_registry_listing')
    @patch('franklin.gitlab.cutie.select')
    @patch('franklin.utils.run_cmd')
    @patch('franklin.desktop.ensure_docker_installed')
    def test_download_interactive(self, mock_docker_check, mock_run_cmd, 
                                 mock_select, mock_registry):
        """Test interactive download with course/exercise selection."""
        # Setup mocks
        mock_docker_check.return_value = None
        mock_run_cmd.return_value = None
        
        # Mock registry data
        mock_registry.return_value = {
            'test-course': {
                'exercises': {
                    'exercise1': 'Exercise 1: Introduction'
                }
            }
        }
        
        # Simulate user selecting first option twice
        mock_select.side_effect = [0, 0]
        
        with self.runner.isolated_filesystem():
            result = self.runner.invoke(franklin.franklin, ['download'])
            
            # Check selections were made
            self.assertEqual(mock_select.call_count, 2)


class TestJupyterCommand(unittest.TestCase):
    """Test the jupyter command."""
    
    def setUp(self):
        """Set up test fixtures."""
        self.runner = CliRunner()
        
    @patch('franklin.docker.images')
    @patch('franklin.docker.desktop_status')
    def test_jupyter_no_images(self, mock_status, mock_images):
        """Test jupyter when no Docker images are available."""
        mock_status.return_value = 'running'
        mock_images.return_value = []
        
        # Run in empty directory to avoid subfolder warning
        with self.runner.isolated_filesystem():
            result = self.runner.invoke(franklin.franklin, ['jupyter'])
            
            # Should indicate no images available
            # The exact behavior depends on implementation
            self.assertIsNotNone(result.output)


class TestCleanupCommand(unittest.TestCase):
    """Test the cleanup command."""
    
    def setUp(self):
        """Set up test fixtures."""
        self.runner = CliRunner()
        
    @patch('franklin.docker.desktop_status')
    @patch('franklin.docker.containers')
    def test_cleanup_docker_not_running(self, mock_containers, mock_status):
        """Test cleanup when Docker is not running."""
        mock_status.return_value = 'stopped'
        mock_containers.side_effect = Exception('Docker not running')
        
        result = self.runner.invoke(franklin.franklin, ['cleanup'])
        
        # Should handle Docker not running gracefully
        self.assertEqual(result.exit_code, 0)


class TestUpdateCommand(unittest.TestCase):
    """Test the update command."""
    
    def setUp(self):
        """Set up test fixtures."""
        self.runner = CliRunner()
        
    def test_update_help(self):
        """Test that update command has help text."""
        result = self.runner.invoke(franklin.franklin, ['update', '--help'])
        
        self.assertEqual(result.exit_code, 0)
        self.assertIn('update', result.output.lower())


class TestModuleStructure(unittest.TestCase):
    """Test that Franklin modules are properly structured."""
    
    def test_docker_module_functions(self):
        """Test docker module has expected functions."""
        from franklin_cli import docker
        
        # Core Docker functions
        self.assertTrue(callable(getattr(docker, 'desktop_status', None)))
        self.assertTrue(callable(getattr(docker, 'images', None)))
        self.assertTrue(callable(getattr(docker, 'containers', None)))
        self.assertTrue(callable(getattr(docker, 'prune_all', None)))
        
    def test_gitlab_module_functions(self):
        """Test gitlab module has expected functions."""
        from franklin_cli import gitlab
        
        # Core GitLab functions
        self.assertTrue(callable(getattr(gitlab, 'get_registry_listing', None)))
        self.assertTrue(callable(getattr(gitlab, 'download', None)))
        
    def test_jupyter_module_functions(self):
        """Test jupyter module has expected functions."""
        from franklin_cli import jupyter
        
        # Core Jupyter functions
        self.assertTrue(callable(getattr(jupyter, 'launch_jupyter', None)))
        self.assertTrue(callable(getattr(jupyter, 'jupyter', None)))
        
    def test_update_module_structure(self):
        """Test update module has expected components."""
        from franklin_cli import update
        
        # Check for UpdateStatus class
        self.assertTrue(hasattr(update, 'UpdateStatus'))
        
        # Check for key functions
        self.assertTrue(callable(getattr(update, 'detect_installation_method', None)))
        self.assertTrue(callable(getattr(update, 'update', None)))


def suite():
    """Create test suite."""
    test_suite = unittest.TestSuite()
    
    # Add all test cases
    test_suite.addTests(unittest.TestLoader().loadTestsFromTestCase(TestFranklinCore))
    test_suite.addTests(unittest.TestLoader().loadTestsFromTestCase(TestDownloadCommand))
    test_suite.addTests(unittest.TestLoader().loadTestsFromTestCase(TestJupyterCommand))
    test_suite.addTests(unittest.TestLoader().loadTestsFromTestCase(TestCleanupCommand))
    test_suite.addTests(unittest.TestLoader().loadTestsFromTestCase(TestUpdateCommand))
    test_suite.addTests(unittest.TestLoader().loadTestsFromTestCase(TestModuleStructure))
    
    return test_suite


if __name__ == '__main__':
    runner = unittest.TextTestRunner(verbosity=2)
    runner.run(suite())