# AUTOGENERATED! DO NOT EDIT! File to edit: 01_tune.ipynb (unless otherwise specified).

__all__ = ['calculate_metrics_by_thresh', 'coverage', 'calculate_metric_ignoring_nan', 'fbeta']

# Cell
from collections import defaultdict
from functools import partial
from typing import Callable, Iterable, Optional, Sequence, Union

import numpy as np
import pandas as pd
from sklearn import metrics

# Cell
def calculate_metrics_by_thresh(
    y_true: np.array,
    y_prob: np.array,
    metrics: Union[Callable, Sequence[Callable]],
    prob_to_pred: Union[str, Callable] = "infer",
) -> pd.DataFrame:
    """Calculate classification metrics as a function of threshold

    Parameters:
    - `y_true`: Ground-truth values
    - `y_prob`: Probability distributions
    - `metrics`: Callables that take `y_true`, `y_pred` as positional arguments
    and return a number. Must have a `__name__` attribute.
    - `prob_to_pred`: How to convert probabilities into predictions.
    Options:
        - "infer": Use "binary" if `y_true` has two distinct values,
        "multiclass" otherwise.
        - "binary": `1` if the second column is greater than the
        threshold, `0` otherwise.
        - "multiclass": The index of the column with the greatest value
        if that value is greater than the threshold, `np.nan` otherwise
        - A `Callable` that takes `y_prob` and a float `thresh` as
        keyword arguments and returns a 1d array that will be
        passed to your metrics as `y_pred`.

    Returns: DataFrame with one column "thresh" indicating the
    thresholds used, which is 0 and the sorted set of values that occur in
    `y_prob`, and an additional column for each input metric giving
    the value of that metric at that threshold.
    """

    def _get_prob_to_pred_func(prob_to_pred, y_true):
        if callable(prob_to_pred):
            return prob_to_pred
        if prob_to_pred == "binary" or (
            prob_to_pred == "infer" and len(np.unique(y_true)) == 2
        ):
            return lambda y_prob, thresh: np.where(y_prob[:, 1] > thresh, 1, 0)
        elif prob_to_pred in ["multiclass", "infer"]:
            return lambda y_prob, thresh: np.where(
                y_prob.max(axis=1) > thresh, y_prob.argmax(axis=1), np.nan
            )

    if callable(metrics):
        metrics = [metrics]
    prob_to_pred = _get_prob_to_pred_func(prob_to_pred=prob_to_pred, y_true=y_true)

    results = defaultdict(list)
    results["thresh"] = sorted(np.hstack([0, np.unique(y_prob)]))
    for thresh in results["thresh"]:
        y_pred = prob_to_pred(y_prob=y_prob, thresh=thresh)
        for metric in metrics:
            results[metric.__name__].append(metric(y_true, y_pred))

    return pd.DataFrame(results)

# Cell
def coverage(y_true: np.array, y_pred: np.array):
    """How often the model makes a prediction, where `np.nan` indicates
    abstaining from predicting.

    Parameters:
    - `y_true`: Ground-truth values
    - `y_pred`: Predicted values, possibly including `np.nan` to
    indicate abstraining from predicting
    """
    return (~np.isnan(y_pred)).mean()

# Cell
def calculate_metric_ignoring_nan(
    y_true: np.array, y_pred: np.array, metric: Callable, *args, **kwargs
):
    """Calculate `metric` ignoring `np.nan` predictions

    Parameters:
    - `y_true`: Ground-truth values
    - `y_pred`: Predicted values, possibly including `np.nan` to
    indicate abstraining from predicting
    - `metric`: Function that takes `y_true`, `y_pred` as keyword
    arguments

    Any additional arguments will be passed to `metric`
    """
    return metric(
        y_true=y_true[~np.isnan(y_pred)],
        y_pred=y_pred[~np.isnan(y_pred)],
        *args,
        **kwargs
    )

# Cell
def fbeta(precision: float, recall: float, beta: float = 1):
    weighted_mean_of_inverses = (
        1 / (1 + beta ** 2) * (1 / precision + beta ** 2 * 1 / recall)
    )
    weighted_harmonic_mean = 1 / weighted_mean_of_inverses
    return weighted_harmonic_mean